#include <QtTest>
#include <QCoreApplication>
#include <QMessageBox>

#include "fmvipdispatcher.h"
#include "fmtask.h"
#include "fmnumpad.h"

class TestPlugin : public QObject
{
    Q_OBJECT

public:
    TestPlugin();
    ~TestPlugin();

private:
    FMVipDispatcher fmvip;

private slots:
private:
    void initTestCase();
    void cleanupTestCase();

    void test_Sign_data();
    void test_Sign();

    void test_JsonObjToMap_data();
    void test_JsonObjToMap();

private slots:
    void test_dotask_data();
    void test_dotask();
};

TestPlugin::TestPlugin()
{

}

TestPlugin::~TestPlugin()
{

}

void TestPlugin::initTestCase()
{

}

void TestPlugin::cleanupTestCase()
{

}

void TestPlugin::test_dotask_data()
{
    QTest::addColumn<QByteArray>("reqData");

    QTest::newRow("Refund") << QByteArray("{\"fm_cmd\": 1004,\"order_id\": \"20171018003\"}");
    QTest::newRow("Refund not") << QByteArray("{\"fm_cmd\": 1004,\"order_id\": \"12345\"}");
    QTest::newRow("SetStoreInfo") << QByteArray("{\"fm_cmd\": 1000,\"store_id\": \"99999\",\"pos_id\": \"1\",\"business_date\": \"20171016\",\"operator_id\": \"001\"}");
    QTest::newRow("SetStoreInfo_need_posId") << QByteArray("{\"fm_cmd\": 1000,\"store_id\": \"fm9999\",\"pos_id\": \"\",\"business_date\": \"20171016\",\"operator_id\": \"001\"}");
    QTest::newRow("SetStoreInfo_notnull_posId") << QByteArray("{\"fm_cmd\": 1000,\"store_id\": \"fm9999\",\"business_date\": \"20171016\",\"operator_id\": \"001\"}");
    QTest::newRow("SetStoreInfo_error_type") << QByteArray("{\"fm_cmd\": 1000,\"store_id\": \"fm9999\",\"pos_id\": 1,\"business_date\": \"20171016\",\"operator_id\": \"001\"}");
    QTest::newRow("Login") << QByteArray("{\"fm_cmd\": 1001}");
    QTest::newRow("Pay") << QByteArray("{"
                                       "    \"fm_cmd\": 1003,"
                                       "    \"order_amount\":900,"
                                       "    \"trans_amount\": 800,"
                                       "    \"undis_amount\": 800,"
                                       "    \"order_id\": \"20171018003\","
                                       "    \"trans_id\": \"100001\","
                                       "    \"products\": ["
                                       "    	{"
                                       "            \"pid\": \"0079020\","
                                       "            \"name\": \"测试商品1\","
                                       "            \"price\": 100,"
                                       "            \"consume_num\": 1"
                                       "        },"
                                       "        {       "
                                       "            \"pid\": \"0077842\","
                                       "            \"name\": \"测试商品2\","
                                       "            \"price\": 800,"
                                       "            \"consume_num\": 2"
                                       "        }"
                                       "    ]"
                                       "}");
    QTest::newRow("Order") << QByteArray("{\"fm_cmd\": 1007,\"order_id\": \"20171018003\"}");
    QTest::newRow("Fund") << QByteArray("{\"fm_cmd\": 1002,\"order_id\": \"20171018001\"}");
    QTest::newRow("CouponPay") << QByteArray("{\"fm_cmd\": 10032}");
}

void TestPlugin::test_dotask()
{
    QFETCH(QByteArray, reqData);

    QByteArray rspData;
    fmvip.doTask(reqData, rspData);

    qDebug() << "RspData: " << rspData.data();

    QMessageBox::information(nullptr, "Return to POS", QString(rspData));
}

void TestPlugin::test_JsonObjToMap_data()
{
    QTest::addColumn<QByteArray>("Json");
    QTest::addColumn<QByteArray>("Map");

    QByteArray json = "{\"null\":null,\"bool\":0,\"int\":100,\"double\":66.6,\"string\":\"string\",\"array\":[{\"a1\":\"a1\"},{\"a2\":\"a2\"}],\"object\":{\"o1s\":\"o1s\",\"o2\":[{\"o2a\":\"o2a\"}],\"o3\":{\"o3o\":\"o3o\"}}}" ;
    QByteArray map = "bool=0&double=66.6&int=100&null=&object.o1s=o1s&object.o3.o3o=o3o&string=stringfreemud";

    QTest::newRow("Simple") << json << map;
}

void TestPlugin::test_JsonObjToMap()
{
    QFETCH(QByteArray, Json);
    QFETCH(QByteArray, Map);

    QJsonObject obj = QJsonDocument::fromJson(Json).object();

    QCOMPARE(FMTask::JsonObjToMap(obj), Map);
}

void TestPlugin::test_Sign_data()
{
    QTest::addColumn<QString>("SignJson");
    QTest::addColumn<QString>("SignStr");

    QString jsonStr1 = "{\"amount\":100,\"null\":null,\"empty\":\"\",\"bool\":true,\"data\":{\"account\":\"123123123\",\"object\":{\"string\":\"string\"},\"products\":[{\"name\":\"p1\",\"pid\":\"1001\"},{\"name\":\"p2\",\"pid\":\"1002\"},{\"name\":\"p3\",\"pid\":\"1003\"}]}}";
    QString signStr1 = "d27e68bc5801c51f16458b3a384befa5";

    QString jsonStr2 = "{\"amount\":100,\"null\":null,\"empty\":\"\",\"bool\":true,\"data\":{\"object\":{\"string\":\"string\",\"obj\":{\"b\":1,\"a\":0,\"ab\":\"\",\"aa\":null},\"array\":[\"a\"]},\"bool\":false,\"account\":\"123123123\",\"products\":[{\"name\":\"p1\",\"pid\":\"1001\"},{\"name\":\"p2\",\"pid\":\"1002\"}]}}";
    QString signStr2 = "ce543c28a6b062f32e5c2e5ac68f52ec";

    QTest::newRow("First") << jsonStr1 << signStr1;
    QTest::newRow("Second") << jsonStr2 << signStr2;
}

void TestPlugin::test_Sign()
{
    QFETCH(QString, SignJson);
    QFETCH(QString, SignStr);

    QJsonParseError error;
    QJsonObject jsonObj1 = QJsonDocument::fromJson(SignJson.toUtf8(), &error).object();
    QVERIFY2(error.error==QJsonParseError::NoError, error.errorString().toStdString().data());

    QCOMPARE(FMTask::Sign(jsonObj1), SignStr);
}

QTEST_MAIN(TestPlugin)

#include "tst_testplugin.moc"
