#include <QtTest>
#include "global.h"
#include "items/item.h"
#include "items/order.h"
#include "items/pay.h"
#include "items/product.h"
#include "items/storeinfo.h"
#include "fmtool.h"
#include "database/dbop.h"
#include <QJsonObject>

// add necessary includes here

using FMItem::Item;
using FMItem::Order;
using FMItem::Pay;
using FMItem::Product;
using FMItem::StoreInfo;

class TestItem : public QObject
{
    Q_OBJECT

public:
    TestItem();
    ~TestItem();


    Item *gItem;
    Order *gOrder;
    Item *gPay;
    Item *gProduct;


    QJsonObject json1;
    QJsonObject json2;

private slots:
    void initTestCase();
    void cleanupTestCase();

    void test_SetPropertiesByJson_data();
    void test_SetPropertiesByJson();

    void test_PropertyNames_data();
    void test_PropertyNames();

    void test_DBSave_data();
    void test_DBSave();

    void test_GetOrderByOrderId_data();
    void test_GetOrderByOrderId();

    void test_GetTableLastItem_data();
    void test_GetTableLastItem();
};

TestItem::TestItem()
{

}

TestItem::~TestItem()
{

}

void TestItem::initTestCase()
{
    gItem = new Item();
    gOrder = new Order();
    gPay = new Pay();
    gProduct = new Product();

    json1["order_id"] = "11111";
    json1["fm_order_id"] = "11111";
    json1["order_amount"] = 100;
    json1["canPay"] = true;

    json2["order_id"] = "22222";
    json2["canPay"] = false;
}

void TestItem::cleanupTestCase()
{
    delete gItem;
    delete gOrder;
    delete gPay;
    delete gProduct;
}

void TestItem::test_SetPropertiesByJson_data()
{
    QTest::addColumn<QJsonObject>("json");
    QTest::addColumn<QString>("orderId");
    QTest::addColumn<QString>("fmOrderId");
    QTest::addColumn<int>("orderAmount");
    QTest::addColumn<bool>("canPay");

//    QJsonArray pays;
//    QJsonObject pay1, pay2;
//    pay1[PosProps.Pay_id] = "1";
//    pay2[PosProps.Pay_id] = "2";
//    pays << pay1 << pay2;
//    json1["payList"] = pays;

//    FMItem::ItemList payList;
//    Pay *p1 = new Pay();
//    p1->setPayId("1");
//    Pay *p2 = new Pay();
//    p2->setPayId("2");
//    payList << p1 << p2;


    QTest::newRow("Whole") << json1 << "11111" << "11111" << 100 << true;
    QTest::newRow("Part") << json2 << "22222" << "11111" << 100 << false;
}

void TestItem::test_SetPropertiesByJson()
{
    QFETCH(QJsonObject, json);
    QFETCH(QString, orderId);
    QFETCH(QString, fmOrderId);
    QFETCH(int, orderAmount);
    QFETCH(bool, canPay);
//    QFETCH(FMItem::ItemList, payList);

    Order order;
    order.SetPropertiesByJson(json);

    QCOMPARE(order.orderId(), SearchJsonValue(json, "order_id").toString());
    QCOMPARE(order.fmOrderId(), SearchJsonValue(json, "fm_order_id").toString());
    QCOMPARE(order.orderAmount(), SearchJsonValue(json, "order_amount").toInt());
    QCOMPARE(order.property("canPay").toBool(), SearchJsonValue(json, "canPay").toBool());
//    QCOMPARE(order.payList(), payList);

    gOrder->SetPropertiesByJson(json);

    QCOMPARE(gOrder->orderId(), orderId);
    QCOMPARE(gOrder->fmOrderId(), fmOrderId);
    QCOMPARE(gOrder->orderAmount(), orderAmount);
    QCOMPARE(gOrder->property("canPay").toBool(), canPay);
//    QCOMPARE(gOrder->payList(), payList);
}

void TestItem::test_PropertyNames_data()
{

    QTest::addColumn<Item*>("item");
    QTest::addColumn<int>("propertyCount");
    QTest::addColumn<QStringList>("propertyNames");

    QStringList itemPNames, orderPNames, payPNames, productPNames;

    itemPNames << "objectName" << "DBID";
    QTest::newRow("Item") << gItem << itemPNames.length() << itemPNames;

    orderPNames << "objectName" << "DBID" <<"order_id" << "fm_order_id" << "order_amount" << "paid_amount" << "undis_amount"
                << "fm_open_id" << "products" << "settled" << "refunded" << "storeInfoDBID";
    QTest::newRow("Order") << qobject_cast<Item*>(gOrder) << orderPNames.length() << orderPNames;

    payPNames << "objectName" << "DBID" <<"pay_id" << "pay_str" << "pay_amount" << "refund_amount" << "dis_amount" << "account"
              << "trans_id" << "fm_trans_id" << "third_trans_id" << "orderDBID";
    QTest::newRow("Pay") << gPay << payPNames.length() << payPNames;

    productPNames << "objectName" << "DBID" << "id" <<"quantity" << "price" << "name";
    QTest::newRow("Product") << gProduct << productPNames.length() << productPNames;
}

void TestItem::test_PropertyNames()
{
    QFETCH(Item *, item);
    QFETCH(int, propertyCount);
    QFETCH(QStringList, propertyNames);

    QCOMPARE(item->PropertyCount(), propertyCount);
    QCOMPARE(item->PropertyNames(), propertyNames);
}

void TestItem::test_DBSave_data()
{
    QTest::addColumn<Item*>("item");
    QTest::addColumn<bool>("isSuccess");

    Pay *pay = new Pay();
    pay->setPayId("1");
    pay->setPayStr("pay1");
    pay->setTransId("12345");
    pay->setPayAmount(100);

    QTest::newRow("Pay") << qobject_cast<Item*>(pay) << true;
    QTest::newRow("Order") << qobject_cast<Item*>(gOrder) << true;

    gOrder->setFmOrderId("11111");
    gOrder->setOrderId("12345");
    gOrder->setOrderAmount(1000);
    gOrder->setProductText("{\"pid\":\"1\"},{\"pid\":\"2\"}");
    QTest::newRow("Order") << qobject_cast<Item*>(gOrder) << true;

    pay->deleteLater();
}

void TestItem::test_DBSave()
{
    QFETCH(Item *, item);
    QFETCH(bool, isSuccess);

    bool isOk = DBOP::Save(item);

    QCOMPARE(isOk, isSuccess);
    QVERIFY(item->DBID() > 0);
}

void TestItem::test_GetOrderByOrderId_data()
{
    QTest::addColumn<Item*>("in_order");
    QTest::addColumn<QString>("in_orderId");
    QTest::addColumn<Item*>("in_pay");

    FMItem::Item *order = new Order();
    order->SetPropertiesByJson(json1);
    qsrand(QDateTime::currentMSecsSinceEpoch());
    QString orderId = QDateTime::currentDateTimeUtc().toString("yyyyMMddhhmmsszzz") + QString::number(qrand());
    order->setProperty("order_id", orderId);

    FMItem::Item *pay = new Pay();

    QTest::newRow("Order1") << order << orderId << pay;
}

void TestItem::test_GetOrderByOrderId()
{
    QFETCH(Item*, in_order);
    QFETCH(QString, in_orderId);
    QFETCH(Item*, in_pay);

    bool isOk = DBOP::Save(in_order);
    QVERIFY2(isOk, "Save in_order failed.");

    in_pay->setProperty("orderDBID", in_order->DBID());
    isOk = DBOP::Save(in_pay);
    QVERIFY2(isOk, "Save in_pay failed.");

    Order* order = new Order();
    DBOP::GetOrderByOrderId(in_orderId, order);

    QVERIFY(order->isEqual(in_order));
    delete order;

    QSharedPointer<Order> orderPointer = DBOP::GetOrderByOrderId(in_orderId);
    QVERIFY(orderPointer.data()->isEqual(in_order));

    QList<QSharedPointer<Pay> > payList;
    payList = DBOP::GetPaysByOrderDBId(in_order->DBID());
    QVERIFY(payList.length() == 1);

    QVERIFY(payList.first().data()->isEqual(in_pay));
}

void TestItem::test_GetTableLastItem_data()
{
    QTest::addColumn<Item*>("save_item");
    QTest::addColumn<Item*>("get_item");


    Order *order_save = new Order(this);
    order_save->SetPropertiesByJson(json1);
    Order *order_get = new Order(this);
    Pay *pay_save = new Pay(this);
    pay_save->SetPropertiesByJson(json1);
    Pay *pay_get = new Pay(this);
    StoreInfo *si_save = new StoreInfo(this);
    si_save->setStoreId("fm9999");
    si_save->setPosId("2");
    si_save->setBusinessDate("2017");
    si_save->setOperatorId("001");
    StoreInfo *si_get = new StoreInfo(this);
    QTest::newRow("Order") << qobject_cast<Item*>(order_save) << qobject_cast<Item*>(order_get);
    QTest::newRow("Pay") << qobject_cast<Item*>(pay_save) << qobject_cast<Item*>(pay_get);
    QTest::newRow("StoreInfo") << qobject_cast<Item*>(si_save) << qobject_cast<Item*>(si_get);
}

void TestItem::test_GetTableLastItem()
{
    QFETCH(Item *, save_item);
    QFETCH(Item *, get_item);

    bool isOk = DBOP::Save(save_item);
    QVERIFY2(isOk, "Save item failed.");


    DBOP::GetLastItem(get_item);
    QVERIFY2(save_item->isEqual(get_item), "Get table last item failed.");
}

QTEST_MAIN(TestItem)

#include "tst_testitem.moc"
