﻿#include "taskrefund.h"
#include "fmviprefund.h"
#include "database/dbop.h"
#include "items/item.h"
#include "items/order.h"
#include "items/pay.h"
#include "items/storeinfo.h"
#include "taskothers.h"
#include <QSharedPointer>
#include <QJsonDocument>
#include "fmmsgwnd.h"

TaskRefund::TaskRefund(QJsonObject &jsonObj, Session *session, QObject *parent)
    : FMTask(jsonObj, FM_Refund, session, parent)
    , _refundAmount(0),
      _order(new FMItem::Order(this))
{

}

TaskRefund::~TaskRefund()
{
    delete _order;
}

void TaskRefund::packagePOSReq()
{
    QString orderId = getPosJsonValue(PosProps.OrderId).toString();
    session()->addData(PosProps.OrderId, orderId);

    FMItem::StoreInfo storeInfo;
    bool isOk = DBOP::GetLastItem(&storeInfo);
    if(isOk) {
        posReqJsonObj = storeInfo.toJson(QStringList(), posReqJsonObj);
    } else {
        QString info = QString::fromLocal8Bit("未设置门店信息，不能进行退款.");
        FMP_WARN() << info;
        setError(FM_API_ERROR, info);
        return;
    }

    DBOP::GetOrderByOrderId(orderId, _order);
    if(_order->isNew()) {
        QString info = QString::fromLocal8Bit("未找到订单信息，不能进行退款.");
        FMP_WARN() << info;
        setError(FM_API_ERROR, info);
        return;
    } else if(DBOP::GetPaysByOrderDBId(_order->DBID()).isEmpty()) {
        QString info = QString::fromLocal8Bit("订单中没有可以进行退款的支付.");
        FMP_WARN() << info;
        setError(FM_API_ERROR, info);
        return;
    }
}

void TaskRefund::setWindow()
{
    // 已结算则直接退订单
    if(_order->settled()) {
        onRefundOrder(_order->DBID());
    }
    // 未结算，仅有一笔支付则直接退该笔支付
//    else if(_order->payList().size() == 1) {
//        onRefundPay(_order->payList().first()->DBID());
//    }
    // 其他情况显示界面
    else {
        _window = new FMVipRefund();

        connect(qobject_cast<FMVipRefund*>(_window), &FMVipRefund::refundPay, this, &TaskRefund::onRefundPay);
        connect(qobject_cast<FMVipRefund*>(_window), &FMVipRefund::refundOrder, this, &TaskRefund::onRefundOrder);
    }
}

void TaskRefund::packageServerReq()
{
}

void TaskRefund::packagePOSRsp()
{
    posRspJsonObj[PosProps.StatusCode] = FM_API_SUCCESS;
    posRspJsonObj[PosProps.Msg] = "Success";
    posRspJsonObj["refund_amount"] = _refundAmount;
}

void TaskRefund::onRefundPay(int DBID)
{
    posReqJsonObj[PosProps.TransId] = session()->data(PosProps.TransId).toString();

    QSharedPointer<FMItem::Pay> payPointer = DBOP::GetPayByDBId(DBID);
    QSharedPointer<FMItem::Order> orderPointer = DBOP::GetOrderByDBId(payPointer->orderDBID());

    posReqJsonObj = orderPointer->toJson(QStringList(), posReqJsonObj);

    posReqJsonObj[PosProps.TransId] = payPointer->fmTransId();
    TaskRefundPay refundPay(posReqJsonObj, this);
    QByteArray rspData = refundPay.doTask();

    QJsonObject rspObj = QJsonDocument::fromJson(rspData).object();
    if(rspObj[PosProps.StatusCode].toInt() == FM_API_SUCCESS) {
        payPointer->setRefundAmount(payPointer->payAmount());
        bool isOk = DBOP::Save(payPointer.data());
        if(isOk) {
            orderPointer->setPaidAmount(orderPointer->paidAmount() - payPointer->refundAmount());
            isOk = DBOP::Save(orderPointer.data());
            _refundAmount += payPointer->payAmount();
        }

        if(_window!=nullptr) {
            qobject_cast<FMVipRefund*>(_window)->refresh();
            qobject_cast<FMVipRefund*>(_window)->setIsRefundSuccess(true);
        }

    } else {
        QString msg = searchJsonValue(rspObj, PosProps.Msg).toString();
        FMMsgWnd::FailureWnd(msg, _window);
    }
}

void TaskRefund::onRefundOrder(int DBID)
{
    QSharedPointer<FMItem::Order> orderPointer = DBOP::GetOrderByDBId(DBID);
    posReqJsonObj[PosProps.Fm_order_id] = orderPointer->fmOrderId();
    posReqJsonObj[PosProps.OrderId] = orderPointer->orderId();
    TaskRefundOrder refundOrder(posReqJsonObj, this);
    QByteArray rspData = refundOrder.doTask();

    QJsonObject rspObj = QJsonDocument::fromJson(rspData).object();
    if(rspObj[PosProps.StatusCode].toInt() == FM_API_SUCCESS) {
        orderPointer->setRefunded(true);
        DBOP::Save(orderPointer.data());
        if(_window!=nullptr) {
            qobject_cast<FMVipRefund*>(_window)->refresh();
            qobject_cast<FMVipRefund*>(_window)->setIsRefundSuccess(true);
        }

        _refundAmount = orderPointer->paidAmount();
    } else {
        QString msg = searchJsonValue(rspObj, PosProps.Msg).toString();
        FMMsgWnd::FailureWnd(msg, _window);
    }
}
