﻿#include "taskfinal.h"
#include "fmbackup.h"
#include "fmnetwork.h"
#include "fmp_logger_i.h"
#include "fmmsgwnd.h"
#include "fmp_vip_settings.h"
#include "dbop.h"
#include "items/order.h"
#include "items/pay.h"
#include "items/storeinfo.h"
#include <QDateTime>
#include <QJsonDocument>
#include <QJsonArray>
#include <QCryptographicHash>

TaskFinal::TaskFinal(QJsonObject &jsonObj, Session *session, QObject *parent)
    :FMTaskNoWnd(jsonObj, FM_Final, session, parent)
    ,_order(new FMItem::Order(this))
    ,isUseVipPay(false)
{
}

TaskFinal::~TaskFinal()
{
    delete _order;
}

void TaskFinal::packagePOSReq()
{
    FMItem::StoreInfo storeInfo;
    bool isOk = DBOP::GetLastItem(&storeInfo);
    if(isOk) {
        posReqJsonObj = storeInfo.toJson(QStringList(), posReqJsonObj);
        _order->setStoreInfoDBID(storeInfo.DBID());
    } else {
        QString info = QString::fromLocal8Bit("未设置门店信息，不能进行结算.");
        FMP_WARN() << info;
        setError(FM_API_ERROR, info);
        return;
    }

    QString orderId = getPosJsonValue(PosProps.OrderId).toString();
    DBOP::GetOrderByOrderId(orderId, _order);

    if(_order->isNew()) {
        _order->SetPropertiesByJson(posReqJsonObj);
        DBOP::Save(_order);
    } else if(_order->settled()) {
        QString info = QString::fromLocal8Bit("订单已结算，不能重复结算.");
        FMP_WARN() << info;
        setError(FM_API_ERROR, info);
    }
}

void TaskFinal::packageServerReq()
{
    QMap<QString, QString> key2proMap;
    QStringList keys;
    QJsonObject dataObj;
    keys << PosProps.Fm_open_id << PosProps.OrderAmount << PosProps.PaidAmount << PosProps.Fm_order_id << PosProps.TransId;
    key2proMap.clear();
    foreach (QString key, keys) {
        key2proMap[ServerProps(key)] = key;
    }
    dataObj = _order->toJson(key2proMap, dataObj);
    dataObj[ServerProps(PosProps.TransId)] = getPosJsonValue(PosProps.OrderId);

    FMItem::StoreInfo storeInfo;
    bool isOk = DBOP::GetItemByDBID(_order->storeInfoDBID(), &storeInfo);
    if(isOk) {
        dataObj[ServerProps(PosProps.BussinessDate)] = storeInfo.businessDate();
    }

    // Products
    QString productText = _order->productText();
    QStringList proKeys;
    proKeys << PosProps.ConsumeNum << PosProps.ProductId << PosProps.Price;
    foreach (const QString &proKey, proKeys) {
        productText = productText.replace(proKey, ServerProps(proKey));
        dataObj[ServerProps(PosProps.PaidAmount)] = _order->orderAmount();
    }
    QJsonArray proArray = QJsonDocument::fromJson(_order->productText().toUtf8()).array();
    dataObj[ServerProps(PosProps.Products)] = proArray;

//    QJsonArray proArray = QJsonDocument::fromJson(_order->productText().toUtf8()).array();
//    foreach (QJsonValue proValue, proArray) {
//        QJsonObject proObj;
//        QStringList proKeys;
//        proKeys << PosProps.ConsumeNum << PosProps.ProductId << PosProps.Price;
//        foreach (const QString &proKey, proKeys) {
//            proObj[ServerProps(proKey)] = proValue.toObject()[proKey];
//        }

//    }

    // 从数据库读取支付信息
    QList<QSharedPointer<FMItem::Pay> > payList = DBOP::GetPaysByOrderDBId(_order->DBID());

    // 将POS新传的支付方式加入已付到金额中
    foreach (QJsonValue payValue, getPosJsonValue(PosProps.Pay_list).toArray()) {
        QJsonObject pay_id = payValue.toObject();
        QSharedPointer<FMItem::Pay> payPointer = QSharedPointer<FMItem::Pay>(new FMItem::Pay(this));
        payPointer->SetPropertiesByJson(pay_id);
        payPointer->setOrderDBID(_order->DBID());
        if(payPointer->payId() == "20001" || payPointer->payId() == "10001" || payPointer->payId() == "10004") {
            FMP_WARN() << QString::fromLocal8Bit("POS无需传会员支付和非码支付信息.");
            continue;
        } else {
            _order->setPaidAmount(_order->paidAmount() + payPointer->payAmount());
            payList.append(payPointer);
        }
    }

    // 如果订单还未付完，默认以现金补全
    if(_order->orderAmount() > _order->paidAmount()) {
        QSharedPointer<FMItem::Pay> cashPay = QSharedPointer<FMItem::Pay>(new FMItem::Pay());
        cashPay->setPayId("20005");
        cashPay->setPayStr(QString::fromLocal8Bit("现金支付(默认)"));
        cashPay->setPayAmount(_order->orderAmount() - _order->paidAmount());
        cashPay->setOrderDBID(_order->DBID());
        bool isOk = DBOP::Save(cashPay.data());
        if(isOk) {
            _order->setPaidAmount(_order->orderAmount());
            payList.append(cashPay);
        }
    }

    // 转成服务端需要的payArray
    QJsonArray payArray;
    for (QList<QSharedPointer<FMItem::Pay> >::const_iterator it = payList.constBegin(); it!=payList.constEnd(); it++) {
        QJsonObject payObj;
        payObj[ServerProps(PosProps.Amount)] = it->data()->payAmount();
        payObj[ServerProps(PosProps.TransId)] = it->data()->transId();
        payObj["thirdPayTransId"] = it->data()->fmTransId();
        payObj["typeModeFlag"] = it->data()->payId();
        payArray.append(payObj);
    }

    dataObj[ServerProps(PosProps.Pay_list)] = payArray;

    serverReqJsonObj["data"] = dataObj;
}

bool TaskFinal::sendToServer(bool isShowMsg)
{
    Q_UNUSED(isShowMsg);
    FMTask::sendToServer(false);
    
    // 备份
    if(error() == FM_API_NETWORERROR || error() == FM_API_TIMEOUT) {
        if(!isUseVipPay) {
            QString fm_id = backup();
            if(!fm_id.isEmpty()){
                setError(FM_API_SUCCESS);
                serverRspJsonObj[ServerProps(PosProps.StatusCode)] = FM_API_SUCCESS;
                serverRspJsonObj[ServerProps(PosProps.Msg)] = QString::fromLocal8Bit("结算成功");
                serverRspJsonObj[ServerProps(PosProps.Fm_order_id)] = fm_id;
                serverRspJsonObj[ServerProps(PosProps.Prompt)] = 1;
                serverRspJsonObj[ServerProps(PosProps.Print1)] = "";
                serverRspJsonObj[ServerProps(PosProps.Print2)] = "";
            }
        }
    }
    bool isOk = (error() == FM_API_SUCCESS);
    if(!isOk)
    {
        FMMsgWnd::FailureWnd(errorString(), _window);
    }
    return isOk;
}

void TaskFinal::packagePOSRsp()
{
    std::vector<QString> p;
    p.push_back(PosProps.StatusCode);
    p.push_back(PosProps.Msg);
    p.push_back(PosProps.Fm_order_id);
    foreach(auto prop , p)
    {
        posRspJsonObj[prop] = getServerJsonValue(prop);
    }

    QJsonObject ext;
    ext[PosProps.Print] = getServerJsonValue(PosProps.Print1);
    posRspJsonObj["ext"] = ext;


    if(getServerJsonValue(PosProps.StatusCode).toInt() == FM_API_SUCCESS) {

        // 保存其它支付方式
        foreach(QJsonValue payVluae, getPosJsonValue(PosProps.Pay_list).toArray()) {
            QJsonObject pay_id = payVluae.toObject();
            int amount = pay_id[PosProps.Pay_amount].toInt();
            if(amount!=0) {
                FMItem::Pay *pay = new FMItem::Pay(this);
                pay->SetPropertiesByJson(pay_id);
                pay->setOrderDBID(_order->DBID());
                DBOP::Save(pay);
                delete pay;
            }
        }

        _order->setFmOrderId(getServerJsonValue(PosProps.Fm_order_id).toString());
        _order->setSettled(true);
        DBOP::Save(_order);
    }
}

QString TaskFinal::backup()
{
    int newId = FMBackup::instance()->insertNewRow();
    QString uuid = QString("E%1%2%3%4")
            .arg(getPosJsonValue(PosProps.StoreId).toString())
            .arg(getPosJsonValue(PosProps.PosId).toString().toInt(), 2, 10, QChar('0'))
            .arg(QDateTime::currentDateTime().toString("yyMMddhhmmsszzz"))
            .arg(newId, 5, 10, QChar('0'));
    QJsonDocument json(serverReqJsonObj);
    bool isOk = FMBackup::instance()->updateReqData(newId, this->url, json.toJson(QJsonDocument::Compact));
    if(isOk) {
        return uuid;
    }
    return "";
}
