﻿#include "fmtask.h"
#include "fmnetwork.h"
#include <QDateTime>
#include <QSettings>
#include <QJsonDocument>
#include <QJsonParseError>
#include <QCryptographicHash>
#include <QFile>

FMTask::FMTask(QJsonObject &jsonObj, FM_TYPE fmType, Session *session, QObject *parent) :
    QObject(parent),
    posReqJsonObj(jsonObj),
    _window(nullptr),
    isCreator(false),
    _FM_Type(fmType),
    preTask(nullptr),
    _error(FM_API_SUCCESS),
    _errorMsg("")
{
    if(session == 0) {
        _session = new Session();
        isCreator = true;
    } else {
        _session = session;
    }
}

FMTask::~FMTask()
{
    del_p(_window);
    del_p(preTask);
    if(isCreator) {
        del_p(_session);
    }
}

QByteArray FMTask::doTask()
{
    FMP_DEBUG() << __FUNCTION__;
#ifdef FM_DEBUG
    if(!checkReqJson()) {
        return errorStringJson().toUtf8();
    }
#endif

    RunFunction(packagePOSReq);

#ifdef FASTPAY
#else
    RunFunction(copyPros);
#endif
    RunFunction(setWindow);
    RunFunction(showWindow);
    RunFunction(packagePOSRsp);

    QJsonDocument json(posRspJsonObj);
    return json.toJson(QJsonDocument::Compact);
}

void FMTask::stopTask()
{
    FMP_DEBUG() << __FUNCTION__;
    if(preTask != nullptr) {
        preTask->stopTask();
    }
    if(_window != nullptr) {
        _window->reject();
        FMMsgWnd::WarningWnd(QString::fromLocal8Bit("通讯中断，交易取消"), _window);
        _window = nullptr;
    }

    setError(FM_API_WINDOWCLOSE);
}

void FMTask::copyPros()
{
    FMP_DEBUG() << __FUNCTION__;
    for(int i=0; i<sizeof(DefaultProps)/sizeof(DefaultProps[0]); ++i) {
        QString posPro = DefaultProps[i];
        serverReqJsonObj[ServerProps(posPro)] = posReqJsonObj[posPro];
    }
}

void FMTask::showWindow()
{
    FMP_DEBUG() << __FUNCTION__;
    QString sL[] = {
        PosProps.StoreId,
        PosProps.PosId,
        PosProps.OperatorId,
        PosProps.BussinessDate,
     };

    for(int i=0; i<sizeof(sL)/sizeof(sL[0]); ++i) {
        QString s = sL[i];
        _session->addData(s, posReqJsonObj[s].toString());
    }

    if(_window != nullptr) {
        _window->initWnd(this->session());
        int ret = _window->exec();
        if(ret != 1) {
            setError(FM_API_WINDOWCLOSE);
        }
    }
}

bool FMTask::sendToServer(bool isShowMsg)
{
    FMP_DEBUG() << __FUNCTION__;

    url = FMPVipSettings::instance()->getServerUrl();
    // 固定部分
#ifdef FASTPAY


//    QJsonObject hrtAttrs;
//    hrtAttrs[ServerProps(PosProps._PartnerId)] = H_PARTNER_ID;
//    hrtAttrs[ServerProps(PosProps.ApiVersion)] = API_VERSION;
//    hrtAttrs[ServerProps(PosProps.AppSubID)] = APP_SUB_ID;
//    hrtAttrs[ServerProps(PosProps.Format)] = FORMAT;
//    hrtAttrs[ServerProps(PosProps.TimeStamp)] = QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss:zzz");
//    hrtAttrs[ServerProps(PosProps.AppToken)] = APP_TOKEN;
//    hrtAttrs[ServerProps(PosProps.SignMethod)] = SIGN_METHOD;
//    hrtAttrs[ServerProps(PosProps.SysID)] = SYS_ID;

//    QJsonObject request;
//    request[ServerProps(PosProps.HrtAttrs)] = hrtAttrs;

//    serverReqJsonObj["REQUEST"] = request;
//    packageServerReq();

//    QJsonObject req = serverReqJsonObj["REQUEST"].toObject();
//    QJsonObject hrt = req[ServerProps(PosProps.HrtAttrs)].toObject();
//    hrt[ServerProps(PosProps._Sign)] = sign();
//    req[ServerProps(PosProps.HrtAttrs)] = hrt;
//    serverReqJsonObj["REQUEST"] = req;


//    qDebug() << serverReqJsonObj;
    packageServerReq();
    url = url  + ReqUrl.at(FM_Type());
#else
    serverReqJsonObj[PosProps.AppId] = APP_ID;
    serverReqJsonObj[ServerProps(PosProps.PartnerId)] = PARTNER_ID;
    serverReqJsonObj[PosProps.T] = QString::number(QDateTime::currentMSecsSinceEpoch());

    serverReqJsonObj[ServerProps(PosProps.Fm_cmd)] = QString::number(FM_Type());

    packageServerReq();
    serverReqJsonObj[PosProps.Sign] = sign();

    url = url + "/" + ReqUrl.at(FM_Type());

#endif

    QJsonDocument json(serverReqJsonObj);
    QByteArray data = json.toJson(QJsonDocument::Compact);
    QByteArray rspData;

    FMNetwork net;
    net.send(url, data, rspData);

    FMP_INFO() << "Server rsponse: " << rspData.data();

    // 网络错误
    if(net.error != FM_API_SUCCESS) {
        setError(net.error, net.errorMsg);
    } else {
        QJsonParseError jsonErr;
        QJsonDocument rspJson = QJsonDocument::fromJson(rspData, &jsonErr);

        // Json错误
        if(jsonErr.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON);
        } else {
            serverRspJsonObj = rspJson.object();

            // 服务器返回的错误
            if(serverRspJsonObj.contains("errcode")) {
                setError(FM_API_SERVERERROR, serverRspJsonObj["errcode"].toInt(), serverRspJsonObj["errmsg"].toString());
            }
        }
    }

    if(_window != nullptr) {
        _window->setIsBusy(false);
    }

    bool isOk = (error() == FM_API_SUCCESS);
    if(!isOk && isShowMsg)
    {
        FMMsgWnd::FailureWnd(errorString(), _window);
    }

    return isOk;
}

QJsonValue FMTask::searchJsonValue(QJsonObject &searchJson, QString searchKey)
{
    if(searchJson.contains(searchKey)) {
        return searchJson[searchKey];
    } else {
        foreach(QString key , searchJson.keys()) {
            if(searchJson[key].isObject()) {
                QJsonObject ob = searchJson[key].toObject();
                QJsonValue value = searchJsonValue(ob, searchKey);
                if(!value.isNull()){
                    return value;
                }
            }
        }
    }
    return QJsonValue();
}

QJsonValue FMTask::getServerJsonValue(const QString prop)
{
    return searchJsonValue(serverRspJsonObj, ServerProps(prop));
}

QJsonValue FMTask::getPosJsonValue(const QString prop)
{
    return searchJsonValue(posReqJsonObj, prop);
}

QString FMTask:: sign()
{
    FMP_DEBUG() << __FUNCTION__;
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;

    for(int i=0; i<sizeof(SignProps)/sizeof(SignProps[0]); ++i) {
        QString word = SignProps[i];

        QJsonValue wordJson = searchJsonValue(serverReqJsonObj, ServerProps(word));

        if(wordJson.isDouble()) {
            mapData[word] = QString::number(wordJson.toDouble());
        } else {
            mapData[word] = wordJson.toString();
        }
    }

    if (serverRspJsonObj[ServerProps(PosProps.Fm_cmd)].toInt() == FM_Fund) {
        mapData[ServerProps(PosProps.TransId)] = serverReqJsonObj[ServerProps(PosProps.TransId)].toString();
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }

#ifdef FASTPAY
    qDebug() <<"||||||||||||||||||||||" <<PUBLIC_KEY;
    sb.append(PUBLIC_KEY);
#else
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KEY_CODE);
#endif
    qDebug() << sb;
    QByteArray bt;
    bt.append(sb);

    FMP_INFO() << "Sign String: " << bt;

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}

QByteArray FMTask::JsonObjToMap(const QJsonObject &jsonObj, QString parentJson)
{
    parentJson = parentJson.isEmpty() ? "" : parentJson+".";
    QByteArray signArray;
    foreach (const QString &key, jsonObj.keys()) {
        switch (jsonObj[key].type()) {
        case QJsonValue::Array:
            break;
        case QJsonValue::Null:
            signArray.append(QString("%1=&").arg(parentJson+key));
            break;
        case QJsonValue::Bool:
            signArray.append(QString("%1=%2&").arg(parentJson+key).arg(jsonObj[key].toBool() ? "true" : "false"));
            break;
        case QJsonValue::Double:
            signArray.append(QString("%1=%2&").arg(parentJson+key).arg(QString::number(jsonObj[key].toDouble())));
            break;
        case QJsonValue::String:
            signArray.append(QString("%1=%2&").arg(parentJson+key).arg(jsonObj[key].toString()));
            break;
        case QJsonValue::Object:
            signArray.append(JsonObjToMap(jsonObj[key].toObject(), parentJson+key));
            break;
        default:
            break;
        }
    }
    if(parentJson.isEmpty()) {
        signArray = signArray.mid(0, signArray.length()-1);
#ifdef FMTEST
        signArray.append("freemud");
#else
        signArray.append(KEY_CODE);
#endif
    }
    return signArray;
}

QString FMTask::Sign(const QJsonObject &signJsonObj)
{
    QByteArray signArray = JsonObjToMap(signJsonObj);
    FMP_INFO() << "Sign byte array:" << JsonObjToMap(signJsonObj);
    QString md5str = QCryptographicHash::hash(signArray, QCryptographicHash::Md5).toHex();
    return md5str;
}

bool FMTask::checkReqJson()
{
    bool isOk = false;
    QString filePath = qApp->applicationDirPath()+"/FMCMDKeys.json";
    QFile jsonFile(filePath);
    if(!jsonFile.open(QFile::ReadOnly)) {
        setError(FM_API_BADJSON, QString::fromLocal8Bit("读取%1失败(%2).").arg(filePath).arg(jsonFile.errorString()));
    } else {
        QByteArray jsonArray = jsonFile.readAll();
        QJsonParseError error;
        QJsonDocument jsonDoc = QJsonDocument::fromJson(jsonArray, &error);
        if(error.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON, QString::fromLocal8Bit("解析FMCMDKeys文件内容失败(%1)").arg(error.errorString()));
        } else {
            QJsonObject jsonObj = jsonDoc.object();
            _checkJsonObj = jsonObj[QString::number(FM_Type())].toObject();
            isOk = true;
        }
    }
    jsonFile.close();

    if(isOk) {
        QJsonObject mandatoryFieldObj = _checkJsonObj[PosProps.MandatoryField].toObject();
        QJsonObject optionalFieldObj = _checkJsonObj[PosProps.OptionalField].toObject();
        QStringList checkKeys = mandatoryFieldObj.keys() + optionalFieldObj.keys();
        foreach (const QString &checkKey, checkKeys) {
            if(!posReqJsonObj.contains(checkKey)) {
                if(mandatoryFieldObj.contains(checkKey)) {
                    isOk = false;
                    setError(FM_API_BADJSON, QString::fromLocal8Bit("请求不合法，需要[%1]字段。").arg(checkKey));
                    break;
                } else if(optionalFieldObj.contains(checkKey)) {
                    break;
                }
            }

            QVariant expectValue;
            if(mandatoryFieldObj.contains(checkKey)) {
                expectValue = mandatoryFieldObj[checkKey].toVariant();
            } else {
                expectValue = optionalFieldObj[checkKey].toVariant();
            }
            QVariant actualValue = posReqJsonObj[checkKey].toVariant();
            QVariant::Type expectType = expectValue.type();
            QVariant::Type actualType = actualValue.type();

            if(expectType == QVariant::Double) {
                //! NOTE 判断期待是否是int类型
                double tempExpect = expectValue.toDouble();
                if(tempExpect == (int)tempExpect) {
                    expectType = QVariant::Int;
                }
            }

            //! NOTE 当期待是int类型时，判断实际值是否是int类型
            if(expectType==QVariant::Int && actualType == QVariant::Double) {
                double tempActual = actualValue.toDouble();
                if(tempActual == (int)tempActual) {
                    actualType = QVariant::Int;
                }
            }

            if(mandatoryFieldObj.contains(checkKey) && actualType == QVariant::String && posReqJsonObj[checkKey].toString().isEmpty()) {
                isOk = false;
                setError(FM_API_BADJSON, QString::fromLocal8Bit("请求不合法，字段[%1]值不能为空。").arg(checkKey));
                break;
            }

            if(expectType != actualType) {
                isOk = false;
                setError(FM_API_BADJSON, QString::fromLocal8Bit("请求不合法，字段[%1]应该为 %2 类型。")
                         .arg(checkKey)
                         .arg(QVariant::typeToName(expectType)));
                break;
            }
        }
    }

    return isOk;
}
