﻿#include "fmp_vip_server.h"
#include "fmvipdispatcher.h"
#include "fmp_vip_settings.h"
#include <QTcpSocket>
#include <QDebug>

FMPVipServer *FMPVipServer::instance()
{
    static FMPVipServer server;
    return &server;
}

FMPVipServer::FMPVipServer() :
    QTcpServer(),
    socket(nullptr),
    dispatcher(new FMVipDispatcher(this))
{
    Listen(23770);
    connect(this, &QTcpServer::newConnection, this, &FMPVipServer::onNewConnection);
    isNeedSocketHeader = FMPVipSettings::instance()->getIsNeedSocketHeader();
}

FMPVipServer::~FMPVipServer()
{
    if(dispatcher!=nullptr) {
        delete dispatcher;
    }
}

void FMPVipServer::Listen(quint16 port)
{
    if(!this->listen(QHostAddress::Any, port)) {
        FMP_ERROR() << "Listen error: " << this->errorString();
    }
}

void FMPVipServer::Write(const QByteArray &data)
{
    if(!socket || socket->write(data) == -1) {
        FMP_ERROR() << "Write error: " << (socket ? socket->errorString() : "connection closed.");
    }
}

void FMPVipServer::SetPluginContext(ctkPluginContext *ctx)
{
    dispatcher->setPluginContext(ctx);
}

void FMPVipServer::onNewConnection()
{
    socket = nextPendingConnection();

    connect(socket, &QTcpSocket::disconnected, this, &FMPVipServer::onDisconnected);
    connect(socket, &QTcpSocket::readyRead, this, &FMPVipServer::onReadyRead);
}

void FMPVipServer::onDisconnected()
{
    FMP_DEBUG() << "Socket disconnected.";

    if(dispatcher!=nullptr) {
        dispatcher->stopTask();
    }

    emit SocketDisconnected();
}

void FMPVipServer::onReadyRead()
{
    QByteArray recvData;
    int needLen = -1, totalLen=0, dataLen = 0;

    while(needLen == -1 || totalLen < needLen)
    {
        // 如果不是第一次接收数据，则阻塞
        if(totalLen > 0) {
            bool isNext = socket->waitForReadyRead(30*1000);
            if(!isNext) {
                FMP_ERROR() << "Wait recv data time out. Recved length: " << totalLen << " need length: " << needLen << " Recved data: " << recvData;
                socket->close();
                return;
            }
        }
        QByteArray tempData = socket->readAll();

        totalLen += tempData.size();
        recvData.append(tempData);

        if(isNeedSocketHeader) {
            // 消息头接收完成
            if(needLen == -1 && totalLen >= sizeof(FMSOCKEHEADER)) {
                FMSOCKEHEADER header = {0};
                memcpy(&header, recvData.data(), sizeof(FMSOCKEHEADER));
                if(header.flag != FMSOCKFLAG) {
                    FMP_ERROR() << "Incompatible protocol. Recved data: " << recvData;
                    socket->close();
                    return;
                } else {
                    dataLen = header.len;
                    needLen = header.len + sizeof(FMSOCKEHEADER);
                }
            }
        } else {
            dataLen = tempData.size();
            needLen = tempData.size();
        }
    }
    if(isNeedSocketHeader) {
        recvData = recvData.mid(sizeof(FMSOCKEHEADER));
    }
    if(totalLen > needLen) {
        FMP_WARN() << "Recv total length: " << totalLen << " need length: " << needLen;
        recvData = recvData.mid(0, dataLen);
    }

    QByteArray rspData;
    dispatcher->doTask(recvData, rspData);

    Write(rspData);

    if(socket != nullptr && socket->isValid() && socket->isOpen()) {
        socket->waitForBytesWritten();
        socket->close();
    }
}


void FMPVipServer::callVipPay(const QByteArray &reqData, QByteArray &rspData)
{
    dispatcher->doTask(reqData,rspData);
}
