#include "dbop.h"
#include "fmp_database.h"
#include "items/item.h"
#include "items/order.h"
#include "items/pay.h"
#include "items/scorelist.h"
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSharedPointer>
#include <fmp_logger_i.h>
#include <QDebug>

using namespace FMItem;

DBOP::DBOP()
{
    _db = new FMPDataBase("FreemudPOS.db", "fmp_order_manage");
}

DBOP *DBOP::Instance()
{
    static DBOP dbOp;
    return &dbOp;
}

void DBOP::InsertTablePay(const QString &transId, const QString &fmId, const QByteArray &reqData, const QByteArray &rspData)
{
    InserTable(TABLE_PAY, transId, fmId, reqData, rspData);
}

void DBOP::InsertTableOrder(const QString &transId, const QString &fmId, const QByteArray &reqData, const QByteArray &rspData)
{
    InserTable(TABLE_ORDER, transId, fmId, reqData, rspData);
}


void DBOP::InserTable(const QString &tableName, const QString &transId, const QString &fmId, const QByteArray &reqData, const QByteArray &rspData)
{
    QVariantHash hash;
    hash[PosProps.TransId] = transId;
    hash[PosProps.Fm_order_id] = fmId;
    hash["reqData"]  = reqData;
    hash["rspData"]  = rspData;
    Instance()->_db->insert(tableName, hash);
}

bool DBOP::CreateTable(const QString &tableName)
{
    QString tableOrderSql = "CREATE TABLE [Order]("
            "[id] INTEGER PRIMARY KEY autoincrement, "
            "[order_id] VARCHAR(50), "
            "[fm_order_id] VARCHAR(50), "
            "[order_amount] INT DEFAULT 0, "
            "[paid_amount] INT DEFAULT 0, "
            "[undis_amount] INT DEFAULT 0, "
            "[fm_open_id] VARCHAR(50), "
            "[products] TEXT,"
            "[settled] BOOL DEFAULT 0,"
            "[refunded] BOOL DEFAULT 0,"
            "[storeInfoDBID] INTEGER DEFAULT 0"
            ");";
    QString tablePaySql = "CREATE TABLE [Pay]("
            "[id] INTEGER PRIMARY KEY autoincrement, "
            "[pay_id] VARCHAR(50), "
            "[pay_str] VARCHAR(50), "
            "[pay_amount] INT DEFAULT 0, "
            "[refund_amount] INT DEFAULT 0, "
            "[dis_amount] INT DEFAULT 0, "
            "[account] VARCHAR(50), "
            "[trans_id] VARCHAR(50), "
            "[fm_trans_id] VARCHAR(50),"
            "[third_trans_id] VARCHAR(50),"
            "[orderDBID] INTEGER DEFAULT 0"
            ");";
    QString tableStoreInfoSql = "CREATE TABLE [StoreInfo]("
            "[id] INTEGER PRIMARY KEY autoincrement, "
            "[store_id] VARCHAR(50), "
            "[pos_id] VARCHAR(50), "
            "[operator_id] VARCHAR(50), "
            "[business_date] VARCHAR(50)"
            ");";


    QString tableScoreListSql = "CREATE TABLE [ScoreList]("
            "[listId] VARCHAR(50) PRIMARY KEY, "
            "[score] INTEGER, "
            "[couponId] VARCHAR(50), "
            "[listTime] VARCHAR(50), "
            ");";


    QString sql;
    if(tableName.compare(TABLE_ORDER) == 0) {
        sql = tableOrderSql;
    } else if(tableName.compare(TABLE_PAY) == 0) {
        sql = tablePaySql;
    } else if(tableName.compare(TABLE_STOREINFO) == 0) {
        sql = tableStoreInfoSql;
    }else if(tableName.compare(TABLE_SCORELIST) == 0) {
        sql = tableScoreListSql;
    }

    bool isOk = _db->creat(sql);
    qDebug() << "Create table: " << tableName << (isOk ? " Success." : " Failed.");
    return isOk;
}

bool DBOP::ExistTable(const QString &tableName)
{
    QStringList tables = _db->getDb().tables();
    return tables.contains(tableName);
}

int DBOP::GetLastInsertId()
{
    QSqlQuery query(Instance()->_db->getDb());
    bool isOk = query.exec("select last_insert_rowid();");
    int id = -1;
    if(isOk && query.next()) {
        id = query.record().value(0).toInt();
    }
    return id;
}

bool DBOP::Save(FMItem::Item *item)
{
    QString tableName = item->ShortClassName();
    bool isOk = false;

    if(!Instance()->ExistTable(tableName)) {
        if(!Instance()->CreateTable(tableName)) {
            return false;
        }
    }

    QVariantHash hash;
    foreach (const QString &key, item->PropertyNames()) {
        hash.insert(key, item->property(key.toUtf8().data()));
    }

    if(item->DBID() > 0) {
        //! Update
        QString condition =  QString("id=='%1'").arg(item->DBID());
        QSqlQuery query;
        isOk = Instance()->_db->find(tableName, query, QStringList(), condition);
        if(isOk && query.next()) {
            isOk = Instance()->_db->update(tableName, hash, condition);
        }
    } else {
        //! Insert
        isOk = Instance()->_db->insert(tableName, hash);
        if(isOk) {
            item->setDBID(Instance()->GetLastInsertId());
        }
    }

    return isOk;
}

void DBOP::GetOrderByOrderId(const QString &orderId, Order *order)
{
    Q_ASSERT(order != NULL);

    QSqlQuery query;
    QString condition = QString("order_id == '%1'").arg(orderId);
    bool isOk = Instance()->_db->find(TABLE_ORDER, query, QStringList(), condition);
    if(isOk && query.next()) {
        order->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find order: " << orderId;
    }
}

QSharedPointer<FMItem::Order> DBOP::GetOrderByOrderId(const QString &orderId)
{
    QSharedPointer<FMItem::Order> order(new Order);
    GetOrderByOrderId(orderId, order.data());
    return order;
}

void DBOP::GetOrderByDBId(unsigned int DBID, Order *order)
{
    Q_ASSERT(order != NULL);

    QSqlQuery query;
    QString condition = QString("id == '%1'").arg(DBID);
    bool isOk = Instance()->_db->find(TABLE_ORDER, query, QStringList(), condition);
    if(isOk && query.next()) {
        order->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find order: " << DBID;
    }
}

QSharedPointer<FMItem::Order> DBOP::GetOrderByDBId(unsigned int DBID)
{
    QSharedPointer<FMItem::Order> order(new Order);
    GetOrderByDBId(DBID, order.data());
    return order;
}

QList<QSharedPointer<FMItem::Pay> > DBOP::GetPaysByOrderDBId(int orderDBID)
{
    QList<QSharedPointer<FMItem::Pay> > payList;
    QSqlQuery query;
    QString condition = QString("orderDBID == '%1'").arg(orderDBID);
    Instance()->_db->find(TABLE_PAY, query, QStringList(), condition);
    while(query.next()) {
        QSharedPointer<FMItem::Pay> payPointer(new FMItem::Pay());
        payPointer->SetPropertiesBySqlRecord(query.record());
        payList.append(payPointer);
    }
    return payList;
}

void DBOP::GetPayByDBId(unsigned int DBID, Pay *pay)
{
    Q_ASSERT(pay != NULL);

    QSqlQuery query;
    QString condition = QString("id == '%1'").arg(DBID);
    bool isOk = Instance()->_db->find(TABLE_PAY, query, QStringList(), condition);
    if(isOk && query.next()) {
        pay->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find pay: " << DBID;
    }
}

QSharedPointer<FMItem::Pay> DBOP::GetPayByDBId(unsigned int DBID)
{
    QSharedPointer<FMItem::Pay> pay(new Pay);
    GetPayByDBId(DBID, pay.data());
    return pay;
}

void DBOP::GetListByListId(const QString &listId, ScoreList *scoreList)
{
    Q_ASSERT(scoreList != NULL);

    QSqlQuery query;
    QString condition = QString("listId == '%1'").arg(listId);
    bool isOk = Instance()->_db->find(TABLE_SCORELIST, query, QStringList(), condition);
    if(isOk && query.next()) {
        scoreList->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find scoreList: " << listId;
    }
}

QSharedPointer<ScoreList> DBOP::GetListByListId(const QString &listId)
{
    QSharedPointer<FMItem::ScoreList> scoreList(new ScoreList);
    GetListByListId(listId, scoreList.data());
    return scoreList;
}

void DBOP::GetListByDBId(unsigned int DBID, ScoreList *scoreList)
{
    Q_ASSERT(scoreList != NULL);

    QSqlQuery query;
    QString condition = QString("id == '%1'").arg(DBID);
    bool isOk = Instance()->_db->find(TABLE_SCORELIST, query, QStringList(), condition);
    if(isOk && query.next()) {
        scoreList->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find scoreList: " << DBID;
    }
}

QSharedPointer<ScoreList> DBOP::GetListByDBId(unsigned int DBID)
{
    QSharedPointer<FMItem::ScoreList> scoreList(new ScoreList);
    GetListByDBId(DBID, scoreList.data());
    return scoreList;
}

bool DBOP::GetLastItem(Item *item)
{
    Q_ASSERT(item != nullptr);

    QSqlQuery query;
    QString condition = "1=1 ORDER BY id DESC LIMIT 1";
    bool isOk = Instance()->_db->find(item->ShortClassName(), query, QStringList(), condition);
    if(isOk && query.next()) {
        item->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Dont't find last item in table: " << item->ShortClassName();
    }
    return isOk;
}

//template<typename T>
//QSharedPointer<T> DBOP::GetLastItem()
//{
//    QSharedPointer<T> itemPointer(new T());
//    GetLastItem(itemPointer.data());
//    return itemPointer;
//}

bool DBOP::GetItemByDBID(unsigned int DBID, Item *item)
{
    Q_ASSERT(item != NULL);

    QSqlQuery query;
    QString condition = QString("id == '%1'").arg(DBID);
    bool isOk = Instance()->_db->find(item->ShortClassName(), query, QStringList(), condition);
    if(isOk && query.next()) {
        item->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find item by DBID: " << DBID;
    }
    return isOk;
}
