﻿#include "fmtask.h"
#include "fmnetwork.h"
#include <QDateTime>
#include <QSettings>
#include <QJsonDocument>
#include <QJsonParseError>
#include <QCryptographicHash>
#include "fmp_vip_settings.h"

#define RunFunction(function)   do {    \
        function##();                   \
        if(error() != FM_API_SUCCESS) { \
            FMP_WARN() << QString("Task error %1: %2").arg(error()).arg(errorString());   \
            return QString(ErrorMsgJson).arg(error()).arg(errorString()).toLatin1();    \
        }                               \
} while(0);

FMTask::FMTask(QJsonObject &jsonObj, FM_TYPE fmType, Session *session, QObject *parent) :
    QObject(parent),
    posReqJsonObj(jsonObj),
    _window(nullptr),
    isCreator(false),
    _FM_Type(fmType),
    preTask(nullptr),
    _error(FM_API_SUCCESS),
    _errorMsg("")
{
    if(session == 0) {
        _session = new Session();
        isCreator = true;
    } else {
        _session = session;
    }
}

FMTask::~FMTask()
{
    del_p(_window);
    del_p(preTask);
    if(isCreator) {
        del_p(_session);
    }
}

QByteArray FMTask::doTask()
{
    FMP_DEBUG() << __FUNCTION__;
    RunFunction(copyPros);
    RunFunction(setWindow);
    RunFunction(showWindow);
    RunFunction(packagePOSRsp);

    QJsonDocument json(posRspJsonObj);
    return json.toJson(QJsonDocument::Compact);
}

void FMTask::stopTask()
{
    FMP_DEBUG() << __FUNCTION__;
    if(preTask != nullptr) {
        preTask->stopTask();
    }
    if(_window != nullptr) {
        _window->accept();
        _window = nullptr;
    }

    setError(FM_API_WINDOWCLOSE);
}

void FMTask::copyPros()
{
    FMP_DEBUG() << __FUNCTION__;
    for(int i=0; i<sizeof(DefaultProps)/sizeof(DefaultProps[0]); ++i) {
        QString posPro = DefaultProps[i];
        serverReqJsonObj[ServerProps(posPro)] = posReqJsonObj[posPro];
    }
}

void FMTask::showWindow()
{
    FMP_DEBUG() << __FUNCTION__;
    QString sL[] = {
         PosProps.OperatorId,
         PosProps.BussinessDate,
     };

    for(int i=0; i<sizeof(sL)/sizeof(sL[0]); ++i) {
        QString s = sL[i];
        _session->addData(s, posReqJsonObj[s].toString());
    }

    if(_window != nullptr) {
        _window->initWnd(this->session());
        int ret = _window->exec();
        if(ret != 1) {
            setError(FM_API_WINDOWCLOSE);
        }
    }
}

bool FMTask::sendToServer(bool isShowMsg)
{
    FMP_DEBUG() << __FUNCTION__;
    // 固定部分
    serverReqJsonObj[PosProps.AppId] = APP_ID;
    serverReqJsonObj[PosProps.PartnerId] = PARTNER_ID;
    serverReqJsonObj[PosProps.T] = QString::number(QDateTime::currentMSecsSinceEpoch());

    serverReqJsonObj[ServerProps(PosProps.Fm_cmd)] = QString::number(FM_Type());
    serverReqJsonObj[PosProps.Sign] = sign();
    packageServerReq();

    QJsonDocument json(serverReqJsonObj);
    QByteArray data = json.toJson(QJsonDocument::Compact);

    url = FMPVipSettings::instance()->getServerUrl() + "/" + ReqUrl.at(FM_Type());
    QByteArray rspData;

    FMNetwork net;
    net.send(url, data, rspData);

    FMP_INFO() << "Server rsponse: " << rspData;

    // 网络错误
    if(net.error != FM_API_SUCCESS) {
        setError(net.error, net.errorMsg);
    } else {
        QJsonParseError jsonErr;
        QJsonDocument rspJson = QJsonDocument::fromJson(rspData, &jsonErr);

        // Json错误
        if(jsonErr.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON);
        } else {
            serverRspJsonObj = rspJson.object();

            // 服务器返回的错误
            if(serverRspJsonObj.contains("errcode")) {
                setError(FM_API_SERVERERROR, serverRspJsonObj["errcode"].toInt(), serverRspJsonObj["errmsg"].toString());
            }
        }
    }

    if(_window != nullptr) {
        _window->setIsBusy(false);
    }

    bool isOk = (error() == FM_API_SUCCESS);
    if(!isOk && isShowMsg)
    {
        FMMsgWnd::FailureWnd(errorString(), _window);
    }

    return isOk;
}

QJsonValue FMTask::searchJsonValue(QJsonObject &searchJson, QString searchKey)
{
    if(searchJson.contains(searchKey)) {
        return searchJson[searchKey];
    } else {
        foreach(QString key , searchJson.keys()) {
            if(searchJson[key].isObject()) {
                QJsonObject ob = searchJson[key].toObject();
                QJsonValue value = searchJsonValue(ob, searchKey);
                if(!value.isNull()){
                    return value;
                }
            }
        }
    }
    return QJsonValue();
}

QJsonValue FMTask::getServerJsonValue(const QString prop)
{
    return searchJsonValue(serverRspJsonObj, ServerProps(prop));
}

QJsonValue FMTask::getPosJsonValue(const QString prop)
{
    return searchJsonValue(posReqJsonObj, prop);
}

QString FMTask::sign() const
{
    FMP_DEBUG() << __FUNCTION__;
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;

    for(int i=0; i<sizeof(SignProps)/sizeof(SignProps[0]); ++i) {
        QString word = SignProps[i];
        if(serverReqJsonObj[word].isDouble()) {
            mapData[word] = QString::number(serverReqJsonObj[word].toDouble());
        } else {
            mapData[word] = serverReqJsonObj[word].toString();
        }
    }

    if (serverRspJsonObj[ServerProps(PosProps.Fm_cmd)].toInt() == FM_Fund) {
        mapData[ServerProps(PosProps.TransId)] = serverReqJsonObj[ServerProps(PosProps.TransId)].toString();
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KEY_CODE);

    QByteArray bt;
    bt.append(sb);

    FMP_INFO() << "Sign String: " << bt;

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}
