#include <QtTest>
#include "global.h"
#include "items/item.h"
#include "items/order.h"
#include "items/pay.h"
#include "items/product.h"
#include "fmtool.h"
#include "database/dbop.h"
#include <QJsonObject>

// add necessary includes here

using FMItem::Item;
using FMItem::Order;
using FMItem::Pay;
using FMItem::Product;

class TestItem : public QObject
{
    Q_OBJECT

public:
    TestItem();
    ~TestItem();


    Item *gItem;
    Order *gOrder;
    Item *gPay;
    Item *gProduct;


    QJsonObject json1;
    QJsonObject json2;

private slots:
    void initTestCase();
    void cleanupTestCase();

    void test_SetPropertiesByJson_data();
    void test_SetPropertiesByJson();

    void test_PropertyNames_data();
    void test_PropertyNames();

    void test_DBSave_data();
    void test_DBSave();

    void test_GetOrderByTransId_data();
    void test_GetOrderByTransId();
};

TestItem::TestItem()
{

}

TestItem::~TestItem()
{

}

void TestItem::initTestCase()
{
    gItem = new Item();
    gOrder = new Order();
    gPay = new Pay();
    gProduct = new Product();

    json1["trans_id"] = "11111";
    json1["fm_id"] = "11111";
    json1["order_amount"] = 100;
    json1["canPay"] = true;

    json2["trans_id"] = "22222";
    json2["canPay"] = false;
}

void TestItem::cleanupTestCase()
{
    delete gItem;
    delete gOrder;
    delete gPay;
    delete gProduct;
}

void TestItem::test_SetPropertiesByJson_data()
{
    QTest::addColumn<QJsonObject>("json");
    QTest::addColumn<QString>("transId");
    QTest::addColumn<QString>("fmId");
    QTest::addColumn<int>("orderAmount");
    QTest::addColumn<bool>("canPay");

//    QJsonArray pays;
//    QJsonObject pay1, pay2;
//    pay1[PosProps.Pay_id] = "1";
//    pay2[PosProps.Pay_id] = "2";
//    pays << pay1 << pay2;
//    json1["payList"] = pays;

//    FMItem::ItemList payList;
//    Pay *p1 = new Pay();
//    p1->setPayId("1");
//    Pay *p2 = new Pay();
//    p2->setPayId("2");
//    payList << p1 << p2;


    QTest::newRow("Whole") << json1 << "11111" << "11111" << 100 << true;
    QTest::newRow("Part") << json2 << "22222" << "11111" << 100 << false;
}

void TestItem::test_SetPropertiesByJson()
{
    QFETCH(QJsonObject, json);
    QFETCH(QString, transId);
    QFETCH(QString, fmId);
    QFETCH(int, orderAmount);
    QFETCH(bool, canPay);
//    QFETCH(FMItem::ItemList, payList);

    Order order;
    order.SetPropertiesByJson(json);

    QCOMPARE(order.transId(), SearchJsonValue(json, "trans_id").toString());
    QCOMPARE(order.fmId(), SearchJsonValue(json, "fm_id").toString());
    QCOMPARE(order.orderAmount(), SearchJsonValue(json, "order_amount").toInt());
    QCOMPARE(order.property("canPay").toBool(), SearchJsonValue(json, "canPay").toBool());
//    QCOMPARE(order.payList(), payList);

    gOrder->SetPropertiesByJson(json);

    QCOMPARE(gOrder->transId(), transId);
    QCOMPARE(gOrder->fmId(), fmId);
    QCOMPARE(gOrder->orderAmount(), orderAmount);
    QCOMPARE(gOrder->property("canPay").toBool(), canPay);
//    QCOMPARE(gOrder->payList(), payList);
}

void TestItem::test_PropertyNames_data()
{

    QTest::addColumn<Item*>("item");
    QTest::addColumn<int>("propertyCount");
    QTest::addColumn<QStringList>("propertyNames");

    QStringList itemPNames, orderPNames, payPNames, productPNames;

    itemPNames << "objectName" << "DBID";
    QTest::newRow("Item") << gItem << itemPNames.length() << itemPNames;

    orderPNames << "objectName" << "DBID" <<"trans_id" << "fm_id" << "order_amount" << "paid_amount" << "undis_amount"
                << "store_id" << "pos_id" << "operator_id" << "business_date" << "fm_open_id" << "products" << "settled";
    QTest::newRow("Order") << qobject_cast<Item*>(gOrder) << orderPNames.length() << orderPNames;

    payPNames << "objectName" << "DBID" <<"pay_id" << "pay_str" << "pay_amount" << "discount_amount" << "account" << "trans_id" << "payTransId" << "orderId";
    QTest::newRow("Pay") << gPay << payPNames.length() << payPNames;

    productPNames << "objectName" << "DBID" << "id" <<"quantity" << "price" << "name";
    QTest::newRow("Product") << gProduct << productPNames.length() << productPNames;
}

void TestItem::test_PropertyNames()
{
    QFETCH(Item *, item);
    QFETCH(int, propertyCount);
    QFETCH(QStringList, propertyNames);

    QCOMPARE(item->PropertyCount(), propertyCount);
    QCOMPARE(item->PropertyNames(), propertyNames);
}

void TestItem::test_DBSave_data()
{
    QTest::addColumn<Item*>("item");
    QTest::addColumn<bool>("isSuccess");

    Pay *pay = new Pay();
    pay->setPayId("1");
    pay->setPayStr("pay1");
    pay->setTransId("1111");
    pay->setPayAmount(100);

    QTest::newRow("Pay") << qobject_cast<Item*>(pay) << true;
    QTest::newRow("Order") << qobject_cast<Item*>(gOrder) << true;
    gOrder->setDBID(gOrder->DBID());
    gOrder->setFmId("11111");
    gOrder->setTransId("11111");
    gOrder->setOrderAmount(1000);
    gOrder->setProductText("{\"pid\":\"1\"},{\"pid\":\"2\"}");
    QTest::newRow("Order") << qobject_cast<Item*>(gOrder) << true;
}

void TestItem::test_DBSave()
{
    QFETCH(Item *, item);
    QFETCH(bool, isSuccess);

    bool isOk = DBOP::Save(item);

    QCOMPARE(isOk, isSuccess);
    QVERIFY(item->DBID() > 0);
}

void TestItem::test_GetOrderByTransId_data()
{
    QTest::addColumn<Item*>("in_order");
    QTest::addColumn<QString>("in_transId");
    QTest::addColumn<Item*>("in_pay");

    FMItem::Item *order = new Order();
    order->SetPropertiesByJson(json1);
    qsrand(QDateTime::currentMSecsSinceEpoch());
    QString transId = QDateTime::currentDateTimeUtc().toString("yyyyMMddhhmmsszzz") + QString::number(qrand());
    order->setProperty("trans_id", transId);

    FMItem::Item *pay = new Pay();
    pay->setProperty("trans_id", transId);

    QTest::newRow("Order1") << order << transId << pay;
}

void TestItem::test_GetOrderByTransId()
{
    QFETCH(Item*, in_order);
    QFETCH(QString, in_transId);
    QFETCH(Item*, in_pay);

    bool isOk = DBOP::Save(in_order);
    QVERIFY2(isOk, "Save in_order failed.");

    in_pay->setProperty("orderId", in_order->DBID());
    isOk = DBOP::Save(in_pay);
    QVERIFY2(isOk, "Save in_pay failed.");

    Order* order = new Order();
    DBOP::GetOrderByTransId(in_transId, order);

    QVERIFY(order->isEqual(in_order));
    delete order;

    QSharedPointer<Order> orderPointer = DBOP::GetOrderByTransId(in_transId);
    QVERIFY(orderPointer.data()->isEqual(in_order));

    QList<QSharedPointer<Pay> > payList;
    payList = DBOP::GetPaysByOrderId(in_order->DBID());
    QVERIFY(payList.length() == 1);

    QVERIFY(payList.first().data()->isEqual(in_pay));
}

QTEST_MAIN(TestItem)

#include "tst_testitem.moc"
