#include "dbop.h"
#include "fmp_database.h"
#include "items/item.h"
#include "items/order.h"
#include "items/pay.h"
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSharedPointer>
#include <fmp_logger_i.h>
#include <QDebug>

using namespace FMItem;

DBOP::DBOP()
{
    _db = new FMPDataBase("FreemudPOS.db", "fmp_order_manage");
}

DBOP *DBOP::Instance()
{
    static DBOP dbOp;
    return &dbOp;
}

void DBOP::InsertTablePay(const QString &transId, const QString &fmId, const QByteArray &reqData, const QByteArray &rspData)
{
    InserTable(TABLE_PAY, transId, fmId, reqData, rspData);
}

void DBOP::InsertTableOrder(const QString &transId, const QString &fmId, const QByteArray &reqData, const QByteArray &rspData)
{
    InserTable(TABLE_ORDER, transId, fmId, reqData, rspData);
}


void DBOP::InserTable(const QString &tableName, const QString &transId, const QString &fmId, const QByteArray &reqData, const QByteArray &rspData)
{
    QVariantHash hash;
    hash[PosProps.TransId] = transId;
    hash[PosProps.Fm_id] = fmId;
    hash["reqData"]  = reqData;
    hash["rspData"]  = rspData;
    Instance()->_db->insert(tableName, hash);
}

void DBOP::GetPaysByTransId(const QString &transId, QVector<QByteArray> &payDataVec)
{

}

bool DBOP::CreateTable(const QString &tableName)
{
    QString tableOrderSql = "CREATE TABLE [Order]("
            "[id] INTEGER PRIMARY KEY autoincrement, "
            "[trans_id] VARCHAR(50), "
            "[fm_id] VARCHAR(50), "
            "[order_amount] INT DEFAULT 0, "
            "[paid_amount] INT DEFAULT 0, "
            "[undis_amount] INT DEFAULT 0, "
            "[store_id] VARCHAR(50), "
            "[pos_id] VARCHAR(50), "
            "[operator_id] VARCHAR(50), "
            "[business_date] VARCHAR(50), "
            "[fm_open_id] VARCHAR(50), "
            "[products] TEXT"
            "[settle] BOOL DEFAULT 0"
            ");";
    QString tablePaySql = "CREATE TABLE [Pay]("
            "[id] INTEGER PRIMARY KEY autoincrement, "
            "[pay_id] VARCHAR(50), "
            "[pay_str] VARCHAR(50), "
            "[pay_amount] INT DEFAULT 0, "
            "[dis_amount] INT DEFAULT 0, "
            "[account] VARCHAR(50), "
            "[trans_id] VARCHAR(50), "
            "[payTransId] VARCHAR(50),"
            "[orderId] INTEGER"
            ");";

    QString sql;
    if(tableName.compare("Order") == 0) {
        sql = tableOrderSql;
    } else if(tableName.compare("Pay") == 0) {
        sql = tablePaySql;
    }
    bool isOk = _db->creat(sql);
    qDebug() << "Create table: " << tableName << (isOk ? " Success." : " Failed.");
    return isOk;
}

bool DBOP::ExistTable(const QString &tableName)
{
    QStringList tables = _db->getDb().tables();
    return tables.contains(tableName);
}

int DBOP::GetLastInsertId()
{
    QSqlQuery query(Instance()->_db->getDb());
    bool isOk = query.exec("select last_insert_rowid();");
    int id = -1;
    if(isOk && query.next()) {
        id = query.record().value(0).toInt();
    }
    return id;
}

bool DBOP::Save(FMItem::Item *item)
{
    QString tableName = item->ShortClassName();
    bool isOk = false;

    if(!Instance()->ExistTable(tableName)) {
        if(!Instance()->CreateTable(tableName)) {
            return false;
        }
    }

    QVariantHash hash;
    foreach (const QString &key, item->PropertyNames()) {
        hash.insert(key, item->property(key.toUtf8().data()));
    }

    if(item->DBID() > 0) {
        //! Update
        QString condition =  QString("id=='%1'").arg(item->DBID());
        QSqlQuery query;
        isOk = Instance()->_db->find(tableName, query, QStringList(), condition);
        if(isOk && query.next()) {
            isOk = Instance()->_db->update(tableName, hash, condition);
        }
    } else {
        //! Insert
        isOk = Instance()->_db->insert(tableName, hash);
        if(isOk) {
            item->setDBID(Instance()->GetLastInsertId());
        }
    }

    return isOk;
}

void DBOP::GetOrderByTransId(const QString &transId, Order *order)
{
    Q_ASSERT(order != NULL);

    QSqlQuery query;
    QString condition = QString("trans_id == '%1'").arg(transId);
    bool isOk = Instance()->_db->find(TABLE_ORDER, query, QStringList(), condition);
    if(isOk && query.next()) {
        order->SetPropertiesBySqlRecord(query.record());
    } else {
        FMP_WARN() << "Don't find order: " << transId;
    }
}

QSharedPointer<FMItem::Order> DBOP::GetOrderByTransId(const QString &transId)
{
    QSharedPointer<FMItem::Order> order(new Order);
    GetOrderByTransId(transId, order.data());
    return order;
}

QList<QSharedPointer<FMItem::Pay> > DBOP::GetPaysByOrderId(int orderId)
{
    QList<QSharedPointer<FMItem::Pay> > payList;
    QSqlQuery query;
    QString condition = QString("orderId == '%1'").arg(orderId);
    Instance()->_db->find(TABLE_PAY, query, QStringList(), condition);
    while(query.next()) {
        QSharedPointer<FMItem::Pay> payPointer(new FMItem::Pay());
        payPointer->SetPropertiesBySqlRecord(query.record());
        payList.append(payPointer);
    }
    return payList;
}
