﻿#include <QDebug>
#include <QDateTime>
#include <fmp_settings_i.h>
#include <QCoreApplication>
#include <ctkServiceReference.h>
#include "fmp_epay_p.h"
#include "fmp_epay_def.h"
#include "fmp_logger_i.h"
#include "fmp_pe_handlers.h"
#include "fmp_epayview_dialog.h"
#include "fmp_network_i.h"
#include "fmp_database.h"
#include "fmp_epay_checkmodel.h"

#include <QSslCertificate>
#include <QJsonParseError>
#include <QJsonDocument>
#include <QJsonValue>
#include <QJsonObject>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QtConcurrent>
#include <QTimer>
#include <QFuture>

#include <QSqlQuery>

#include <fmp_home_i.h>


unsigned int FMPePayPrivate::s_ClientReqCount = 10000;
#define ERROR_ORDER_STATUS        102         //异常订单状态码

FMPePayPrivate::FMPePayPrivate(FMPePay *parent)
    : QObject(parent),
      q_ptr(parent),
      _payDialog(nullptr),
      _setting(nullptr),
      _network(nullptr),
      _model(nullptr),
      _db(nullptr),
      _watcher(nullptr),
      _reverse_flag(false),
      _is_api(false),
      _queryThread(nullptr)
{
    Q_Q(FMPePay);
    _watcher = new QFutureWatcher<QByteArray>();
    connect( _watcher, SIGNAL( finished() ), this, SLOT( witedata() ) );

    if(_db == nullptr)
    {
        _db = new FMPDataBase(q->_databasename);

        QString sql = "create table " + q->_table + " ("
                      "fmId varchar(40), "
                      "code varchar(40), "
                      "pay_transId varchar(40), "
                      "trans_id varchar(40),"
                      "pay_id varchar(40), "
                      "pay_ebcode varchar(20), "
                      "total_amount integer, "
                      "pcoupon_amount integer, "
                      "mcoupon_amount integer, "
                      "alipay_amount integer, "
                      "invoice_amount integer, "
                      "business_date date, "
                      "isrefund boolean, "
                      "refund_date date,"
                      "statusCode integer,"
                      "operator_id varchar(40),"
                      "addtime TIMESTAMP default (datetime('now', 'localtime')),"
                      "PRIMARY KEY (trans_id)"
                      ")";
        FMP_INFO() << " creat table sql:" << sql;
        if(_db->creat(sql) == false)
        {
            FMP_WARN() << "creat table fmp_pay failed";
        }

        sql = "create table " + q->_redeem_table + "("
              "trans_id varchar(40),"
              "pay_id varchar(50),"
              "code varchar(20),"
              "pid varchar(20),"
              "total_amount integer)"; //卡券抵扣额，以分为单位
        FMP_INFO() << "create table sql: " << sql;
        if(_db->creat(sql) == false)
        {
            FMP_WARN() << "create table fmp_redeem failed";
        }
    }

    if(_queryThread==nullptr)
        _queryThread=new CQueryThread(this);

    //读取配置文件信息
    QString apppath=QCoreApplication::applicationDirPath();
    QSettings *settings = new QSettings(QString("%1/FreemudPOS.ini").arg(apppath), QSettings::IniFormat);
    q->_url = settings->value(FMP_INIKEY_EPAYURL).toString();
    q->_partner_id = settings->value(FMP_INIKEY_LOGINPARTNERID).toString();
    q->_time_out = settings->value(FMP_INIKEY_EPAYTIMEOUT).toInt();
    q->_time_out = (q->_time_out > 60 ? q->_time_out : 60);
    q->_store_id = settings->value(FMP_INIKEY_LOGINSTOREID).toString();
    q->_station_id = settings->value(FMP_INIKEY_LOGINPOSID).toString();
    q->_operator_id = settings->value(FMP_INIKEY_LOGINCASHIER).toString();
    delete settings;

//    _setting = FMP::GetService<FMPSettingsInterface>();
//    q->_url = _setting->GetString(FMP_INIKEY_EPAYURL);
//    q->_partner_id = _setting->GetString(FMP_INIKEY_LOGINPARTNERID);
//    q->_time_out = _setting->GetInt(FMP_INIKEY_EPAYTIMEOUT);
//    q->_time_out = (q->_time_out > 60 ? q->_time_out : 60);
//    q->_store_id = _setting->GetString(FMP_INIKEY_LOGINSTOREID);
//    q->_station_id = _setting->GetString(FMP_INIKEY_LOGINPOSID);
//    q->_operator_id = _setting->GetString(FMP_INIKEY_LOGINCASHIER);

    //检查异常订单
    CheckErrorOrder();
}

FMPePayPrivate::~FMPePayPrivate()
{   
    if(_watcher != nullptr){
        delete _watcher;
    }

    if(_payDialog != nullptr) {
        delete _payDialog;
        _payDialog = nullptr;
    }

    if(_model != nullptr){
        delete _model;
    }

    if(_db != nullptr){
        delete _db;
        _db = nullptr;
    }
    if(_queryThread != nullptr){
        delete _queryThread;
        _queryThread = nullptr;
    }
}

void FMPePayPrivate::Uninit()
{
    if (_is_api && _origin_response.isEmpty()) {
        QJsonParseError err;
        QByteArray json = "{\"statusCode\":106, \"msg\":\"\347\252\227\345\217\243\345\267\262\345\205\263\351\227\255\357\274\214\344\272\244\346\230\223\345\217\226\346\266\210\", \"prompt\":0}";
        QJsonDocument d = QJsonDocument::fromJson(json, &err);
        _origin_response = d.object();
        emit apiError();
//        QEventLoop evt;
//        connect(this, SIGNAL(apiFinish()), &evt, SLOT(quit()));
//        evt.exec();
    }
    else if (_is_api) {
        emit apiError();
//        QEventLoop evt;
//        connect(this, SIGNAL(apiFinish()), &evt, SLOT(quit()));
//        evt.exec();
    }

    Q_Q(FMPePay);
    q->_inited = false;
    _is_api = false;
    if(_watcher != nullptr){
        delete _watcher;
        _watcher = nullptr;
    }

    if(_payDialog != nullptr) {
        delete _payDialog;
        _payDialog = nullptr;
    }

    if(_model != nullptr){
        delete _model;
        _model = nullptr;
    }
}

void FMPePayPrivate::Init()
{
    Q_Q(FMPePay);
    q->_inited = true;

    clearorder();

    if(_network == nullptr)
    {
        _network = FMP::GetService<FMPNetworkInterface>();
    }

    if(_model == nullptr)
    {
        _model = new FMPPayCheckModel(NULL, _db->getDb());
        _model->setTable(q->_table);
        _model->setEditStrategy(QSqlTableModel::OnManualSubmit);
    }

    if(_payDialog == nullptr) {

        QVariantHash hash;



        hash[FMP_EPAY_PARTNERID] = q->_partner_id;
        hash[FMP_EPAY_TIMEOUT] = q->_time_out;
        //更新日期
        q->_businessdate=QDateTime::currentDateTime().toString("yyyy-MM-dd");
        if (!_is_api) {
            //hash[FMP_EPAY_ANIMATION] = _setting->GetBool(FMP_INIKEY_ANIMATION);
            hash[FMP_EPAY_ANIMATION] = true;
            hash[FMP_EPAY_BUSINESSDATE] = q->_businessdate;
            hash[FMP_EPAY_STOREID] = q->_store_id;
            hash[FMP_EPAY_STATIONID] = q->_station_id;
            hash[FMP_EPAY_OPERATORID] = q->_operator_id;
        }
        else if (_is_api && !_origin_request.isEmpty()) {
            hash["fm_cmd"] = _origin_request["fm_cmd"].toVariant();
            hash[FMP_EPAY_STOREID] = _origin_request[FMP_EPAY_STOREID].toVariant();
            hash[FMP_EPAY_STATIONID] = _origin_request["pos_id"].toVariant();
            hash[FMP_EPAY_OPERATORID] = _origin_request[FMP_EPAY_OPERATORID].toVariant();
            hash[FMP_EPAY_BUSINESSDATE] = _origin_request[FMP_EPAY_BUSINESSDATE].toVariant();
            QJsonObject trans = _origin_request["transactions"].toObject();
            if (_origin_request["fm_cmd"].toInt() == 10031) {
                hash["amount"] = trans["order_amount"].toVariant();
            }
            else if (_origin_request["fm_cmd"].toInt() == 10041) {
                hash["amount"] = trans["refund_amount"].toVariant();
                hash[FMP_JKEY_FM_ORDERID] = trans[FMP_JKEY_FM_ORDERID].toVariant();

                if (trans.contains("refund_amount") && trans["refund_amount"].toInt() != 0
                    && trans.contains(FMP_JKEY_FM_ORDERID) && !trans[FMP_JKEY_FM_ORDERID].toString().isEmpty()
                        || trans.contains(FMP_JKEY_PAY_TRANSID) && !trans[FMP_JKEY_PAY_TRANSID].toString().isEmpty()) {
                    ControlRefundJson(trans);
                    return;
                }
            }
            QJsonArray pos_products = trans["products"].toArray();
            QJsonArray products;
            int i = 1;
            foreach (QJsonValue v, pos_products) {
                QJsonObject o = v.toObject();
                o["seq"] = i++;
                products.append(o);
            }
            hash["products"] = products;
        }
        FMP_INFO() << "++++++++++++++++++request JOSN" <<hash;
        _payDialog = new FMPPayDialog(this, hash);
    }
    _payDialog->show();
}

void FMPePayPrivate::clearorder()
{
    Q_Q(FMPePay);

    QtConcurrent::run( [q, this]()
    {
        FMPDataBase db(q->_databasename, QString("fmp_pay_clean")) ;

        QDateTime date = QDateTime::currentDateTime();
        QDateTime tmpdate = date.addDays(-(q->_ordershelflife));
        QString deletedate = tmpdate.toString("yyyy-MM-dd");

        FMP_INFO() << "deletedate" <<deletedate;

        db.dlt(q->_table, QString(QString(SQL_KEY_BUSSINEDATE) + "<datetime('%1')").arg(deletedate));

    });
}

QSqlTableModel *FMPePayPrivate::model() const
{
    return _model;
}

void FMPePayPrivate::DockPayRequest(const QByteArray &json)
{
    _is_api = true;
    _origin_request = QJsonDocument::fromJson(json).object();
    _pos_trans_id=_origin_request[FMP_EPAY_TRANSID].toString();
    FMP_INFO() << "pay request json:" <<json;
}

#define FMP_EPAY_STATUSCODE         "statusCode"
#define FMP_EPAY_ERRORMSG           "msg"
#define FMP_EPAY_FMID               "fm_id"
#define FMP_EPAY_PAYED_AMOUNT       "paid_total_amount"
#define FMP_EPAY_INVOICE_AMOUNT     "invoice_amount"
#define FMP_EPAY_DISCOUNT_AMOUNT    "discount_amount"
#define FMP_EPAY_PAY_IDS            "pay_ids"
#define FMP_EPAY_PAY_ID             "pay_id"
#define FMP_EPAY_PAY_DESC           "pay_str"


QByteArray FMPePayPrivate::DockPayRespond()
{
    QEventLoop evt;
    connect(this, SIGNAL(finished(QJsonObject)), &evt, SLOT(quit()));
    connect(this, SIGNAL(apiError()), &evt, SLOT(quit()));
    evt.exec();

    _docked_response = QJsonObject();
    _docked_response[FMP_EPAY_STATUSCODE] = _origin_response[FMP_EPAY_STATUSCODE];
    _docked_response[FMP_EPAY_ERRORMSG] = _origin_response.contains("msg") ? _origin_response[FMP_EPAY_ERRORMSG] : QJsonValue("");
    _docked_response[FMP_EPAY_TRANSID] = _origin_request.contains(FMP_EPAY_TRANSID)?_origin_request[FMP_EPAY_TRANSID] : QJsonValue("");
    _docked_response["prompt"] = 0;
    if (_origin_response["statusCode"].toInt() == 100) {
        _docked_response["fm_transId"] = _origin_response["pay_transId"];
        _docked_response[FMP_EPAY_FMID] = _origin_response["fmId"];
        _docked_response[FMP_EPAY_PAYED_AMOUNT] = _origin_response["total_amount"];
        _docked_response[FMP_EPAY_INVOICE_AMOUNT] = _origin_response[FMP_EPAY_INVOICE_AMOUNT];
        _docked_response[FMP_EPAY_DISCOUNT_AMOUNT] = _origin_response["mcoupon_amount"].toInt();
        QJsonArray pay_channels;
        QJsonObject pay_ch;
        pay_ch[FMP_EPAY_PAY_ID] = _origin_response["pay_ebcode"];
        pay_ch[FMP_EPAY_PAY_DESC] = _origin_response["pay_id"];
        pay_ch["code"] = _origin_response["code"];
        pay_ch["pay_amount"] = _origin_response["total_amount"];
        pay_ch["pay_account"] = _origin_response.contains("pay_acount")?_origin_response["pay_acount"] : QJsonValue("");
        pay_ch["platform_discount"] = _origin_response["mcoupon_amount"];
        //pay_ch["merchant_discount"] = _origin_response["pcoupon_amount"];
        pay_ch["merchant_discount"] = _origin_response.contains("pcoupon_amount")?_origin_response["pcoupon_amount"] : QJsonValue(0);

        if (_origin_response.contains("ext")) {
            pay_ch["ext"] = _origin_response["ext"];
        }
        pay_channels.append(pay_ch);

        _docked_response[FMP_EPAY_PAY_IDS] = pay_channels;
    }

    _origin_response = QJsonObject();
    QByteArray json = QJsonDocument(_docked_response).toJson();

    return json;
}

void FMPePayPrivate::DockRefundRequest(const QByteArray &json)
{
    _is_api = true;
    _origin_request = QJsonDocument::fromJson(json).object();
    FMP_INFO() << "DockRefundRequest : " << _docked_request;
}

QByteArray FMPePayPrivate::DockRefundRespond()
{
    QEventLoop evt;
    connect(this, SIGNAL(finished(QJsonObject)), &evt, SLOT(quit()));
    connect(this, SIGNAL(apiError()), &evt, SLOT(quit()));
    evt.exec();

    QByteArray json;
    //!
    _docked_response = QJsonObject();
    _docked_response["statusCode"] = _origin_response["statusCode"];
    if (_origin_response.contains("msg")) {
        _docked_response["msg"] = _origin_response["msg"];
    }
    if (_origin_response.contains("ext")) {
        _docked_response["ext"] = _origin_response["ext"];
    }
    _origin_response = QJsonObject();
    json = QJsonDocument(_docked_response).toJson();

    emit apiFinish();
    return json;
}

void FMPePayPrivate::ControlPayJson(QString sum, QString code)
{
    Q_Q(FMPePay);

    if(_queryThread->isRunning()){
        emit error(QString::fromLocal8Bit("网络异常，请稍后重试！"));
        return ;
    }

    _origin_response = QJsonObject();
    QtConcurrent::run( [q, sum, code, this ]()
    {
        if(_reverse_flag)
        {
            emit error(QString::fromLocal8Bit("网络连接异常(冲正...)"));
            return ;
        }

        if(!GetPayJson(sum, code))
        {
            emit error(QString::fromLocal8Bit("获取门店信息失败"));
            return ;
        }

        FMP_INFO() << "pay json : " << _docked_request;

        QString errors;
        QJsonObject outjson;

        if(!HttpPost(outjson, _docked_request ,errors, q->_time_out))
        {
            emit error(errors);
            _origin_response = outjson;
            writeOrderToSqlite();
            if(outjson[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt() == ERROR_ORDER_STATUS){
                _queryThread->setReqJson(_docked_request);
                _queryThread->start();
            }
        }
        else
        {
            outjson.insert(SQL_KEY_ISREFUND, false);
            outjson.insert(SQL_KEY_CODE, code);
            outjson.insert(SQL_KEY_BUSSINEDATE, q->_businessdate);
            outjson.insert(SQL_KEY_TRANSID, _docked_request[SQL_KEY_TRANSID].toString());

            _origin_response = outjson;
            FMP_INFO() << "result json : " << outjson;
            writeOrderToSqlite();
//            if (_is_api) {
//                _origin_response = outjson;
//                _db->insert(q->_table, outjson.toVariantHash());
//            }
//            else
//                _db->update(q->_table, outjson.toVariantHash(),QString("trans_id = '%1'").arg(_docked_request[SQL_KEY_TRANSID].toString()));
            emit finished(outjson);
        }
    });
}

//void FMPePayPrivate::ControlPayJson(QString sum, QString code)
//{
//    Q_Q(FMPePay);

//    QByteArray data = "{ \"reqtype\" : 0}";

//    HttpPost(q->_url, data, _watcher);

//    qDebug() << "***********************" <<_watcher->future().result();
//}

void FMPePayPrivate::witedata()
{
    qDebug() << _watcher->future().result();
}

void FMPePayPrivate::HttpPost(const QString &url, const QByteArray &data, QFutureWatcher<QByteArray> *watcher, int timeout)
{
    QFuture<QByteArray> furture = QtConcurrent::run( [url, data, timeout, this]()->QByteArray
    {
        QEventLoop loop;
        QTimer timer;

        QNetworkAccessManager manger;
        QNetworkRequest request(url);

        request.setRawHeader("Content-Type","text/json");

        QNetworkReply* reply = manger.post(request, data);

        connect(&timer, &QTimer::timeout, &loop, &QEventLoop::quit);
        connect(&manger, &QNetworkAccessManager::finished, &loop, &QEventLoop::quit);
        connect(reply, static_cast<void (QNetworkReply::*)(QNetworkReply::NetworkError)>(&QNetworkReply::error), &loop, &QEventLoop::quit);
        timer.start(timeout*1000);

        loop.exec();

        reply->deleteLater();

        if(reply->error() != QNetworkReply::NoError)
        {
            return QByteArray();
        }

        return reply->readAll();
    });
    watcher->setFuture(furture);
}

bool FMPePayPrivate::HttpPost(QJsonObject& outjson, QJsonObject json, QString &error, int timeout)
{
    Q_Q(FMPePay);
    QEventLoop loop;
    QTimer timer;

    QSslConfiguration config;
    QList<QSslCertificate> certs = QSslCertificate::fromPath(qApp->applicationDirPath() + "/client01.pem");
    config.setPeerVerifyMode(QSslSocket::VerifyNone);
    config.setProtocol(QSsl::SslV3);
    config.setCaCertificates(certs);

    QNetworkAccessManager manger;
    QNetworkRequest request(q->_url);

    request.setSslConfiguration(config);
    request.setRawHeader("Content-Type","text/json");

    QNetworkReply* reply = manger.post(request, QJsonDocument(json).toJson());

    connect(&timer, &QTimer::timeout, &loop, &QEventLoop::quit);
    connect(&manger, &QNetworkAccessManager::finished, &loop, &QEventLoop::quit);
    connect(reply, static_cast<void (QNetworkReply::*)(QNetworkReply::NetworkError)>(&QNetworkReply::error), &loop, &QEventLoop::quit);
    timer.start(timeout*1000);

    loop.exec();

    reply->deleteLater();

    if(reply->error() != QNetworkReply::NoError)
    {
        error = QString::fromLocal8Bit("网络异常，支付失败！");
        outjson[FMP_RPAY_PAY_RETURN_STATUSCODE]=ERROR_ORDER_STATUS;
        outjson[FMP_RPAY_PAY_RETURN_MSG]=error;
        return false;
    }

    if(!CheckReturnJson(reply->readAll(), outjson))
    {
        error = QString::fromLocal8Bit("网络异常，支付失败！");
        outjson[FMP_RPAY_PAY_RETURN_STATUSCODE]=ERROR_ORDER_STATUS;
        outjson[FMP_RPAY_PAY_RETURN_MSG]=error;
        return false;
    }

    FMP_INFO() << outjson;

    if(outjson[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt() == 100)
       return true;

    if(outjson.contains(FMP_RPAY_PAY_RETURN_MSG))
        error = outjson[FMP_RPAY_PAY_RETURN_MSG].toString() + QString("[%1]").arg(outjson[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt());
    else
        error = QString::fromLocal8Bit("未定义错误") + QString("[%1]").arg(outjson[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt());
    if(outjson[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt() == 204)   //超时
        outjson[FMP_RPAY_PAY_RETURN_STATUSCODE]=ERROR_ORDER_STATUS;
    else
        outjson[FMP_RPAY_PAY_RETURN_STATUSCODE]=105;
    return false;

}

bool FMPePayPrivate::CheckReturnJson(QByteArray data, QJsonObject &returnjson)
{
    QJsonParseError json_error;
    QJsonDocument doc = QJsonDocument::fromJson( data, &json_error);

    if(json_error.error != QJsonParseError::NoError || !doc.isObject())
    {
        return false;
    }
    returnjson = doc.object();

    FMP_INFO() << "get return json : " << returnjson;

    return true;
}

void FMPePayPrivate::SetBasicInfo(QVariantHash hash)
{
    if(_payDialog != NULL)
        _payDialog->setBasicInfo(hash);
}

void FMPePayPrivate::GetCheckMode(QString sum)
{
    QString sql = QString("fmId like '%%%1%%' or pay_transId like '%%%1%%'").arg(sum);
    _model->setFilter(sql);
    _model->select();
}

void FMPePayPrivate::GetMode()
{
    _model->setFilter(QString(""));
    _model->select();
}

void FMPePayPrivate::GetApiReqMode(int type)
{
    Q_Q(FMPePay);
    QString sql;
    //QString date=q->_businessdate;
    switch (type) {
    case ORDER_NEED_PAY:
        sql=QString("addtime like '%1%%' and (statusCode=105 or statusCode=106)").arg(q->_businessdate);
        break;
    case ORDER_HAS_ERROR:
        sql=QString("statusCode=102 and addtime like '%1%%'").arg(q->_businessdate);
        break;
    case ORDER_PAY_SUCCESS:
        sql=QString("statusCode=100 and addtime like '%1%%'").arg(q->_businessdate);
        break;
    case ORDER_DONE:
        sql=QString("statusCode=108 and addtime like '%1%%'").arg(q->_businessdate);
        break;
    default:
        break;
    }
    _model->setFilter(sql);
    _model->select();
}

void FMPePayPrivate::ControlRefundJson(const QJsonObject &trans)
{
    Q_Q(FMPePay);

    _origin_response = QJsonObject();
    FMP_INFO() << "RefundJson trans: " << trans;
    QtConcurrent::run( [q, trans, this ]()
    {
        QStringList keylist;
        QSqlQuery query;
        keylist.append(SQL_KEY_EBCODE);
        keylist.append(SQL_KEY_TRANSID);
        bool dock_success = false;
        if(_reverse_flag) {
            emit error(QString::fromLocal8Bit("网络连接异常(冲正...)"));
            return ;
        }

        QString payTransId = trans[FMP_JKEY_PAY_TRANSID].toString();
        QString fmTransId = trans[FMP_JKEY_FM_ORDERID].toString();
        QJsonObject t = trans;
        if (!payTransId.isEmpty()) {
            if(!_db->find(q->_table, query, keylist, QString(QString(SQL_KEY_PAYTRANSID) + " = '%1'").arg(payTransId)) || !query.next()) {
                emit error(QString::fromLocal8Bit("交易记录不存在"));
                return;
            }
            else {
                t[FMP_JKEY_PAY_EBCODE] = query.value(0).toString();
                t[FMP_JKEY_POS_TRANSID] = query.value(1).toString();
            }
        }

        if(!GetRefundJson(t)) {
            emit error(QString::fromLocal8Bit("获取门店信息失败"));
            return;
        }

        FMP_INFO() << "refund json : " << _docked_request;

        QString errors;
        QJsonObject outjson;

        if(!HttpPost(outjson, _docked_request ,errors, q->_time_out)) {
            emit error(errors);
            _origin_response = outjson;
        }
        else {
            dock_success = true;
            QVariantHash hash;
            QSqlQuery tmpquery;
            QStringList tmpkeylist;
            tmpkeylist << SQL_KEY_PAYTRANSID << SQL_KEY_FMID << SQL_KEY_PAYID;

            hash.insert(SQL_KEY_ISREFUND, true);
            hash.insert(SQL_KEY_REFUND_DATE, QDateTime::currentDateTime().toString("yyyy-MM-dd"));


            if(!_db->update(q->_table, hash, QString("pay_transId = '%1' or fmId = '%2'").arg(payTransId).arg(fmTransId)))
            {
                FMP_ERROR() << "refund data update failed";
            }

            if(!_db->find(q->_table, tmpquery, tmpkeylist,  QString("pay_transId = '%1' or fmId = '%2'").arg(payTransId).arg(fmTransId)) || !tmpquery.next())
            {
                FMP_ERROR() << "refund data update failed";

                outjson.insert(SQL_KEY_PAYTRANSID, payTransId);
                outjson.insert(SQL_KEY_FMID, fmTransId);
                outjson.insert(SQL_KEY_PAYID, QString::fromLocal8Bit("未知"));
                outjson.insert(SQL_KEY_TOTALAMOUNT, trans[FMP_JKEY_REFUND_AMOUNT]);
            }
            else
            {
                outjson.insert(SQL_KEY_PAYTRANSID, tmpquery.value(0).toString());
                outjson.insert(SQL_KEY_FMID, tmpquery.value(1).toString());
                outjson.insert(SQL_KEY_PAYID, tmpquery.value(2).toString());
                outjson.insert(SQL_KEY_TOTALAMOUNT, trans[FMP_JKEY_REFUND_AMOUNT]);
            }

            FMP_INFO() << "refund success view json : " << outjson;

            _model->setFilter(QString(""));
            _model->select();

            if (_is_api) {
                _origin_response = outjson;
            }
            emit finished(outjson);
        }

        if (!dock_success) {
            emit apiError();
        }
    });
}

void FMPePayPrivate::ControlReverseJson()
{
    _reverse_flag = true;

    GetReverseJson();

    qDebug() << "using network send Reverse data to server";
    qDebug() << _docked_request;
}

bool FMPePayPrivate::GetPayJson(const QString& sum, const QString& code)
{
    Q_Q(FMPePay);

    _docked_request = QJsonObject::fromVariantMap( QVariantMap() );

    QJsonObject transaction;
    QJsonArray transactionarry;


    /**
     * settings get store infomation(or return false)
     **/
    qDebug() << "get store infomation from settings";

    int ver = 2;

    transaction.insert(FMP_EPAY_TRANSTRACTION_CODE,  code);
    if (_is_api) {
        QJsonObject origin_trans = _origin_request["transactions"].toObject();
        transaction[FMP_EPAY_TRANSTRACTION_AMOUNT] = origin_trans["order_amount"];
        QJsonArray pos_products = _origin_request["products"].toArray();
        QJsonArray products;
        int i = 1;
        foreach (QJsonValue v, pos_products) {
            QJsonObject o = v.toObject();
            o["seq"] = i++;
            products.append(o);
        }
        transaction["products"] = products;
        _docked_request[FMP_EPAY_STOREID] = q->_store_id;
        _docked_request[FMP_EPAY_STATIONID] = q->_station_id;
        _docked_request[FMP_EPAY_OPERATORID] = _origin_request[FMP_EPAY_OPERATORID];
        _docked_request[FMP_EPAY_BUSINESSDATE] = _origin_request[FMP_EPAY_BUSINESSDATE];
    }
    else {
        transaction.insert(FMP_EPAY_TRANSTRACTION_AMOUNT, (int)((sum.toDouble() + 0.005) * 100));
        _docked_request.insert( FMP_EPAY_BUSINESSDATE, q->_businessdate);
        _docked_request.insert( FMP_EPAY_STOREID, q->_store_id);
        _docked_request.insert( FMP_EPAY_STATIONID, q->_station_id);
        _docked_request.insert( FMP_EPAY_OPERATORID, q->_operator_id);
    }

    transactionarry.append( transaction);

    _docked_request.insert( FMP_EPAY_VER, ver);
    _docked_request.insert( FMP_EPAY_REQUESTTYPE, 72);

    _docked_request.insert( FMP_EPAY_PARTNERID, q->_partner_id.toInt());
    _docked_request.insert( FMP_EPAY_CLIENTREQCOUNT, (int)((++s_ClientReqCount)%=10000000));

    _docked_request.insert( FMP_EPAY_TRANSID, _origin_request.contains(FMP_EPAY_TRANSID)?_origin_request[FMP_EPAY_TRANSID].toString() : _pos_trans_id);

    _docked_request.insert( FMP_EPAY_TRANSTRACTION, transactionarry);

    return true;
}

bool FMPePayPrivate::GetRefundJson(const QJsonObject &trans)
{
    Q_Q(FMPePay);
    _docked_request = QJsonObject::fromVariantMap( QVariantMap() );

    QJsonObject transaction;
    QJsonArray transactionarry;


    /**
     * settings get store infomation(or return false)
     **/
    qDebug() << "get store infomation from settings";

    int ver = 2;

    FMP_INFO() << "RefundJson trans: " << trans;
    if(trans.contains(FMP_JKEY_FM_ORDERID) && !trans[FMP_JKEY_FM_ORDERID].toString().isEmpty())
    {
        transaction.insert(FMP_EPAY_REFUND_TRANSTRACTION_FMID,  trans[FMP_JKEY_FM_ORDERID]);
        if (_is_api) {
            QJsonObject orig_trans = _origin_request["transactions"].toObject();
            transaction.insert(FMP_EPAY_REFUND_TRANSTRACTION_AMOUNT, (int)orig_trans["refund_amount"].toDouble());
        }
        else {
            transaction.insert(FMP_EPAY_REFUND_TRANSTRACTION_AMOUNT, (int)trans[FMP_JKEY_REFUND_AMOUNT].toDouble());
        }
        _docked_request.insert(FMP_EPAY_TRANSID, QDateTime::currentDateTime().toString("yyyyMMddhhmmsszzz"));
    }
    else {
        transaction.insert(FMP_EPAY_REFUND_TRANSTRACTION_EBCODE,  trans[FMP_JKEY_PAY_EBCODE]);
        transaction.insert(FMP_EPAY_REFUND_TRANSTRACTION_TRANSID,  trans[FMP_JKEY_PAY_TRANSID]);
        transaction.insert(FMP_EPAY_REFUND_TRANSTRACTION_AMOUNT, (int)trans[FMP_JKEY_REFUND_AMOUNT].toDouble());
        _docked_request.insert(FMP_EPAY_TRANSID, trans[FMP_JKEY_POS_TRANSID]);
    }

    transactionarry.append( transaction);

    _docked_request.insert( FMP_EPAY_VER, ver);
    _docked_request.insert( FMP_EPAY_REQUESTTYPE, 62);
    _docked_request.insert( FMP_EPAY_PARTNERID, q->_partner_id.toInt());
    _docked_request.insert( FMP_EPAY_CLIENTREQCOUNT, (int)((++s_ClientReqCount)%=10000000));

    if (_is_api) {
        _docked_request[FMP_EPAY_STOREID] = _origin_request[FMP_EPAY_STOREID];
        _docked_request[FMP_EPAY_STATIONID] = _origin_request["pos_id"];
        _docked_request[FMP_EPAY_OPERATORID] = _origin_request[FMP_EPAY_OPERATORID];
        _docked_request[FMP_EPAY_BUSINESSDATE] = _origin_request[FMP_EPAY_BUSINESSDATE];
        _docked_request.insert( FMP_EPAY_TRANSTRACTION, transactionarry);
    }
    else {
        _docked_request.insert( FMP_EPAY_STOREID, q->_store_id);
        _docked_request.insert( FMP_EPAY_STATIONID, q->_station_id);
        _docked_request.insert( FMP_EPAY_OPERATORID, q->_operator_id);
        _docked_request.insert( FMP_EPAY_TRANSTRACTION, transactionarry);
    }
    FMP_INFO() << "Refund Json:" << _docked_request;
    return true;
}

void FMPePayPrivate::GetReverseJson()
{
    _docked_request[FMP_EPAY_REQUESTTYPE] = 3;
}
void FMPePayPrivate::ClosePayWindow(int type)
{
   // Q_Q(FMPePay);

    //返回支付结果json;发送给socket
    //状态码：107 现金支付
    //状态码：106 程序强制关闭
    if(type==DIALOG_CASH_CLOSE){
        _origin_response[FMP_EPAY_STATUSCODE]=107;
        _origin_response[FMP_EPAY_ERRORMSG]=QString::fromLocal8Bit("支付失败，默认现金收银（已结算）");
    }
    else if(type==DIALOG_FORCE_CLOSE){
        _origin_response[FMP_EPAY_STATUSCODE]=106;
        _origin_response[FMP_EPAY_ERRORMSG]=QString::fromLocal8Bit("程序运行时被关闭");
    }
    writeOrderToSqlite();
}

void FMPePayPrivate::writeOrderToSqlite()
{
    Q_Q(FMPePay);
    QJsonObject _dbWrite=_origin_response;

    if(!_is_api){
        // 由于从服务端接收到的字段名称和协议中定义的不同，这里需要转换
        QJsonObject response;
        response["fm_transId"] = _origin_response["pay_transId"];
        response[FMP_EPAY_FMID] = _origin_response["fmId"];
        response[FMP_EPAY_PAYED_AMOUNT] = _origin_response["total_amount"];
        response[FMP_EPAY_INVOICE_AMOUNT] = _origin_response.contains(FMP_EPAY_INVOICE_AMOUNT)?_origin_response[FMP_EPAY_INVOICE_AMOUNT] : QJsonValue(0);
        response[FMP_EPAY_DISCOUNT_AMOUNT] = _origin_response["mcoupon_amount"].toInt();
        response[FMP_EPAY_TRANSID]=_pos_trans_id;
        response[FMP_RPAY_PAY_RETURN_STATUSCODE]=_dbWrite[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt();
        response[FMP_EPAY_ERRORMSG] = _origin_response.contains("msg") ? _origin_response[FMP_EPAY_ERRORMSG] : QJsonValue("");
        response["prompt"] = 0;
        QJsonArray pay_channels;
        QJsonObject pay_ch;
        pay_ch[FMP_EPAY_PAY_ID] = _origin_response["pay_ebcode"];
        pay_ch[FMP_EPAY_PAY_DESC] = _origin_response["pay_id"];
        pay_ch["code"] = _origin_response["code"];
        pay_ch["pay_amount"] = _origin_response["total_amount"];
        pay_ch["pay_account"] = _origin_response.contains("pay_acount")?_origin_response["pay_acount"] : QJsonValue("");
        pay_ch["platform_discount"] = _origin_response["mcoupon_amount"];
        pay_ch["merchant_discount"] = _origin_response.contains("pcoupon_amount")?_origin_response["pcoupon_amount"] : QJsonValue(0);

        if (_origin_response.contains("ext")) {
            pay_ch["ext"] = _origin_response["ext"];
        }
        pay_channels.append(pay_ch);
        response[FMP_EPAY_PAY_IDS] = pay_channels;

        QByteArray json = QJsonDocument(response).toJson();

        if(!SendToMonitor(json)){
            emit error(QString::fromLocal8Bit("数据发送监控程序失败！"));
            FMP_ERROR() << "数据发送监控程序失败：" << json;
        }
    }
    _dbWrite[FMP_EPAY_TRANSID]=_pos_trans_id;
    if(_dbWrite[FMP_RPAY_PAY_RETURN_STATUSCODE].toInt() == 100){
        if (!_db->isRecordExist(q->_table,_pos_trans_id)){
            _dbWrite[FMP_EPAY_OPERATORID] = _origin_request[FMP_EPAY_OPERATORID];
            _db->insert(q->_table, _dbWrite.toVariantHash());
        }
        else
            _db->update(q->_table, _dbWrite.toVariantHash(),QString("trans_id = '%1'").arg(_dbWrite[SQL_KEY_TRANSID].toString()));
    }
    else{      
        if(!_db->isRecordExist(q->_table,_pos_trans_id)){
            QJsonObject origin_trans = _origin_request["transactions"].toObject();
            _dbWrite["total_amount"]=origin_trans["order_amount"];
            _dbWrite["business_date"]=_origin_request["business_date"];
            //_dbWrite[FMP_EPAY_TRANSID] = _origin_request[FMP_EPAY_TRANSID];
            _dbWrite[FMP_EPAY_OPERATORID] = _origin_request[FMP_EPAY_OPERATORID];
            _db->insert(q->_table, _dbWrite.toVariantHash());
        }
        else{     
            _db->update(q->_table, _dbWrite.toVariantHash(),QString("trans_id = '%1'").arg(_dbWrite[FMP_EPAY_TRANSID].toString()));
        }
    }

    FMP_INFO() << "writeOrderToSqlite json : " << _dbWrite;
}

void FMPePayPrivate::WriteRedeemToSqlite(QJsonObject redeemInfo)
{
    Q_Q(FMPePay);
    redeemInfo[FMP_EPAY_TRANSID] = _pos_trans_id;
    FMP_DEBUG() << redeemInfo;
    if(!_db->insert(q->_redeem_table, redeemInfo.toVariantHash()))
    {
        FMP_ERROR() << "write redeem info to fmp_redeem table failed";
    }
}

FMPDataBase* FMPePayPrivate::GetDB()const
{
    return _db;
}

void FMPePayPrivate::updateSqlite(const QJsonObject &json)
{
    Q_Q(FMPePay);
    QByteArray byteJson = QJsonDocument(json).toJson();
    if(!_is_api){
        if(!SendToMonitor(byteJson)){
            emit error(QString::fromLocal8Bit("数据发送监控程序失败！"));
            FMP_ERROR() << "数据发送监控程序失败：" << byteJson;
        }
    }
    _db->update(q->_table, json.toVariantHash(),QString("trans_id = '%1'").arg(json[FMP_EPAY_TRANSID].toString()));

}

void FMPePayPrivate::setPosTransId(QString orderId)
{
    _pos_trans_id=orderId;
    _origin_response.empty();
}

bool FMPePayPrivate::SendToMonitor(const QByteArray &data)
{
    FMP_INFO() << "SendToMonitor:" << data;
    bool result=true;
    QTcpSocket client;

    int jsonLength=data.length();
    char* m_pFmPackage = new char[jsonLength + sizeof(FMSOCKEHEADER)];
    FMSOCKEHEADER header = { 0, 0, 0 };
    header.flag = 0x4d46;
    header.len = jsonLength;
    header.ver = 0x1;

    memcpy(m_pFmPackage, &header, sizeof(FMSOCKEHEADER));
    memcpy(m_pFmPackage+sizeof(FMSOCKEHEADER), data, jsonLength);


    int toSendLength = jsonLength + sizeof(FMSOCKEHEADER);
    int curSendLength = 0;

    client.connectToHost("127.0.0.1", 23771);
    if (client.waitForConnected()){
        while(curSendLength < toSendLength)
        {
            int rlt= client.write(m_pFmPackage+curSendLength,toSendLength - curSendLength);
            client.waitForBytesWritten();
            if(rlt==-1){
                FMP_ERROR() << "发送数据失败：" << data;
                result=false;
                break;
            }
            curSendLength += rlt;
        }
    }
    else{
        FMP_ERROR() << "socket 连接失败！";
        result=false;
    }
    client.close();
    delete[] m_pFmPackage;
    return result;
}

void FMPePayPrivate::CheckErrorOrder()
{
    Q_Q(FMPePay);
    QStringList keylist;
    QSqlQuery query;
    keylist <<FMP_EPAY_TRANSID  << FMP_EPAY_OPERATORID;
    if(!_db->find(q->_table, query, keylist, QString("statusCode = %1 and business_date= '%2'").arg(ERROR_ORDER_STATUS).arg(q->_businessdate)) || !query.next()) {
        return;
    }
    else {
        QJsonObject queryJson;
        queryJson[FMP_EPAY_TRANSID] = query.value(0).toString();
        queryJson[FMP_EPAY_OPERATORID] = query.value(1).toString();
        queryJson[FMP_EPAY_STOREID]=q->_store_id;
        queryJson[FMP_EPAY_STATIONID]=q->_station_id;
        queryJson[FMP_EPAY_PARTNERID]=q->_partner_id;
        queryJson[FMP_EPAY_VER]=2;

        _queryThread->setReqJson(queryJson);
        _queryThread->start();
    }
}

void FMPePayPrivate::GetDailyBillData(QVector<PayDetail> &vecPay)
{
    Q_Q(FMPePay);
    QStringList keylist;
    keylist<<"count(*) as num"<<"sum(total_amount) as total";
    QSqlQuery query;
    PayDetail pay;
    //支付宝查询
    if(!_db->find(q->_table, query, keylist, QString("statusCode = %1 and addtime like '%2%%' and pay_ebcode='%3'").arg(100).arg(q->_businessdate).arg("10001")) || !query.next()) {
        FMP_ERROR() << "查询支付宝数据错误！";
    }
    else {
        pay.name=QString::fromLocal8Bit("支付宝");
        pay.num = query.value(0).toString();
        pay.total = QString::number(query.value(1).toDouble()/100);
        pay.status=1;
        vecPay.push_back(pay);
    }
    //微信查询
    if(!_db->find(q->_table, query, keylist, QString("statusCode = %1 and addtime like '%2%%' and pay_ebcode='%3'").arg(100).arg(q->_businessdate).arg("10004")) || !query.next()) {
        FMP_ERROR() << "查询微信数据错误！";
    }
    else {
        pay.name=QString::fromLocal8Bit("微信");
        pay.num = query.value(0).toString();
        pay.total = QString::number(query.value(1).toDouble()/100);
        pay.status=1;
        vecPay.push_back(pay);
    }
    //全部查询
    if(!_db->find(q->_table, query, keylist, QString("statusCode = %1 and addtime like '%2%%'").arg(100).arg(q->_businessdate)) || !query.next()) {
        FMP_ERROR() << "查询全部支付数据错误！";
    }
    else {
        pay.name=QString::fromLocal8Bit("全部");
        pay.num = query.value(0).toString();
        pay.total = QString::number(query.value(1).toDouble()/100);
        pay.status=1;
        vecPay.push_back(pay);
    }
//QString::number(json[FMP_RPAY_PAY_RETURN_TOTAL].toDouble()/100)
    //未成功支付
    FMP_INFO() << "未成功支付查询！";
    if(!_db->find(q->_table, query, keylist, QString("statusCode != %1 and addtime like '%2%%'").arg(100).arg(q->_businessdate)) || !query.next()) {
        FMP_ERROR() << "查询未成功支付数据错误！";
    }
    else {
        pay.name=QString::fromLocal8Bit("");
        pay.num = query.value(0).toString();
        pay.total = QString::number(query.value(1).toDouble()/100);
        pay.status=0;
        vecPay.push_back(pay);
    }
}
