﻿#ifndef FMP_LOGGER_I_H
#define FMP_LOGGER_I_H

#include <fmp_base_i.h>
#include "fmp_logstream.h"
#include <ctkServiceReference.h>
#include <ctkPluginContext.h>

/**
 * 非业务插件接口，必须继承自 BaseInterface
 * @brief The FMPLoggerInterface class
 */
class FMPLoggerInterface : public FMPBaseInterface
{
public:
    enum { LOG_ERROR = 1, LOG_WARNING, LOG_INFO, LOG_DEBUG };

    /**
     * @brief Log
     * 函数方式记录日志
     * @param level
     * @param file
     * @param func
     * @param line
     */
    virtual void            Log(short level, const QString &msg, const char* file, const char* func, int line) = 0;

    /**
     * @brief GetLogLevel
     * 获取当前日志记录级别
     * @return
     */
    virtual short           GetLogLevel() const = 0;

    /**
     * @brief SetLogLevel
     * 设置当前日志级别
     * @param level
     */
    virtual void            SetLogLevel(short level) = 0;

    /**
     * @brief GetMaxFilesize
     * 获取日志文件最大容量(MB)
     * @return 0 默认大小   n MB
     */
    virtual uint            GetMaxFilesize() const = 0;

    /**
     * @brief SetMaxFilesize
     * 设置日志文件最大容量(MB)
     */
    virtual void            SetMaxFilesize(uint) = 0;

    /**
     * @brief GetFilename
     * 获取当前日志文件路径
     * @return 文件路径
     */
    virtual QString         GetFilename() const = 0;

    /**
     * @brief GetPath
     * 获取日志存储位置
     * @return
     */
    virtual QString         GetPath() const = 0;

    /**
     * @brief SetPath
     * 设置日志存储位置
     * @param path
     */
    virtual void            SetPath(const QString &path) = 0;

    /**
     * @brief GetIODevice
     * 获取文件输出设备指针(Stream使用)
     * @return QIODevice*
     */
    virtual QIODevice*      GetIODevice() = 0;
};

Q_DECLARE_INTERFACE(FMPLoggerInterface, "com.fmp.logger")


//! 快速日志记录宏定义
#define FMP_LOG(lvl, log) ((log && log->GetLogLevel() >= lvl) ? \
    FMPLogStream(lvl, __FILE__, __FUNCTION__, __LINE__, log->GetIODevice()) : \
    FMPNullLogStream(lvl, __FILE__, __FUNCTION__, __LINE__))

#define FMP_DEBUG(log)  FMP_LOG(FMPLoggerInterface::LOG_DEBUG, log)
#define FMP_INFO(log)   FMP_LOG(FMPLoggerInterface::LOG_INFO, log)
#define FMP_WARN(log)   FMP_LOG(FMPLoggerInterface::LOG_WARNING, log)
#define FMP_ERROR(log)  FMP_LOG(FMPLoggerInterface::LOG_ERROR, log)

#define FMP_LOG_CTX(lvl, ctx) ((ctx && ctx->getServiceReference<FMPLoggerInterface>()) ? \
    FMP_LOG(lvl, ctx->getService<FMPLoggerInterface>(ctx->getServiceReference<FMPLoggerInterface>())) : \
    FMP_LOG(lvl, ctx->getService<FMPLoggerInterface>(ctx->getServiceReference<FMPLoggerInterface>())))

#define FMP_DEBUG_CTX(ctx)  FMP_LOG_CTX(FMPLoggerInterface::LOG_DEBUG, ctx)
#define FMP_INFO_CTX(ctx)   FMP_LOG_CTX(FMPLoggerInterface::LOG_INFO, ctx)
#define FMP_WARN_CTX(ctx)   FMP_LOG_CTX(FMPLoggerInterface::LOG_WARNING, ctx)
#define FMP_ERROR_CTX(ctx)  FMP_LOG_CTX(FMPLoggerInterface::LOG_ERROR, ctx)

#endif // FMP_LOGGER_I_H
