#ifndef FMLOGGER_H
#define FMLOGGER_H

//#include <stdio.h>
#include <Windows.h>
#include <tchar.h>
#include <share.h>
#include <string>
#include <algorithm>

//////////////////////////////////////////////////////////////////////////
//
//! Autolock class
//
//////////////////////////////////////////////////////////////////////////
class FMAutoLock {
public:
    explicit FMAutoLock(CRITICAL_SECTION &cs) : _cs(cs) 
    {
        EnterCriticalSection(&_cs);
    }
    ~FMAutoLock() 
    {
        LeaveCriticalSection(&_cs);
    }

private:
    CRITICAL_SECTION &_cs;
};

//////////////////////////////////////////////////////////////////////////
//
//! Buffer class
//
//////////////////////////////////////////////////////////////////////////
class FMBuffer
{
public:
    std::string     _buffer; // Must be protected by lock
    long            _total_bytes;

    FMBuffer() : _total_bytes(0) 
    {
        InitializeCriticalSection(&_buffer_cs);
    }

    ~FMBuffer()
    {
        DeleteCriticalSection(&_buffer_cs);
    }
public:
    void Write(const void* p, size_t len)
    {
        InterlockedExchangeAdd(&_total_bytes, (long)len);

        FMAutoLock auto_lock(_buffer_cs);
        _buffer.append((const char*)p, len);
    }

private:
    CRITICAL_SECTION    _buffer_cs;
};

//////////////////////////////////////////////////////////////////////////
//
//! Logger class
//
//////////////////////////////////////////////////////////////////////////
class FMLogger {
public:
    static FMLogger *GetLogger() 
    {
        static FMLogger logger;

        return &logger;
    }

    FMLogger()
    {
        InitializeCriticalSection(&_logger_cs);
        _fp = 0;
        _written_size = 0;
        //! 4 MB
        _filesize = 1024 * 1024 * 4;

        _buffered_size = 0;
    }

    ~FMLogger() 
    {
        DeleteCriticalSection(&_logger_cs);
        if (_fp) {
            fclose(_fp);
            _fp = 0;
        }
    }

    void        FormatLog(LPCWSTR format, ...)
    {
        va_list arg_ptr;
        va_start(arg_ptr, format);
        FormatLogW(format, arg_ptr);
        va_end(arg_ptr);
    }
    void        FormatLog(LPCSTR format, ...)
    {
        va_list arg_ptr;
        va_start(arg_ptr, format);
        FormatLogA(format, arg_ptr);
        va_end(arg_ptr);
    }


    //! Log messages into file
    void        FormatLogW(LPCWSTR format, va_list arg_ptr) 
    {
        if (format) {
            va_list temp;
            temp = arg_ptr;
            int size = wcslen(format) + 1; // Count on '\0'
            LPCWSTR arg_pos = wcschr(format, _T('%'));
            while(arg_pos) {
                ++arg_pos;
                LPCWSTR arg = NULL;
                switch(*arg_pos) {
                case _T('s'):
                    arg = va_arg(temp, LPCWSTR);
                    size += wcslen(arg) + 1;
                    break;
                default:
                    va_arg(temp, UINT);
                    size += 32;
                    break;
                }
                arg_pos = wcschr(arg_pos, _T('%'));
            }
            va_end(temp);

            LPWSTR msg = new WCHAR[size];
            memset((void*)msg, 0, size * sizeof(WCHAR));
            vswprintf_s(msg, size, format, arg_ptr);

            int str_size = WideCharToMultiByte(CP_ACP, 0, msg, -1, NULL, 0, NULL, NULL);
            if (str_size > 0) {
                char *buff = new char[str_size + 1];
                memset(buff, 0, str_size + 1);
                WideCharToMultiByte(CP_ACP, 0, msg, -1, buff, str_size, NULL, NULL);

                {
                    //FMAutoLock auto_lock(_logger_cs);
                    _Log(buff);
                }

                delete[] buff;
            }

            delete []msg;
        }
    }

    void        FormatLogA(LPCSTR format, va_list arg_ptr) 
    {
        if (format) {
            va_list temp;
            temp = arg_ptr;
            int size = strlen(format) + 1; // Count on '\0'
            LPCSTR arg_pos = strchr(format, '%');
            while(arg_pos) {
                ++arg_pos;
                LPCSTR arg = NULL;
                switch(*arg_pos) {
                case 's':
                    arg = va_arg(temp, LPCSTR);
                    size += strlen(arg) + 1;
                    break;
                default:
                    va_arg(temp, UINT);
                    size += 32;
                    break;
                }
                arg_pos = strchr(arg_pos, _T('%'));
            }
            va_end(temp);

            LPSTR msg = new CHAR[size];
            memset((void*)msg, 0, size);
            vsprintf_s(msg, size, format, arg_ptr);

            _Log(msg);
            delete []msg;
        }
    }


private:
    void        _Log(LPCSTR msg)
    {
        if (0 == _fp) {
            _fp = _fsopen(_GetLogFileName(), "w+", _SH_DENYNO);
            
            if (0 != _fp) {
                _DeleteExpiredLogs();
            }
        }

        if (_written_size >= _filesize && _fp) {
            fclose(_fp);
            _written_size = 0;
            _DeleteExpiredLogs();
            _fp = _fsopen(_GetLogFileName(), "w+", _SH_DENYNO);
            if (0 == _fp) {
                printf("Failed to open file %s, error :%ld\n", _filename, GetLastError());
            }
        }

        SYSTEMTIME st;
        GetLocalTime(&st);
        CHAR timestamp[32] = { 0 };
        sprintf_s(timestamp, "%04d-%02d-%02d %02d:%02d:%02d %03d => ", st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
        if (0 == _fp) {
            printf(timestamp);
            printf(msg);
            printf("\n");
        } else {
            UINT written_size = fwrite(timestamp, 1, strlen(timestamp), _fp);
            written_size += fwrite(msg, 1, strlen(msg), _fp);
            written_size += fwrite("\n", 1, 1, _fp);
            _written_size += written_size;
            _buffered_size += written_size;

            fflush(_fp);
//            if (_buffered_size >= 2048) {
//                fflush(_fp);
//                _buffered_size = 0;
//            }
        }
    }

    LPCSTR     _GetLogFileName() 
    {
        SYSTEMTIME st;
        GetLocalTime(&st);

        CHAR path[MAX_PATH] = { 0 };
        LPSTR path_start = path;
        GetModuleFileNameA(NULL, path, MAX_PATH);
        path_start = strrchr(path_start, '\\');
        *path_start = '\0';
        LPSTR module = ++path_start;
        sprintf_s(_filename, "%s\\log\\%s_%04d%02d%02d%02d%02d_pid_%ld.log", path, module, st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, GetCurrentProcessId());

        return _filename;
    }

    VOID       _DeleteExpiredLogs()
    {
        CHAR log_path[MAX_PATH] = { 0 };
        strcpy_s(log_path, _filename);
        CHAR *str_pos = strrchr(log_path, '\\');
        if (str_pos) {
            *++str_pos = '\0';
            strcat_s(log_path, "*");

            DWORD dwStatus = 0;
            WIN32_FIND_DATAA findFileData;
            HANDLE hFind = FindFirstFileA(log_path, &findFileData);
            if (hFind != INVALID_HANDLE_VALUE)
            {
                do 
                {
                    if (strcmp(findFileData.cFileName, ".") == 0 || strcmp(findFileData.cFileName, "..") == 0) {
                        continue;
                    }
                    else {                      
                        SYSTEMTIME st_now;
                        GetSystemTime(&st_now);
                        FILETIME   ft_now; 
                        SystemTimeToFileTime(&st_now, &ft_now); 

                        ULARGE_INTEGER const wt = { findFileData.ftLastWriteTime.dwLowDateTime, 
                                                    findFileData.ftLastWriteTime.dwHighDateTime };
                        ULARGE_INTEGER const nt = { ft_now.dwLowDateTime, ft_now.dwHighDateTime }; 

                        ULARGE_INTEGER et; 
                        et.QuadPart = nt.QuadPart - wt.QuadPart; 

                        FILETIME const eft = { et.LowPart, et.HighPart }; 
                        SYSTEMTIME est; 
                        FileTimeToSystemTime(&eft, &est); 

                        // FILETIME's starting point is 1601-01-01 
                        if (est.wYear != 0) {
                            est.wYear -= 1601; 
                            est.wMonth -= 1; 
                            est.wDay -= 1; 
                            est.wDayOfWeek = 0;
                            if (est.wYear > 0 || est.wMonth > 0 || est.wDay > 3) {
                                *str_pos = '\0';
                                strcat_s(log_path, findFileData.cFileName);
                                DeleteFileA(log_path);
                            }
                        }
                    }

                    if (dwStatus != 0) {
                        break;
                    }
                } while (FindNextFileA(hFind, &findFileData));

                FindClose(hFind);
            }
        }
    }

    //! Disable copy constructor
    FMLogger(const FMLogger&) {}

private:
    CRITICAL_SECTION    _logger_cs;
    FILE               *_fp;
    CHAR                _filename[1024];
    UINT                _filesize;
    UINT                _written_size;
    UINT                _buffered_size;
};


#define FMLOG(...) FMLogger::GetLogger()->FormatLog(__VA_ARGS__)


#endif // FMLOGGER_H
