#include "fmvipforward.h"
#include <QJsonDocument>
#include <QJsonParseError>
#include <QJsonObject>
#include <QJsonArray>
#include <QCryptographicHash>
#include <QDateTime>

#include <QDebug>

FMVipForward* FMVipForward::_instance = NULL;

FMVipForward* FMVipForward::instance()
{
    if(_instance == NULL) {
        _instance = new FMVipForward();
    }
    return _instance;
}

FMVipForward::FMVipForward(QObject *parent) : QObject(parent)
{
    _urlStr = _cfg.GetTransUrlFormat();
    _sessionDataList = QString("operator_id,pos_id,business_date").split(",");
    connect(&_nam, SIGNAL(finished(QNetworkReply*)), this, SLOT(onServerFinished(QNetworkReply*)));
}

FMVipForward::~FMVipForward()
{
    if(_instance != NULL) {
        delete _instance;
        _instance = NULL;
    }
}

void FMVipForward::parseRequest(const QJsonObject &jsonObj)
{
    QString type = jsonObj["fm_cmd"].toString();
    QJsonObject fm_jsonObj;
    if(type == Type_Login) {
        login(jsonObj, fm_jsonObj);
    } else if(type == Type_Fund) {
        fund(jsonObj, fm_jsonObj);
    } else if(type == Type_Coupon) {
        coupon(jsonObj, fm_jsonObj);
    } else if(type == Type_Pay) {
        pay(jsonObj, fm_jsonObj);
    } else if(type == Type_Final) {
        final(jsonObj, fm_jsonObj);
    } else if(type == Type_Order_Refund || type == Type_Fund_Refund) {
        orderRefund(jsonObj, fm_jsonObj);
    } else if(type == Type_Order_Revoke) {
        orderRevoke(jsonObj, fm_jsonObj);
    }

    _reqType = fm_jsonObj["reqType"].toInt();

    // 固定部分
    fm_jsonObj["appId"] = AppId;
    fm_jsonObj["partnerId"] = PartnerId;
    fm_jsonObj["t"] = QString::number(QDateTime::currentMSecsSinceEpoch());

    // 可从请求中提取的部分
    FMApiPropMap *pm = _cfg.GetPropsMap();
    if (pm && pm->size() > 0) {
        FMApiPropMap::iterator iter = pm->begin();
        for(; iter != pm->end(); ++iter) {
            if(jsonObj.contains(iter->first)) {
                QJsonValue v = jsonObj[iter->first];
                fm_jsonObj[iter->second] = v;
            }
        }
    }

    // 向服务器发送请求
    request(fm_jsonObj);
}

void FMVipForward::request(const QJsonObject &reqJob)
{
    // 设置URL
    QString signStr = sign(reqJob);

    _req.setUrl(_urlStr.arg(signStr));

    QJsonDocument json(reqJob);
    QByteArray reqData = json.toJson();

    qDebug() << "向服务器发送 ===>\n" << json;
    qDebug() << "请求签名 ===>\n" << signStr;

    // 设置请求头
    _req.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    _req.setHeader(QNetworkRequest::ContentLengthHeader, reqData.length());

    _nam.post(_req, reqData);

//    clearSessionData();
}

void FMVipForward::onServerFinished(QNetworkReply *reply)
{
    QJsonObject posObj;

    if(reply->error() != QNetworkReply::NoError) {
        posObj["statusCode"] = 404;
        posObj["msg"] = reply->errorString();
    } else {
        QByteArray data = reply->readAll();
        QJsonParseError err;
        QJsonDocument jdoc = QJsonDocument::fromJson(data, &err);

        qDebug() << "服务器返回 ===>\n" << jdoc;

        if(err.error != QJsonParseError::NoError || !jdoc.isObject()) {
            posObj["statusCode"] = 1000;
            posObj["msg"] = "服务器返回数据异常";
        } else {
            QJsonObject serverObj = jdoc.object();

            // 服务器返回错误信息
            if(serverObj.contains("errcode") && serverObj.contains("errmsg")) {
                posObj["statusCode"] = serverObj["errcode"];
                posObj["msg"] = serverObj["errmsg"];
            }
            // 服务器返回正常信息
            else {
                posObj["statusCode"] = serverObj["statusCode"];
                posObj["msg"] = serverObj["msg"];
                switch (_reqType) {
                case FM_VIP_LOGIN:
                    logined(serverObj, posObj);
                    break;
                case FM_VIP_COUPON:
                    couponed(serverObj, posObj);
                    break;
                case FM_VIP_FUND:
                    funded(serverObj, posObj);
                    break;
                case FM_VIP_PAY:
                    payed(serverObj, posObj);
                    break;
                case FM_VIP_FINAL:
                    finaled(serverObj, posObj);
                    break;
                case FM_VIP_REVOKE:
                    orderRevoked(serverObj, posObj);
                    break;
                case FM_VIP_REFUND:
                    orderRefunded(serverObj, posObj);
                    break;
                default:
                    break;
                }
            }
        }
    }

    emit serverResponsed(posObj);
}


void FMVipForward::login(const QJsonObject &job, QJsonObject &fmjob)
{
    Q_UNUSED(job);
    fmjob["reqType"] = FM_VIP_LOGIN;
    fmjob["code"] = sessionData("code");
}
void FMVipForward::fund(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_FUND;

    QJsonObject transObj;
    transObj["account"] = sessionData("fm_open_id");
    transObj["amount"] = job["charge_amount"];
    transObj["cashAmount"] = job["charge_amount"];
    transObj["thirdAmount"] = 0;
    transObj["thirdPayType"] = 0;
    transObj["thirdPayTransId"] = "";

    fmjob["transaction"] = transObj;
}
void FMVipForward::coupon(const QJsonObject &job, QJsonObject &fmjob)
{
    Q_UNUSED(job);
    fmjob["reqType"] = FM_VIP_COUPON;
}
void FMVipForward::pay(const QJsonObject &job, QJsonObject &fmjob)
{
    Q_UNUSED(job);
    fmjob["reqType"] = FM_VIP_PAY;

    _sessionForward = job["forward"].toObject();

    QJsonObject transObj;
    transObj["account"] = sessionData("fm_open_id");
    transObj["codeAmount"] = sessionDataInt("codeAmount");
    transObj["isUseScore"] = sessionDataInt("isUseScore");

    QJsonArray couponsArr;
    for(auto code : _sessionCouponMap.keys())
    {
        couponsArr.append(code);
    }
    transObj["coupons"] = couponsArr;

    fmjob["transaction"] = transObj;
}
void FMVipForward::final(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_FINAL;

    QJsonObject transObj;
    QJsonObject posTransObj = job["transactions"].toObject();
    transObj["account"] = job["fm_open_id"];
    int amount = posTransObj["order_amount"].toInt();
    int paidAmount = posTransObj["paid_amount"].toInt();
    QJsonArray pay_ids = posTransObj["pay_ids"].toArray();
    fmjob["memberTransId"] = job["fm_id"];

    QJsonArray coupons = posTransObj["coupons"].toArray();
    int codeAmount=0, scoreAmount=0, cashAmount=0, thirdAmount=0;
    foreach (QJsonValue pay_v , pay_ids)
    {
        QJsonObject pay_ob = pay_v.toObject();
        QString type = pay_ob["pay_id"].toString();
        if(type == "24") {
            codeAmount = pay_ob["paid_total_amount"].toInt();
        } else if(type == "25") {
            scoreAmount = pay_ob["paid_total_amount"].toInt();
        } else if(type == "77") {
            coupons.append(pay_ob["code"].toString());
        }
    }
    transObj["coupons"] = coupons;

    cashAmount = paidAmount - codeAmount -scoreAmount - thirdAmount;
    transObj["amount"] = amount;
    transObj["payAmount"] = paidAmount;
    transObj["codeAmount"] = codeAmount;
    transObj["scoreAmount"] = scoreAmount;
    transObj["cashAmount"] = cashAmount;
    transObj["thirdAmount"] = thirdAmount;
    transObj["thirdPayType"] = 0;
    transObj["thirdPayTransId"] = "";

    QJsonArray posProdArray = posTransObj["products"].toArray();
    QJsonArray prodArray;
    foreach (QJsonValue v , posProdArray)
    {
        QJsonObject po = v.toObject();
        QJsonObject o;
        o["consumeNum"] = po["consume_num"];
        o["price"] = po["price"];
        o["productId"] = po["pid"];
        prodArray.append(o);
    }
    transObj["products"] = prodArray;
    fmjob["transaction"] = transObj;
}

void FMVipForward::orderRefund(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_REFUND;
    QJsonObject transaction;
    transaction["memberTransId"] = job["fm_id"];
    fmjob["transaction"] = transaction;
}

void FMVipForward::orderRevoke(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_REVOKE;

    Q_UNUSED(job);
}

void FMVipForward::logined(const QJsonObject &serverJob, QJsonObject &posJob)
{
    QJsonObject memberObj = serverJob["memberInfo"].toObject();
    posJob["fm_open_id"] = memberObj["account"];
    posJob["prompt"] = 0;
    posJob["birthday"] = memberObj["birthday"];

    addSessionData("fm_open_id", memberObj["account"].toString());
    addSessionData("amount", getString(memberObj["amount"].toInt()));
    addSessionData("score", getString(memberObj["score"].toInt()));
    addSessionData("canPay", getString(memberObj["canPay"].toInt()));

    QJsonArray couponArr = memberObj["couponList"].toArray();
    foreach(QJsonValue value , couponArr)
    {
        QJsonObject co = value.toObject();
        QString codeStr = co["couponCode"].toString();
        double amount = co["disAmount"].toInt()/100.0;
        QString desc = co["desc"].toString();

        _sessionCouponMap[co["couponCode"].toString()] = Coupon{codeStr, amount, desc};
    }

    QString name = memberObj["name"].toString();
    name = (name.isEmpty()) ? "未知" : name;
    QString mobile = memberObj["mobile"].toString();
    mobile = (mobile.isEmpty()) ? "未知" : mobile;
    addSessionData("name", name);
    addSessionData("mobile", mobile);
}

void FMVipForward::funded(const QJsonObject &serverJob, QJsonObject &posJob)
{
    QJsonObject cardInfo = serverJob["cardInfo"].toObject();
    posJob["fm_id"] = cardInfo["memberTransId"];
    posJob["fm_open_id"] = cardInfo["account"];
    posJob["print"] = serverJob["print"];
    posJob["prompt"] = 0;
}

void FMVipForward::couponed(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["pid"] = serverJob["productCode"];
    posJob["fm_open_id"] = serverJob["account"];
    posJob["prompt"] = 1;
}

void FMVipForward::payed(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 0;
    int codeAmount = serverJob["codeAmount"].toInt();
    int scoreAmount = serverJob["scoreAmount"].toInt();
    int paid_total_amount = sessionDataInt("paid_amount") + codeAmount + scoreAmount;

    posJob["fm_open_id"] = serverJob["account"];
    posJob["fm_id"] = serverJob["memberTransId"];
    posJob["total_amount"] =  sessionDataInt("order_amount");
    posJob["paid_total_amount"] = paid_total_amount;
    posJob["invoice_amount"] =  paid_total_amount;
    posJob["incentives_amount"] =  0;

    QJsonArray pay_ids;
    QJsonObject pay_obj_1, pay_obj_2;
    pay_obj_1["pay_id"] = "24";
    pay_obj_1["pay_str"] = "会员储值金支付";
    pay_obj_1["paid_total_amount"] = serverJob["codeAmount"];
    pay_obj_2["pay_id"] = "25";
    pay_obj_2["pay_str"] = "会员积分支付";
    pay_obj_2["paid_total_amount"] = serverJob["scoreAmount"];
    pay_ids.append(pay_obj_1);
    pay_ids.append(pay_obj_2);

    // 处理代金券
    QJsonArray server_coupons = serverJob["coupons"].toArray();
    for(auto value : server_coupons)
    {
        auto s_coupon = value.toObject();
        QJsonObject p_coupon;
        p_coupon["pay_id"] = "77";
        p_coupon["pay_str"] = "代金券支付";
        p_coupon["paid_total_amount"] = s_coupon["disAmount"];
        p_coupon["code"] = s_coupon["couponCode"];
        pay_ids.append(p_coupon);
    }
    posJob["pay_ids"] = pay_ids;

    QJsonObject forward = _sessionForward;
    posJob["forward"] = forward;
}

void FMVipForward::finaled(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 1;
    posJob["fm_id"] = serverJob["memberTransId"];
    posJob["print1"] = serverJob["print1"];
    posJob["print2"] = serverJob["print2"];
}

void FMVipForward::orderRefunded(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 1;
    Q_UNUSED(serverJob);
}

void FMVipForward::orderRevoked(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 1;
    Q_UNUSED(serverJob);
}











QString FMVipForward::sign(const QJsonObject &reqJob) const
{
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;
    mapData["appId"] = reqJob.value("appId").toString();
    mapData["reqType"] = QString::number(reqJob.value("reqType").toInt());
    mapData["storeId"] = reqJob.value("storeId").toString();
    mapData["stationId"] = reqJob.value("stationId").toString();
    mapData["operatorId"] = reqJob.value("operatorId").toString();
    mapData["partnerId"] = reqJob.value("partnerId").toString();
    mapData["t"] = reqJob.value("t").toString();

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KeyCode);

    QByteArray bt;
    bt.append(sb);

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}

void FMVipForward::resetSessionData(const QJsonObject &jsonObj)
{
    foreach (QString sessionData, _sessionDataList)
    {
        addSessionData(sessionData, jsonObj[sessionData].toString());
    }
    _sessionCouponMap.clear();
}

void FMVipForward::clearSessionData()
{
    _sessionCouponMap.clear();
    _sessionDataMap.clear();
    _sessionForward = QJsonObject();
}

void FMVipForward::addSessionData(const QString key, const QString value)
{
    _sessionDataMap.insert(key, value);
}

QString FMVipForward::sessionData(const QString key) const
{
    QString value = "";
    if (_sessionDataMap.contains(key)) {
        value = _sessionDataMap[key];
    }
    return value;
}

int FMVipForward::sessionDataInt(const QString key) const
{
    QString value = sessionData(key);
    return value.toInt();
}

void FMVipForward::setSessionCouponMap(QMap<QString, Coupon> couponMap)
{
    _sessionCouponMap = couponMap;
}

QMap<QString, Coupon> FMVipForward::sessionCouponMap() const
{
    return _sessionCouponMap;
}
