#include "fmvipdispatcher.h"
#include "fmviplogin.h"
#include "fmvippanel.h"
#include "fmviporder.h"
#include "fmvipfund.h"
#include "fmvipforward.h"
#include "fmmsgwnd.h"
#include <QDebug>
#include <QJsonDocument>
#include <QJsonObject>
#include <QApplication>
#include <QMessageBox>
#include <QMenu>

FMVipDispatcher::FMVipDispatcher(QObject *parent)
    : QObject(parent),
      FMApiRelay(),
      _vindow(0),
      isLastOne(true),
      _windowReturn(0),
      _sysIcon(new QSystemTrayIcon)
{
    connect(this, SIGNAL(requested(QJsonObject)), SLOT(onRequest(QJsonObject)));
    connect(this, SIGNAL(responded(QByteArray)), SLOT(onResponse(QByteArray)));
    connect(FMVipForward::instance(), SIGNAL(serverResponsed(QJsonObject)), SLOT(onServerResponsed(QJsonObject)));

    QIcon icon = QIcon(":/img_logo.png");
    _sysIcon->setIcon(icon);
    _sysIcon->setToolTip("FMVIP");

    auto menu = new QMenu();
    auto action = new QAction("退出", this);
    connect(action, SIGNAL(triggered(bool)), this, SLOT(onQuit()));
    menu->addAction(action);
    _sysIcon->setContextMenu(menu);

    connect(_sysIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), SLOT(onActiveSysTrayIcon(QSystemTrayIcon::ActivationReason)));
    _sysIcon->show();

    _sysIcon->showMessage("非码VIP提醒", "非码VIP程序已启动");
}

FMVipDispatcher::~FMVipDispatcher()
{
    if(_sysIcon != NULL)
    {
        delete _sysIcon;
        _sysIcon = NULL;
    }
}

void FMVipDispatcher::onQuit()
{
    this->_ParseRequest("{\"fm_cmd\":\"-1\"}");
}

void FMVipDispatcher::onActiveSysTrayIcon(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger:
        break;
    default:
        break;
    }
}

INT FMVipDispatcher::_ParseRequest(LPSTR data)
{
    qDebug() << "\n\n\nPOS发来的请求 ===>\n" << data;
    QJsonParseError err;
    QJsonDocument json = QJsonDocument::fromJson(data, &err);
    if (err.error != QJsonParseError::NoError) {
        return FM_API_BADJSON;
    }
    _posReqObj = json.object();
    FMVipForward::instance()->resetSessionData(_posReqObj);

    isLastOne = true;
    _serverRspData.clear();
    isPayLogin = false;

    // 唤起客户端界面
    emit requested(_posReqObj);


    return FM_API_SUCCESS;
}

BOOL FMVipDispatcher::_GetResponse(LPSTR &rsp, UINT &len)
{
    qDebug() << __FUNCTION__;

    mutex.lock();
    if(_serverRspData.isEmpty())
    {
        // 服务器还未返回则一直阻塞
        serverIsBusy.wait(&mutex);
    }
    rsp = _serverRspData.data();
    len = _serverRspData.length();
    mutex.unlock();

    qDebug() << "发给POS ===>\n" << rsp;

    return 1;
}

void FMVipDispatcher::onRequest(const QJsonObject &jsonObj)
{
    _windowReturn = 1;

    QString type = jsonObj["fm_cmd"].toString();

    if (type == Type_Login) {
        login();
    }
    else if (type == Type_Coupon) {
        coupon();
    }
    else if (type == Type_Order_Refund || type == Type_Fund_Refund) {
        orderRefund();
    }
    else if (type == Type_Order_Revoke) {
        orderRevoke();
    }
    else if (type == Type_Final) {
        final();
    }
#ifdef _DEBUG
    else if (type == "-1") {
        qDebug() << "Quit";
        jsonObj["reqType"] = -1;
        QJsonDocument d = QJsonDocument(jsonObj);
        _serverRspData = d.toJson();
        qApp->quit();
    }
#endif
    else if(type == Type_Pay || type == Type_Fund) {
        requestSuccess = true;
        // 支付、充值、结算先检查登陆状态。
        QJsonObject copyJsonObj(jsonObj);
        if(type == Type_Fund && !isLogined())
        {
            _posReqObj["fm_cmd"] = Type_Login;
            isLastOne = false;
            login();
        }
        if(type == Type_Pay)
        {
            _posReqObj["fm_cmd"] = Type_Login;
            isLastOne = false;
            isPayLogin = true;
            login();
        }
        if (requestSuccess == true && _windowReturn != -1) {

            _posReqObj = copyJsonObj;
            isLastOne = true;
            if (type == Type_Pay) {
                pay();
            }
            else if (type == Type_Fund) {
                fund();
            }
        }

        FMVipForward::instance()->clearSessionData();
    } else {
        _serverRspData = Err_ErrorFMCMD;
    }
}

void FMVipDispatcher::onResponse(const QByteArray &rsp)
{
    qDebug() << __FUNCTION__;
    Q_UNUSED(rsp)
}

void FMVipDispatcher::onFinished()
{
    qDebug() << __FUNCTION__;

    if(_windowReturn == -1){
        _serverRspData = Err_WindowClose;
        serverIsBusy.wakeAll();
    }

    _vindow = 0;
}

void FMVipDispatcher::onServerResponsed(const QJsonObject &rspObj)
{
    int status = rspObj["statusCode"].toInt();
    int prompt = rspObj["prompt"].toInt();

    if(_vindow != 0) {
        _vindow->setIsBusy(false);
        // 支付时， 会员认证的返回，如果不可付款
        int canPan = FMVipForward::instance()->sessionDataInt("canPay");

        // 如果是支付认证则不关闭窗口
        if(isPayLogin)
        {
            if(status != 100)
            {
                _vindow->resetWnd();
                FMMsgWnd::FailureWnd(rspObj["msg"].toString());
                return;
            }else if(canPan != 1)
            {
                _vindow->resetWnd();
                FMMsgWnd::FailureWnd("请使用支付码付款");
                return;
            }
            isPayLogin = false;
        }
        _vindow->deleteLater();
    }

    QJsonDocument rspDoc(rspObj);
    QByteArray rspData = rspDoc.toJson();
    _serverRspData = rspData;


    if(status != 100) {
        if(prompt == 0)
        {
            QString msg = rspObj["msg"].toString();
            FMMsgWnd::FailureWnd(msg);
        }
        requestSuccess = false;
        _vindow = 0;
        serverIsBusy.wakeAll();
    }

    if( isLastOne ) {
        serverIsBusy.wakeAll();
    }
}

// 向服务器发送请求
void FMVipDispatcher::onDoPost()
{
    FMVipForward::instance()->parseRequest(_posReqObj);
}

void FMVipDispatcher::login()
{
    if (!_vindow) {
        FMVipLogin *login = new FMVipLogin(isPayLogin);
        connect(login, SIGNAL(destroyed(QObject*)), SLOT(onFinished()));
        connect(login, SIGNAL(doPost()), SLOT(onDoPost()));
        _vindow = login;
        _windowReturn = login->exec();
    }
}

void FMVipDispatcher::coupon()
{
    onDoPost();
}

void FMVipDispatcher::pay()
{
    QJsonObject transactions = _posReqObj["transactions"].toObject();
    int order_amount = transactions["order_amount"].toInt();
    int paid_amount = transactions["paid_amount"].toInt();
    int standard_amount = transactions["standard_amount"].toInt();
    int needPay = order_amount - paid_amount;
    FMVipForward::instance()->addSessionData("paid_amount", QString::number(paid_amount));
    FMVipForward::instance()->addSessionData("order_amount",QString::number(order_amount));
    FMVipForward::instance()->addSessionData("needPay", QString::number(needPay));
    FMVipForward::instance()->addSessionData("standard_amount", QString::number(standard_amount));

    if (!_vindow) {
        FMVipOrder *order = new FMVipOrder;
        connect(order, SIGNAL(destroyed(QObject*)), SLOT(onFinished()));
        connect(order, SIGNAL(doPost()), SLOT(onDoPost()));
        _vindow = order;
        _windowReturn = order->exec();
    }

}

void FMVipDispatcher::fund()
{
    FMVipForward::instance()->addSessionData("charge_amount", QString::number(_posReqObj["charge_amount"].toInt()));

    if (!_vindow) {
        FMVipFund *fund = new FMVipFund;
        connect(fund, SIGNAL(destroyed(QObject*)), SLOT(onFinished()));
        connect(fund, SIGNAL(doPost()), SLOT(onDoPost()));
        _vindow = fund;
        _windowReturn = fund->exec();
    }
}

void FMVipDispatcher::final()
{
    onDoPost();
}

void FMVipDispatcher::orderRefund()
{
    onDoPost();
}

void FMVipDispatcher::orderRevoke()
{
    onDoPost();
}

BOOL FMVipDispatcher::isLogined()
{
    QString account = FMVipForward::instance()->sessionData("fm_open_id");
    return account != "";
}
