#include "fmsockserver.h"
#include <fmutils/fmutils.hpp>

FMSockServer::FMSockServer(FMApiRelay *relay) : 
    _worker(0),
    _worker_tid(0),
    _port(0),
    _relay(relay)
{
    InitializeCriticalSection(&_cs);
    WORD ver_req = MAKEWORD(2,2); 
    WSADATA wsa_data = { 0 };
    int err  =  WSAStartup(ver_req, &wsa_data);
    if(0 != err) {
        FMLOG(_T("init socket failed: %d"), WSAGetLastError());
    }
    _socket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    int time_out = 5000;
    setsockopt(_socket, SOL_SOCKET, SO_RCVTIMEO, (char *)&time_out, sizeof(int));

    CHAR v = 1;
    err = setsockopt(_socket, SOL_SOCKET, SO_REUSEADDR, &v, sizeof(int));
    if (err == -1) {
        FMLOG(_T("Failed setting socket option: %d."), WSAGetLastError());
    }

    memset(&_addr_in, 0, sizeof(_addr_in));

    _addr_in.sin_family = AF_INET;
    _addr_in.sin_addr.S_un.S_addr = htonl(INADDR_ANY);
}


FMSockServer::~FMSockServer()
{
    if (_socket != INVALID_SOCKET) {
        Stop();
    }

    WSACleanup();
    DeleteCriticalSection(&_cs);
}


VOID FMSockServer::Listen(USHORT port)
{
    _port = port;
    _worker = CreateThread(NULL, 0, FMSockServer::RecvWorker, this, 0, &_worker_tid);
    FMLOG(_T("Server thread %ld started."), _worker_tid);
}


BOOL FMSockServer::Stop()
{
    closesocket(_socket);
    _socket = INVALID_SOCKET;

    WaitForSingleObject(_worker, INFINITE);
    CloseHandle(_worker);
    FMLOG(_T("Server thread %ld stopped. exit."), _worker_tid);

    return TRUE;
}


DWORD FMSockServer::RecvWorker(LPVOID param)
{
    FMSockServer *p_svr = (FMSockServer*)param;

    p_svr->_Listen();

    return 0;
}


BOOL FMSockServer::_Listen()
{
    _addr_in.sin_port = htons(_port);

    int sock_res = bind(_socket, (const sockaddr*)&_addr_in, sizeof(_addr_in) );
    if( sock_res == SOCKET_ERROR ) {
        FMLOG(_T("Failed binding on port %d: %d."), _port, WSAGetLastError());
        return FALSE;
    }

    sock_res = listen(_socket, BACKLOG);
    if( sock_res == SOCKET_ERROR ) {
        FMLOG(_T("Failed listening port %d: %d."), _port, WSAGetLastError());
        return FALSE;
    }

    FMLOG(_T("Listening on port %d..."), _port);

    while(TRUE) {
        sockaddr_in addr_client = { 0 };
        int addr_len = sizeof(addr_client);
        SOCKET sclient = INVALID_SOCKET;
        sclient = accept(_socket, (sockaddr*)&addr_client, &addr_len);
        
        if (INVALID_SOCKET != sclient) {
            DWORD str_addr_len = 32;
            TCHAR str_addr[32] = { 0 };
            if (0 == WSAAddressToString((sockaddr*)&addr_client, addr_len, NULL, str_addr, &str_addr_len)) {
                FMLOG(_T("Accepted socket of %s"), str_addr);
            }
            
            LPFMSOCKDATA io_data = NULL;
            io_data = (LPFMSOCKDATA)HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof(FMSOCKDATA));
            //! Receive length first
            io_data->buffer.len = sizeof(io_data->header);
            io_data->buffer.buf = (char*)&io_data->header;
            _relay->SetSocketDescriptor(sclient);
            io_data->relay = _relay;
            io_data->socket = sclient;
            io_data->offset = 0;
            WSARecv(sclient, &io_data->buffer, 1, NULL, &io_data->flags, &io_data->overlap, FMSockServer::RecvRoutine);


            DWORD res = SleepEx(1000, TRUE);
            while (res != WAIT_IO_COMPLETION) {
                FMLOG(_T("Rewaiting for msg header I/O completion..."));
                res = SleepEx(1000, TRUE);
            }
        }
        else {
            DWORD err = WSAGetLastError();
            if (WSAEINTR == err) {
                _socket = INVALID_SOCKET;
                break;
            }
            if (_socket == INVALID_SOCKET) {
                break;
            }
            FMLOG(_T("Failed accepting client: %ld."), err);
            SleepEx(200, TRUE);
        }
    }

    FMLOG(_T("Stopped listening."));

    return TRUE;
}


void FMSockServer::RecvRoutine(DWORD err, DWORD bytes, LPWSAOVERLAPPED overlapped, DWORD flags)
{
    LPFMSOCKDATA io_data = (LPFMSOCKDATA)overlapped;

    if (bytes <= 0) {
        closesocket(io_data->socket);
        HeapFree(GetProcessHeap(), 0, io_data);
        return;
    }
    UINT data_len = io_data->offset + bytes;
    FMLOG(_T("Socket %d received bytes: %ld."), io_data->socket, data_len);

    if (io_data->header.flag == FMSOCKFLAG) {
        if (err != 0 || bytes == 0) {
            //! Connection was closed by client
            closesocket(io_data->socket);
            FMLOG(_T("Error receiving data."));
        }
        else if (data_len < io_data->header.len) {
            //! Receive data
            if (strlen(io_data->msg) == 0) {
                FMLOG(_T("Legal header. Starts receiving data."));
                LPFMSOCKDATA io_new_data = NULL;
                UINT len = sizeof(FMSOCKDATA) + io_data->header.len;
                io_new_data = (LPFMSOCKDATA)HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, len);

                io_new_data->buffer = io_data->buffer;
                io_new_data->buffer.len = io_data->header.len;
                io_new_data->buffer.buf = io_new_data->msg;

                io_new_data->socket = io_data->socket;
                io_new_data->relay = io_data->relay;
                io_new_data->flags = io_data->flags;

                io_new_data->header = io_data->header;

                WSARecv(io_new_data->socket, &io_new_data->buffer, 1, NULL, &io_new_data->flags, &io_new_data->overlap, FMSockServer::RecvRoutine);
                DWORD res = SleepEx(1000, TRUE);
                while (res != WAIT_IO_COMPLETION) {
                    FMLOG(_T("Rewaiting for new data(offset: %ld)I/O completion..."), io_new_data->offset);
                    res = SleepEx(1000, TRUE);
                }

                //! Free header buffer
                HeapFree(GetProcessHeap(), 0, io_data);
            }
            else {
                FMLOG(_T("Received data %ld bytes..."), bytes);
                io_data->offset += bytes;
                io_data->buffer.len = io_data->header.len - io_data->offset;
                io_data->buffer.buf = io_data->msg + io_data->offset;
                WSARecv(io_data->socket, &io_data->buffer, 1, NULL, &io_data->flags, &io_data->overlap, FMSockServer::RecvRoutine);
                DWORD res = SleepEx(1000, TRUE);
                while (res != WAIT_IO_COMPLETION && io_data->offset < io_data->header.len) {
                    FMLOG(_T("Rewaiting for old data(offset: %ld)I/O completion..."), io_data->offset);
                    res = SleepEx(1000, TRUE);
                }
            }
        }
        else {
            //! Legal freemud api data
            FMLOG(_T("Received data %ld bytes."), bytes);
            io_data->offset += bytes;
            io_data->msg[io_data->offset] = '\0';
            
            char *response = NULL; UINT len = 0;
            if (io_data->relay) {
                io_data->relay->Transfer(io_data->msg, response, len);
            }
            if (response) {
                FMLOG("Transfered %s", response);
                send(io_data->socket, response, len, 0);
            }
            else {
                FMLOG(_T("Failed to determine response data."));
            }
            closesocket(io_data->socket);
            
            //! Free all buffer
            HeapFree(GetProcessHeap(), 0, io_data);
        }
    }
    else {
        FMLOG(_T("Incompatible protocol."));
    }
}

VOID FMSockServer::SetRelay(FMApiRelay* relay)
{
    FMAutoLock auto_lock(_cs);
    _relay = relay;
}

