﻿#include "fmvipforward.h"
#include <QJsonDocument>
#include <QJsonParseError>
#include <QJsonObject>
#include <QJsonArray>
#include <QCryptographicHash>
#include <QDateTime>
#include <QEventLoop>
#include <QTimer>
#include <QSettings>
#include <QApplication>
#include "fmbackup.h"

#include <QDebug>

FMVipForward* FMVipForward::_instance = NULL;

FMVipForward* FMVipForward::instance()
{
    if(_instance == NULL) {
        _instance = new FMVipForward();
    }
    return _instance;
}

FMVipForward::FMVipForward(QObject *parent) : QObject(parent)
{
    _urlStr = _cfg.GetTransUrlFormat();
    _sessionDataList = QString("operator_id,pos_id,business_date").split(",");
}

FMVipForward::~FMVipForward()
{
    if(_instance != NULL) {
        delete _instance;
        _instance = NULL;
    }
}

void FMVipForward::parseRequest(const QJsonObject &jsonObj)
{
    QString type = jsonObj["fm_cmd"].toString();
    QJsonObject fm_jsonObj;
    if(type == Type_Login) {
        login(jsonObj, fm_jsonObj);
    } else if(type == Type_Fund) {
        fund(jsonObj, fm_jsonObj);
    } else if(type == Type_Coupon) {
        coupon(jsonObj, fm_jsonObj);
    } else if(type == Type_Pay) {
        pay(jsonObj, fm_jsonObj);
    } else if(type == Type_Final) {
        final(jsonObj, fm_jsonObj);
    } else if(type == Type_Order_Refund || type == Type_Fund_Refund) {
        orderRefund(jsonObj, fm_jsonObj);
    } else if(type == Type_Order_Revoke) {
        orderRevoke(jsonObj, fm_jsonObj);
    }

    _reqType = fm_jsonObj["reqType"].toInt();

    // 固定部分
    fm_jsonObj["appId"] = AppId;
    fm_jsonObj["partnerId"] = PartnerId;
    fm_jsonObj["version"] = VersionInfo;
    fm_jsonObj["t"] = QString::number(QDateTime::currentMSecsSinceEpoch());

    // 可从请求中提取的部分
    FMApiPropMap *pm = _cfg.GetPropsMap();
    if (pm && pm->size() > 0) {
        FMApiPropMap::iterator iter = pm->begin();
        for(; iter != pm->end(); ++iter) {
            if(jsonObj.contains(iter->first)) {
                QJsonValue v = jsonObj[iter->first];
                fm_jsonObj[iter->second] = v;
            }
        }
    }

    // 向服务器发送请求
    request(fm_jsonObj);
}

void FMVipForward::request(const QJsonObject &reqJob)
{
    // 设置URL
    QString signStr = sign(reqJob);

    _req.setUrl(_urlStr.arg(signStr));

    QJsonDocument json(reqJob);
    QByteArray reqData = json.toJson(QJsonDocument::Compact);

    // 结算请求时，如果没用余额、积分、优惠券、代金券则备份请求信息
    auto trans = reqJob["transaction"].toObject();
    if (reqJob["reqType"]== FM_VIP_FINAL) {
        if( trans["codeAmount"].toInt()==0 &&
            trans["scoreAmount"].toInt()==0 &&
            trans["coupons"].toArray().isEmpty() ) {
            _needBackupReqJob = reqJob;
        }
    }

    qDebug() << "向服务器发送:" << json;
    qDebug() << "请求签名:" << signStr;

    // 设置请求头
    _req.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    _req.setHeader(QNetworkRequest::ContentLengthHeader, reqData.length());

    QNetworkAccessManager   _nam;
    auto reply = _nam.post(_req, reqData);

    // 使用定时器处理超时
    QEventLoop loop;
    connect(&_nam, SIGNAL(finished(QNetworkReply*)), &loop, SLOT(quit()));
    QTimer timer;
    timer.setSingleShot(true);
    connect(&timer, SIGNAL(timeout()),&loop,SLOT(quit()));
    timer.start(1000 * 30);
    loop.exec();
    if(timer.isActive())
    {
        timer.stop();
        onServerFinished(reply, false);
    }else{
        //超时
        qDebug() << "定时器超时";
        onServerFinished(reply, true);
    }
}

void FMVipForward::onServerFinished(QNetworkReply *reply, bool isTimeOut)
{
    QJsonObject posObj;

    if(reply->error() != QNetworkReply::NoError || isTimeOut == true) {

        // 如果有需要备份的请求数据，则直接返回成功
        if (!_needBackupReqJob.isEmpty()) {

            int newId = FMBackup::instance()->insertNewRow();
            QString uuid = QString("E%1%2%3%4")
                    .arg(_needBackupReqJob["storeId"].toString())
                    .arg(_needBackupReqJob["stationId"].toString().toInt(), 2, 10, QChar('0'))
                    .arg(QDateTime::currentDateTime().toString("yyMMdd"))
                    .arg(newId, 5, 10, QChar('0'));
            _needBackupReqJob["memberTransId"] = uuid;

            QJsonDocument jdoc(_needBackupReqJob);
            FMBackup::instance()->updateReqData(newId, _req.url().toString(), jdoc.toJson(QJsonDocument::Compact));

            qDebug() << "Backup: " << jdoc;
            posObj["statusCode"] = 100;
            posObj["msg"] = "fmv:结算成功";
            posObj["fm_id"] = uuid;
            posObj["prompt"] = 1;
            posObj["print1"] = "";
            posObj["print2"] = "";
        } else {
            posObj["statusCode"] = 404;
            QString errCode = QString::number(int(reply->error()));
            QString errMsg = "fmv:网络连接错误(%1): %2";
            // 获取错误提示
            QSettings setting(qApp->applicationDirPath()+"/config.ini", QSettings::IniFormat);
            setting.setIniCodec("GBK");
            setting.beginGroup("NetworkError");
            QString errInfo = setting.value(errCode).toString();
            if (errInfo == "") {
                errInfo = setting.value("default").toString();
            }
            posObj["msg"] = errMsg.arg(errCode).arg(errInfo);
            setting.endGroup();
            if (isTimeOut) {
                posObj["msg"] = "fmv:向非码服务器请求超时";
            }
            qDebug() << "Network error : " << posObj["msg"].toString();
        }

    } else {
        QByteArray data = reply->readAll();
        QJsonParseError err;
        QJsonDocument jdoc = QJsonDocument::fromJson(data, &err);

        qDebug() << "服务器返回:" << jdoc;

        if(err.error != QJsonParseError::NoError || !jdoc.isObject()) {
            posObj["statusCode"] = 1000;
            posObj["msg"] = "fmv:服务器返回数据异常";
        } else {
            QJsonObject serverObj = jdoc.object();

            // 服务器返回错误信息
            if(serverObj.contains("errcode") && serverObj.contains("errmsg")) {
                posObj["statusCode"] = serverObj["errcode"];
                posObj["msg"] = serverObj["errmsg"];
            }
            // 服务器返回正常信息
            else {
                posObj["statusCode"] = serverObj["statusCode"];
                posObj["msg"] = serverObj["msg"];
                switch (_reqType) {
                case FM_VIP_LOGIN:
                    logined(serverObj, posObj);
                    break;
                case FM_VIP_COUPON:
                    couponed(serverObj, posObj);
                    break;
                case FM_VIP_FUND:
                    funded(serverObj, posObj);
                    break;
                case FM_VIP_PAY:
                    payed(serverObj, posObj);
                    break;
                case FM_VIP_FINAL:
                    finaled(serverObj, posObj);
                    break;
                case FM_VIP_REVOKE:
                    orderRevoked(serverObj, posObj);
                    break;
                case FM_VIP_REFUND:
                    orderRefunded(serverObj, posObj);
                    break;
                default:
                    break;
                }
            }
        }
    }
    _needBackupReqJob = QJsonObject();
    emit serverResponsed(posObj);
}


void FMVipForward::login(const QJsonObject &job, QJsonObject &fmjob)
{
    Q_UNUSED(job);
    fmjob["reqType"] = FM_VIP_LOGIN;
    fmjob["code"] = SESSIONDATA_STRING("code");
}
void FMVipForward::fund(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_FUND;

    QJsonObject transObj;
    transObj["account"] = SESSIONDATA_STRING("fm_open_id");
    transObj["amount"] = job["charge_amount"];
    transObj["cashAmount"] = job["charge_amount"];
    transObj["thirdAmount"] = 0;
    transObj["thirdPayType"] = 0;
    transObj["thirdPayTransId"] = "";

    fmjob["transaction"] = transObj;
}
void FMVipForward::coupon(const QJsonObject &job, QJsonObject &fmjob)
{
    Q_UNUSED(job);
    fmjob["reqType"] = FM_VIP_COUPON;
}
void FMVipForward::pay(const QJsonObject &job, QJsonObject &fmjob)
{
    Q_UNUSED(job);
    fmjob["reqType"] = FM_VIP_PAY;

    SESSIONDATA_ADD("forward", job["forward"].toObject());

    QJsonObject transObj;
    transObj["account"] = SESSIONDATA_STRING("fm_open_id");
    transObj["codeAmount"] = SESSIONDATA_INT("codeAmount");
    transObj["isUseScore"] = SESSIONDATA_INT("isUseScore");

    QJsonArray couponsArr;
    for(auto code : SESSIONDATA_MAP("payCouponMap").keys())
    {
        couponsArr.append(code);
    }
    transObj["coupons"] = couponsArr;

    fmjob["transaction"] = transObj;
}
void FMVipForward::final(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_FINAL;

    QJsonObject transObj;
    QJsonObject posTransObj = job["transactions"].toObject();
    transObj["account"] = job["fm_open_id"];
    int amount = posTransObj["order_amount"].toInt();
    int paidAmount = posTransObj["paid_amount"].toInt();
    QJsonArray pay_ids = posTransObj["pay_ids"].toArray();
    fmjob["memberTransId"] = job["fm_id"];

    QJsonArray coupons = posTransObj["coupons"].toArray();  // 优惠券
    int codeAmount=0, scoreAmount=0, cashAmount=0, thirdAmount=0;
    transObj["thirdPayType"] = 4;
    foreach (QJsonValue pay_v , pay_ids)
    {
        QJsonObject pay_ob = pay_v.toObject();
        QString type = pay_ob["pay_id"].toString();
        if(type == "24") {
            codeAmount = pay_ob["paid_total_amount"].toInt();
        } else if(type == "25") {
            scoreAmount = pay_ob["paid_total_amount"].toInt();
        } else if(type == "77") {               // 代金券
            coupons.append(pay_ob["code"].toString());
        } else if(type == "72" || type == "73") {
            thirdAmount = pay_ob["paid_total_amount"].toInt();
            transObj["thirdPayTransId"] = pay_ob["pay_transId"];

            if (type == "72") {                 // 支付宝
                transObj["thirdPayType"] = 1;
            } else if(type == "73") {           // 微信
                transObj["thirdPayType"] = 2;
            }
        }
    }
    transObj["coupons"] = coupons;

    cashAmount = paidAmount - codeAmount -scoreAmount - thirdAmount;
    transObj["amount"] = amount;
    transObj["payAmount"] = paidAmount;
    transObj["codeAmount"] = codeAmount;
    transObj["scoreAmount"] = scoreAmount;
    transObj["cashAmount"] = cashAmount;
    transObj["thirdAmount"] = thirdAmount;

    QJsonArray posProdArray = posTransObj["products"].toArray();
    QJsonArray prodArray;
    foreach (QJsonValue v , posProdArray)
    {
        QJsonObject po = v.toObject();
        QJsonObject o;
        o["consumeNum"] = po["consume_num"];
        o["price"] = po["price"];
        o["productId"] = po["pid"];
        prodArray.append(o);
    }
    transObj["products"] = prodArray;
    fmjob["transaction"] = transObj;
}

void FMVipForward::orderRefund(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_REFUND;
    QJsonObject transaction;
    transaction["memberTransId"] = job["fm_id"];
    fmjob["transaction"] = transaction;
}

void FMVipForward::orderRevoke(const QJsonObject &job, QJsonObject &fmjob)
{
    fmjob["reqType"] = FM_VIP_REVOKE;

    Q_UNUSED(job);
}

void FMVipForward::logined(const QJsonObject &serverJob, QJsonObject &posJob)
{
    QJsonObject memberObj = serverJob["memberInfo"].toObject();
    posJob["fm_open_id"] = memberObj["account"];
    posJob["prompt"] = 0;
    posJob["birthday"] = memberObj["birthday"];

    SESSIONDATA_ADD("fm_open_id", memberObj["account"].toString());
    SESSIONDATA_ADD("amount", getString(memberObj["amount"].toInt()));
    SESSIONDATA_ADD("score", getString(memberObj["score"].toInt()));
    SESSIONDATA_ADD("canPay", getString(memberObj["canPay"].toInt()));

    QJsonArray couponArr = memberObj["couponList"].toArray();

    QMap<QString, QVariant> couponMap;
    foreach(QJsonValue value , couponArr)
    {
        QJsonObject co = value.toObject();
        QString codeStr = co["couponCode"].toString();
        double amount = co["disAmount"].toInt()/100.0;
        QString desc = co["desc"].toString();

        QVariant v;
        v.setValue(Coupon{codeStr, amount, desc});
        couponMap[co["couponCode"].toString()] = v;
    }
    SESSIONDATA_ADD("couponMap", couponMap);

    QString name = memberObj["name"].toString();
    name = (name.isEmpty()) ? "未知" : name;
    QString mobile = memberObj["mobile"].toString();
    mobile = (mobile.isEmpty()) ? "未知" : mobile;
    QString birthday = memberObj["birthday"].toString();
    birthday = (birthday.isEmpty()) ? "未知" : birthday;
    SESSIONDATA_ADD("name", name);
    SESSIONDATA_ADD("mobile", mobile);
    SESSIONDATA_ADD("birthday", birthday);
}

void FMVipForward::funded(const QJsonObject &serverJob, QJsonObject &posJob)
{
    QJsonObject cardInfo = serverJob["cardInfo"].toObject();
    posJob["fm_id"] = cardInfo["memberTransId"];
    posJob["fm_open_id"] = cardInfo["account"];
    posJob["print"] = serverJob["print"];
    posJob["prompt"] = 0;
}

void FMVipForward::couponed(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["pid"] = serverJob["productCode"];
    posJob["fm_open_id"] = serverJob["account"];
    posJob["prompt"] = 1;
}

void FMVipForward::payed(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 0;

    QJsonArray pay_ids;
    QJsonObject pay_obj_1, pay_obj_2;
    pay_obj_1["pay_id"] = "24";
    pay_obj_1["pay_str"] = "会员储值金支付";
    pay_obj_1["paid_total_amount"] = serverJob["codeAmount"];
    pay_obj_2["pay_id"] = "25";
    pay_obj_2["pay_str"] = "会员积分支付";
    pay_obj_2["paid_total_amount"] = serverJob["scoreAmount"];
    pay_ids.append(pay_obj_1);
    pay_ids.append(pay_obj_2);

    int couponAmount = 0;
    // 处理代金券
    QJsonArray server_coupons = serverJob["coupons"].toArray();
    for(auto value : server_coupons)
    {
        auto s_coupon = value.toObject();
        QJsonObject p_coupon;
        p_coupon["pay_id"] = "77";
        p_coupon["pay_str"] = "代金券支付";
        couponAmount += s_coupon["disAmount"].toInt();
        p_coupon["paid_total_amount"] = s_coupon["disAmount"];
        p_coupon["code"] = s_coupon["couponCode"];
        pay_ids.append(p_coupon);
    }
    posJob["pay_ids"] = pay_ids;

    int codeAmount = serverJob["codeAmount"].toInt();
    int scoreAmount = serverJob["scoreAmount"].toInt();
    int paid_total_amount = SESSIONDATA_INT("paid_amount") + codeAmount + scoreAmount + couponAmount;

    posJob["fm_open_id"] = serverJob["account"];
    posJob["fm_id"] = serverJob["memberTransId"];
    posJob["total_amount"] =  SESSIONDATA_INT("order_amount");
    posJob["paid_total_amount"] = paid_total_amount;
    posJob["invoice_amount"] =  paid_total_amount;
    posJob["incentives_amount"] =  0;

    QJsonObject forward = SESSIONDATA_JSONOBJECT("forward");
    posJob["forward"] = forward;
}

void FMVipForward::finaled(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 1;
    posJob["fm_id"] = serverJob["memberTransId"];
    posJob["print1"] = serverJob["print1"];
    posJob["print2"] = serverJob["print2"];
}

void FMVipForward::orderRefunded(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 1;
    Q_UNUSED(serverJob);
}

void FMVipForward::orderRevoked(const QJsonObject &serverJob, QJsonObject &posJob)
{
    posJob["prompt"] = 1;
    Q_UNUSED(serverJob);
}


QString FMVipForward::sign(const QJsonObject &reqJob) const
{
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;
    mapData["appId"] = reqJob.value("appId").toString();
    mapData["reqType"] = QString::number(reqJob.value("reqType").toInt());
    mapData["storeId"] = reqJob.value("storeId").toString();
    mapData["stationId"] = reqJob.value("stationId").toString();
    mapData["operatorId"] = reqJob.value("operatorId").toString();
    mapData["partnerId"] = reqJob.value("partnerId").toString();
    mapData["t"] = reqJob.value("t").toString();

    if (reqJob["reqType"].toInt() == FM_VIP_FUND) {
        mapData["transId"] = reqJob.value("transId").toString();
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KeyCode);

    qDebug() << QString("Sign string: %1").arg(sb);

    QByteArray bt;
    bt.append(sb);

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}

void FMVipForward::resetSessionData(const QJsonObject &jsonObj)
{
    foreach (QString sessionData, _sessionDataList)
    {
        SESSIONDATA_ADD(sessionData, jsonObj[sessionData].toString());
    }
}

void FMVipForward::clearSessionData()
{
    _sessionDataMap.clear();
}

QVariant FMVipForward::sessionData(const QString key) const
{
    QVariant value;
    if (_sessionDataMap.contains(key)) {
        value = _sessionDataMap[key];
    }
    return value;
}

void FMVipForward::addSessionData(const QString key, const QVariant value)
{
    _sessionDataMap.insert(key, value);
}

void FMVipForward::addSessionData(const QString key, const QMap<QString, Coupon> couponMap)
{
    QVariantMap v_couponMap;
    for(auto coupon : couponMap) {
        QVariant v;
        v.setValue(coupon);
        v_couponMap[coupon.code] = v;
    }
    SESSIONDATA_ADD(key, v_couponMap);
}

QMap<QString, Coupon> FMVipForward::getCouponMap(const QString key)
{
    QVariantMap v_couponMap = SESSIONDATA_MAP(key);
    QMap<QString, Coupon> couponMap;
    for(auto v_coupon : v_couponMap) {
        Coupon coupon = v_coupon.value<Coupon>();
        couponMap[coupon.code] = coupon;
    }
    return couponMap;
}

