#include "fmvipdispatcher.h"
#include "fmviplogin.h"
#include "fmviporder.h"
#include "fmvipfund.h"
#include "fmvipforward.h"
#include "fmmsgwnd.h"
#include <QDebug>
#include <QJsonDocument>
#include <QJsonObject>
#include <QApplication>
#include <QMessageBox>
#include <QMenu>
#include <QTimer>
#include <winsock2.h>

FMVipDispatcher::FMVipDispatcher(QObject *parent)
    : QObject(parent),
      FMApiRelay(),
      _vindow(0),
      isLastOne(true),
      _windowReturn(0),
      _posSocketTimer(new QTimer(this)),
      _sysIcon(new QSystemTrayIcon)
{
    connect(this, SIGNAL(requested(QJsonObject)), SLOT(onRequest(QJsonObject)));
    connect(this, SIGNAL(responded(QByteArray)), SLOT(onResponse(QByteArray)));
    connect(FMVipForward::instance(), SIGNAL(serverResponsed(QJsonObject)), SLOT(onServerResponsed(QJsonObject)));

    _posSocketTimer->setInterval(5000);
    connect(_posSocketTimer, SIGNAL(timeout()), SLOT(onCheckSocket()));
    connect(this, SIGNAL(stopSocketTimer()), _posSocketTimer, SLOT(stop()));
    connect(this, SIGNAL(startSocketTimer()), _posSocketTimer, SLOT(start()));

    QIcon icon = QIcon(":/img_logo.png");
    _sysIcon->setIcon(icon);
    _sysIcon->setToolTip("FMVIP");

    auto menu = new QMenu();
    auto action = new QAction("退出", this);
    connect(action, SIGNAL(triggered(bool)), this, SLOT(onQuit()));
    menu->addAction(action);
    _sysIcon->setContextMenu(menu);

    connect(_sysIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), SLOT(onActiveSysTrayIcon(QSystemTrayIcon::ActivationReason)));
    _sysIcon->show();

    _sysIcon->showMessage("非码VIP提醒", "非码VIP程序已启动");
}

FMVipDispatcher::~FMVipDispatcher()
{
    if(_sysIcon != NULL)
    {
        delete _sysIcon;
        _sysIcon = NULL;
    }

    if(_posSocketTimer != nullptr) {
        delete _posSocketTimer;
        _posSocketTimer = nullptr;
    }

}

void FMVipDispatcher::onQuit()
{
    this->_ParseRequest("{\"fm_cmd\":\"-1\"}");
}

void FMVipDispatcher::onActiveSysTrayIcon(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger:
        break;
    default:
        break;
    }
}

INT FMVipDispatcher::_ParseRequest(LPSTR data)
{
    qDebug() << "POS发来的请求:" << data;
    QJsonParseError err;
    QJsonDocument json = QJsonDocument::fromJson(data, &err);
    if (err.error != QJsonParseError::NoError) {
        return FM_API_BADJSON;
    }
    _posReqObj = json.object();
    FMVipForward::instance()->resetSessionData(_posReqObj);

    isLastOne = true;
    _serverRspData.clear();
    isPayLogin = false;
    isLogin = false;

    // 唤起客户端界面
    emit requested(_posReqObj);

    return FM_API_SUCCESS;
}

BOOL FMVipDispatcher::_GetResponse(LPSTR &rsp, UINT &len)
{
    emit startSocketTimer();
    mutex.lock();
    if(_serverRspData.isEmpty())
    {
        // 服务器还未返回则一直阻塞
        serverIsBusy.wait(&mutex);
    }
    rsp = _serverRspData.data();
    len = _serverRspData.length();
    mutex.unlock();

    qDebug() << "发给POS:" << rsp;

    emit stopSocketTimer();

    return 1;
}

void FMVipDispatcher::onRequest(const QJsonObject &jsonObj)
{
    _windowReturn = 1;

    QString type = jsonObj["fm_cmd"].toString();

    if (type == Type_Login) {
        login();
    }
    else if (type == Type_Coupon) {
        coupon();
    }
    else if (type == Type_Order_Refund || type == Type_Fund_Refund) {
        orderRefund();
    }
    else if (type == Type_Order_Revoke) {
        orderRevoke();
    }
    else if (type == Type_Final) {
        final();
    }
    else if (type == "-1") {
        serverIsBusy.wakeAll();
        qDebug() << "Quit";
        jsonObj["reqType"] = -1;
        QJsonDocument d = QJsonDocument(jsonObj);
        _serverRspData = d.toJson();
        qApp->quit();
    }
    else if(type == Type_Pay || type == Type_Fund) {
        requestSuccess = true;
        // 支付、充值先登陆。
        QJsonObject copyJsonObj(jsonObj);
        if(type == Type_Fund)
        {
            _posReqObj["fm_cmd"] = Type_Login;
            isLastOne = false;
            login();
        }
        if(type == Type_Pay)
        {
            _posReqObj["fm_cmd"] = Type_Login;
            isLastOne = false;
            isPayLogin = true;
            login();
        }
        if (requestSuccess == true && _windowReturn != -1) {

            _posReqObj = copyJsonObj;
            isLastOne = true;
            if (type == Type_Pay) {
                pay();
            }
            else if (type == Type_Fund) {
                fund();
            }
        }

        FMVipForward::instance()->clearSessionData();
    } else {
        _serverRspData = Err_ErrorFMCMD;
    }
}

void FMVipDispatcher::onResponse(const QByteArray &rsp)
{
    qDebug() << __FUNCTION__;
    Q_UNUSED(rsp)
}

void FMVipDispatcher::onFinished()
{
    if(_windowReturn == -1){
        _serverRspData = Err_WindowClose;
        serverIsBusy.wakeAll();
    }

    _vindow = 0;
}

void FMVipDispatcher::onServerResponsed(const QJsonObject &rspObj)
{   
    int status = rspObj["statusCode"].toInt();
    int prompt = rspObj["prompt"].toInt();

    if(_vindow != 0) {
        _vindow->setIsBusy(false);

        // 支付时， 会员认证的返回，如果不可付款
        int canPay = SESSIONDATA_INT("canPay");

        if(isLogin) {
            // 失败则弹出错误信息
            if(status != 100) {
                _vindow->resetWnd();
                FMMsgWnd::FailureWnd(rspObj["msg"].toString());

                return;
            }
            // 成功，且是最后一步，则弹出认证信息
            else if(isLastOne) {
                QString account = SESSIONDATA_STRING("fm_open_id");
                QString name = SESSIONDATA_STRING("name");
                QString birthday = SESSIONDATA_STRING("birthday");
                FMMsgWnd::LoginSuccess(account, name, birthday);
            }
            // 成功，是支付认证，但不能支付
            else if(isPayLogin) {
                if(canPay != 1) {
                    _vindow->resetWnd();
                    FMMsgWnd::FailureWnd("fmv:请使用支付码付款");
                    return;
                }
                isPayLogin = false;
            }
        }
        isLogin = false;
        _vindow->deleteLater();
    }

    QJsonDocument rspDoc(rspObj);
    QByteArray rspData = rspDoc.toJson();
    _serverRspData = rspData;

    if(status != 100) {
        requestSuccess = false;
        if(prompt == 0)
        {
            QString msg = rspObj["msg"].toString();
            FMMsgWnd::FailureWnd(msg);
        }
        _vindow = 0;
        serverIsBusy.wakeAll();
    }

    if( isLastOne ) {
        serverIsBusy.wakeAll();
    }
}

// 向服务器发送请求
void FMVipDispatcher::onDoPost()
{
    FMVipForward::instance()->parseRequest(_posReqObj);
}

void FMVipDispatcher::login()
{
    isLogin = true;
    if (!_vindow) {
        FMVipLogin *login = new FMVipLogin(isPayLogin);
        connect(login, SIGNAL(destroyed(QObject*)), SLOT(onFinished()));
        connect(login, SIGNAL(doPost()), SLOT(onDoPost()));
        _vindow = login;
        _windowReturn = login->exec();
    }
}

void FMVipDispatcher::coupon()
{
    onDoPost();
}

void FMVipDispatcher::pay()
{
    QJsonObject transactions = _posReqObj["transactions"].toObject();
    int order_amount = transactions["order_amount"].toInt();
    int paid_amount = transactions["paid_amount"].toInt();
    int standard_amount = transactions["standard_amount"].toInt();
    int needPay = order_amount - paid_amount;
    SESSIONDATA_ADD("paid_amount", QString::number(paid_amount));
    SESSIONDATA_ADD("order_amount",QString::number(order_amount));
    SESSIONDATA_ADD("needPay", QString::number(needPay));
    SESSIONDATA_ADD("standard_amount", QString::number(standard_amount));

    if (!_vindow) {
        FMVipOrder *order = new FMVipOrder;
        connect(order, SIGNAL(destroyed(QObject*)), SLOT(onFinished()));
        connect(order, SIGNAL(doPost()), SLOT(onDoPost()));
        _vindow = order;
        _windowReturn = order->exec();
    }

}

void FMVipDispatcher::fund()
{
    SESSIONDATA_ADD("charge_amount", QString::number(_posReqObj["charge_amount"].toInt()));

    if (!_vindow) {
        FMVipFund *fund = new FMVipFund;
        connect(fund, SIGNAL(destroyed(QObject*)), SLOT(onFinished()));
        connect(fund, SIGNAL(doPost()), SLOT(onDoPost()));
        _vindow = fund;
        _windowReturn = fund->exec();
    }
}

void FMVipDispatcher::final()
{
    onDoPost();
}

void FMVipDispatcher::orderRefund()
{
    onDoPost();
}

void FMVipDispatcher::orderRevoke()
{
    onDoPost();
}

BOOL FMVipDispatcher::isLogined()
{
    QString account = SESSIONDATA_STRING("fm_open_id");
    return account != "";
}

void FMVipDispatcher::onCheckSocket()
{
    if(_socket <=0 ) {
        emit stopSocketTimer();
        return;
    }

    WORD wVersionRequested;
    WSADATA wsaData;

    wVersionRequested = MAKEWORD(2, 2);
    WSAStartup(wVersionRequested, &wsaData);

    HANDLE closeEvent = WSACreateEvent();
    WSAEventSelect(_socket, closeEvent, FD_CLOSE);

    DWORD dwRet = WaitForSingleObject(closeEvent, 0);

    if(dwRet == WSA_WAIT_EVENT_0) {
        onDisconnected();
    }

    WSACloseEvent(closeEvent);

    WSACleanup();
}

void FMVipDispatcher::onDisconnected()
{
    if(_vindow != 0) {
        _vindow->close();
    }

    FMMsgWnd::FailureWnd("操作时间过长，POS已断开Socket连接，请重新操作！");
    serverIsBusy.wakeAll();
}
