﻿#include "fmvip.h"
#include "fmlogger.h"
#include <QCryptographicHash>
#include <QJsonParseError>
#include <QJsonDocument>
#include <QJsonObject>
#include <QDateTime>
#include <QDebug>

FMVip::FMVip(QObject *parent) : QObject(parent)
{
}

void FMVip::login(const QString &vipNo, const QString &innerCode, QJsonObject &rspObj)
{
    QJsonObject reqObj;
    reqObj["fm_cmd"]        =   "member_check_request";
    reqObj["member_sign"]   =   vipNo;
    reqObj["inner_code"]    =   innerCode;
    reqObj["member_type"]   =   (int)memberType(vipNo);
    sendToServer(reqObj, rspObj);
}

void FMVip::recharge(const QString &fm_open_id, int amount, QJsonObject &rspObj)
{
    QJsonObject reqObj;
    reqObj["fm_cmd"]        =   "order_recharge_request";
    reqObj["trans_id"]      =   createTransId("117017", "8888");
    QJsonObject transObj;
    transObj["account"]     =   fm_open_id;
    transObj["amount"]     =   amount;
    transObj["cash_amount"] =   amount;
    reqObj["transaction"]   =   transObj;
    sendToServer(reqObj, rspObj);
}

void FMVip::reversal(const QString &fmId)
{
    QJsonObject reqObj;
    reqObj["fm_cmd"]        = "order_correct_request";
    reqObj["fm_id"]         = fmId;
    QJsonObject rspObj;
    sendToServer(reqObj, rspObj);
}

void FMVip::sendToServer(QJsonObject &reqObj, QJsonObject &rspObj)
{
    reqObj["appId"]         =   APP_ID;
    reqObj["business_date"] =   QDate::currentDate().toString("yyyyMMdd");
    reqObj["t"]             =   QString::number(QDateTime::currentMSecsSinceEpoch());
    reqObj["partner_id"]    =   PARTNER_ID;
    reqObj["store_id"]      =   "117017";
    reqObj["pos_id"]        =   "8888";
    reqObj["operator_id"]   =   "8888";

    QJsonDocument jsonDoc(reqObj);

    QString signStr = sign(reqObj);
    QString url = QString("http://member.lxj.sandload.cn/member/service/restful/pos?sign=%1").arg(signStr);

    QByteArray rspData;
    QString error;
    QJsonParseError jsonError;
    QJsonDocument rspJsonDoc;
    if(networker.send(url, jsonDoc.toJson(), rspData, error))
    {
        FM_INFO() << "Server Response:" << QString::fromUtf8(rspData);
        rspJsonDoc = QJsonDocument::fromJson(rspData, &jsonError);
        if(jsonError.error == QJsonParseError::NoError) {
            rspObj = rspJsonDoc.object();
            if(rspObj.contains("errcode")) {
                rspObj["status_code"] = rspObj["errcode"];
                rspObj["msg"] = rspObj["errmsg"];
            }
        } else {
            rspObj["status_code"] = 101;
            rspObj["msg"] = jsonError.errorString();
        }
    } else {
        rspObj["status_code"] = 300;
        rspObj["msg"] = error;
        FM_ERROR() << "Network error: " << error;
    }
}

QString FMVip::createTransId(const QString &storeId, const QString posId) const
{
    QString dateTime = QDateTime::currentDateTime().toString("yyyyMMddhhmmss");
    QString transId = dateTime + storeId + posId;
    return transId;
}

FMVip::MemberType FMVip::memberType(const QString &vipNo) const
{
    switch (vipNo.length()) {
    case 20:
        return FMVip::PayCode;
    case 16:
        return FMVip::EntityCard;
    case 11:
        return FMVip::Phone;
    case 10:
        return FMVip::Account;
    }
    return FMVip::EntityCard;
}

QString FMVip::sign(const QJsonObject &reqJob) const
{
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;

    for (QString word : {"appId","fm_cmd","partner_id","store_id","pos_id","operator_id","t"}) {
        if(reqJob[word].isDouble()) {
            mapData[word] = QString::number(reqJob[word].toDouble());
        } else {
            mapData[word] = reqJob[word].toString();
        }
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KEY_CODE);

    QByteArray bt;
    bt.append(sb);

    FM_INFO() << "Sign String: " << bt;

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}
