﻿#include "cardreader.h"
#include "mwdll/mwrf32.h"
#include "fmsetting.h"
#include <QDebug>

CardReader::CardReader(QObject *parent) : QThread(parent)
{

}

CardReader::~CardReader()
{
}

void CardReader::run()
{
    _isRunning = true;
    int port = FMSetting::GetValue("Port", 0).toInt();
    int baud = FMSetting::GetValue("Baud", 115200).toInt();
    _isRunning &= ConnectDevice(port, baud);

    if(!_isRunning) {
        return;
    }

    def_sector = FMSetting::GetValue("DefaultSector", 4).toInt();
    def_pwd = FMSetting::GetValue("DefaultPwd", "ffffffffffff").toString();
    sector = FMSetting::GetValue("Sector", 1).toInt();
    password = FMSetting::GetValue("Password", "313731383132").toString();

    _isRunning &= LoadKeyHex(def_sector, def_pwd);
    _isRunning &= LoadKeyHex(0, def_pwd);
    _isRunning &= LoadKeyHex(sector, password);
    _isRunning &= LoadKeyHex(15, "000000000000");

    willActiveCard = needActiveCard = "";
    while(_isRunning) {
        OperatorCard();
        Sleep(300);
    }
}

void CardReader::Stop()
{
    _isRunning = false;
}

void CardReader::setError(const QString &error)
{
    _error = error;
    FM_ERROR() << error;
    emit hadError(_error);
}

QString CardReader::LastError() const
{
    return _error;
}

bool CardReader::ConnectDevice(int port, int baud)
{
     _handle = rf_init(port, baud);

     if((int)_handle<=0)
     {
         setError("连接设备失败");
         return false;
     }

    rf_beep(_handle, 30);
    return true;
}

void CardReader::DisConnectDevice()
{
    rf_exit(_handle);
}

bool CardReader::LoadKeyHex(qint8 sec, const QString &password)
{
    char *pwd = password.toLatin1().data();
    if(rf_load_key_hex(_handle, 0, sec, pwd) != 0) {
        setError(QString("%1扇区密码加载失败").arg(sec));
        return false;
    }
    return true;
}

bool CardReader::ReadData(qint8 sec, QString &data, qint8 blockNums, bool isHex)
{
    data.clear();
    // 验证密码
    if(rf_authentication(_handle, 0, sec) != 0) {
        setError(QString("%1扇区密码错误").arg(sec));
        return false;
    }

    for(int i=0; i<blockNums; ++i) {
        if(isHex){
            // 读卡,16进制
            char _data[32];
            ZeroMemory(_data, 32);
            if(rf_read_hex(_handle, sec*4+i, _data) != 0) {
                setError(QString("%1扇区数据读取失败").arg(sec));
                return false;
            }
            data += QString::fromLatin1(_data).trimmed();
        } else {
            unsigned char _data[17];
            ZeroMemory(_data, 17);
            if(rf_read(_handle, sec*4+i, _data) != 0) {
                setError(QString("%1扇区数据读取失败").arg(sec));
                return false;
            }
            data += QString::fromLatin1((char*)_data).trimmed();
        }
    }
    return true;
}

bool CardReader::OperatorCard()
{
    // 检查是否需要激活卡
    if(needActiveCard != "" && needActiveCard==willActiveCard) {
        FM_INFO() << "Active card: " << needActiveCard;
        activedSuccess = Active(needActiveCard);
        FM_INFO() << "Active card success: " << activedSuccess;
        needActiveCard = willActiveCard = lastUUID = "";
        waitActive.wakeAll();
    }
    // 故意鉴权失败，使卡片不用离开读卡器即可再次寻卡
    rf_authentication(_handle, 0, 15);

    bool isActived = true;
    QString data, uuid, vipNo="";
    unsigned long snr;
    if(rf_card(_handle, 0, &snr) != 0) {
        return false;
    }

    if(!ReadData(0, data)) {
        return false;
    }
    uuid = data.mid(0, 10);

    // 判断是否还是上一张卡
    if(uuid == lastUUID) {
        // 同一张卡，直接返回
        return false;
    } else {
        lastUUID = uuid;
        activedSuccess = false;
        needActiveCard = willActiveCard = "";
        FM_INFO() << "Find new card: " << lastUUID;
    }

    // 读默认扇区，看是否是激活过的卡（不用rf_authentcation，因为验证密码出错需重新寻卡）
    if(ReadData(def_sector, data)) {
        if(data == "00000000000000000000000000000000") {
            if(ReadData(sector, data, 2, false)) {
                vipNo = data;
                isActived = true;
            }
        } else {
            // 从def_sector扇区读取账号
            if(ReadData(def_sector, data)){
                vipNo = data.mid(16);
                isActived = false;
                needActiveCard = vipNo;
            }
        }
    }

    FM_INFO() << "Vip No: " << vipNo << " Is actived: " << isActived;

    if(vipNo!="") {
        emit hadVipNo(vipNo, uuid, isActived);
    }

    return false;
}

bool CardReader::ActiveCard(QString vipNo)
{
    mutex.lock();
    willActiveCard = vipNo;
    waitActive.wait(&mutex);
    mutex.unlock();
    return activedSuccess;
}

bool CardReader::Active(const QString &vipNo)
{
    // 将卡号写入sec的扇区
    if(!LoadKeyHex(sector, def_pwd)) {
        return false;
    }
    // 鉴权
    if(rf_authentication(_handle, 0, sector) != 0){
        setError(QString("%1扇区密码错误").arg(sector));
        return false;
    }
    // 写入卡号
    char* writeData = vipNo.toLatin1().toHex().mid(0,32).data();
    if(rf_write_hex(_handle, sector*4, writeData) !=0 ) {
        setError("写入卡号失败");
        return false;
    }
    // 修改密码
    QByteArray tmpData = password.toLatin1();
    unsigned char wkey[7];
    ZeroMemory(wkey, 7);
    a_hex(tmpData.data(), wkey, 12);
    if(rf_changeb3(_handle, sector, wkey, 0, 0, 0, 1, 0, wkey) != 0) {
        setError("修改密码失败");
        return false;
    }

    // 删除原扇区内容
    LoadKeyHex(def_sector, def_pwd);
    // 鉴权
    if(rf_authentication(_handle, 0, def_sector) != 0) {
        setError(QString("%1扇区密码错误").arg(def_sector));
        return false;
    }
    // 初始化为0
    char zeroData[33] = "00000000000000000000000000000000";
    if(rf_write_hex(_handle, def_sector*4, zeroData) !=0 ){
        setError("擦除原数据失败");
        return false;
    }
    rf_beep(_handle, 30);
    rf_halt(_handle);
    return true;
}

void CardReader::ClearInfo()
{
    willActiveCard = needActiveCard = lastUUID = "";
    activedSuccess = false;
}
