﻿#include <QMenu>
#include <QAction>
#include <QTimer>
#include <QJsonDocument>
#include <winsock2.h>
#include "fmvipdispatcher.h"
#include "fmtask.h"
#include "taskfactory.h"
#include "fmvipsetting.h"
#include "taskothers.h"
#include "fmprinter.h"
#include "fmaboutwnd.h"
#include "fmp_redeem.h"

FMVipDispatcher::FMVipDispatcher(QObject *parent) :
    QObject(parent),
    FMApiRelay(),
    _posSocketTimer(new QTimer(this)),
    _confirmRechageTimer(new QTimer(this)),
    fmTask(nullptr)
{
    connect(this, SIGNAL(doTask()), this, SLOT(onDoTask()));

    _posSocketTimer->setInterval(5000);
    connect(_posSocketTimer, SIGNAL(timeout()), SLOT(onCheckSocket()));
    connect(this, SIGNAL(startSocketTimer()), _posSocketTimer, SLOT(start()));
    connect(this, SIGNAL(stopSocketTimer()), _posSocketTimer, SLOT(stop()));

    connect(_confirmRechageTimer, SIGNAL(timeout()), this, SLOT(onConfirmRechageTimeOut()));
    _confirmRechageTimer->setSingleShot(true);
}

FMVipDispatcher::~FMVipDispatcher()
{
    del_p(_posSocketTimer);
    del_p(_confirmRechageTimer);
}

BOOL FMVipDispatcher::Transfer(LPSTR data, LPSTR &rsp, UINT &rsp_len)
{
    if (strcmp(data, QuitCMD) != 0) {
        emit startSocketTimer();
        mutex.lock();
        reqData = QString::fromLocal8Bit(data).toUtf8();
        emit doTask();
        condition.wait(&mutex);
        mutex.unlock();

        QString strData(rspData);
        QByteArray bData = strData.toLocal8Bit();
        rsp = bData.data();
        rsp_len = bData.length();

        emit stopSocketTimer();
    }
    return true;
}

void FMVipDispatcher::onDoTask()
{
    qDebug() << "===============";
    qDebug() << "Recv POS reqData: " << reqData.data();
    
    //-----添加卡券核销功能-----
    QJsonParseError e;
    QJsonDocument doc = QJsonDocument::fromJson(reqData, &e);
    if(e.error != QJsonParseError::NoError)
    {
        FMMsgWnd::FailureWnd("fmv:请求数据解析失败.\n" + e.errorString());
        rspData = "{\"msg:\":\"未知的请求类型\"}";
        return;
    }
    QJsonObject jsonObj = doc.object();
    if(jsonObj["fm_cmd"].toInt() == FM_NotVipCoupon)
    {
        fmTask = new FMPRedeem(jsonObj);
        rspData = fmTask->doTask();
        qDebug() << "Send to pos: " << rspData.data();
        qDebug() << "===============";
    
        condition.wakeAll();
    
        del_p(fmTask);
        return;
    }
    //----------------------------------------

    fmTask = TaskFactory::Task(reqData);

    if (fmTask == nullptr) {
        rspData = "{\"msg:\":\"未知的请求类型\"}";
    } else if(fmTask->FM_Type() == FM_Confirm_Recharge){
        QJsonObject rspObj;
        if(backupReqObj.empty() || backupReqObj[PosProps.Fm_id] != fmTask->getPosJsonValue(PosProps.Fm_id)) {
            QString msg = "未备份该充值订单或非码订单号不一致.";
            qDebug() << msg;
            rspObj[PosProps.RechargeStatus] = FM_API_ERROR;
            rspObj[PosProps.Msg] = msg;
        } else {
            if(fmTask->getPosJsonValue(PosProps.RechargeStatus).toInt() == FM_API_SUCCESS) {
                QString print = backupReqObj[PosProps.Print].toString();
                if(FMPrinter::Print(print)) {
                    rspObj[PosProps.RechargeStatus] = FM_API_SUCCESS;
                } else {
                    qDebug() << "打印小票失败.";
                    rspObj[PosProps.RechargeStatus] = FM_API_ERROR;
                    rspObj[PosProps.Msg] = "打印小票失败";
                }
                _confirmRechageTimer->stop();   // 停止计时
            } else {
                qDebug() << "POS回执不正确，自动冲正上一笔充值.";
                rspObj[PosProps.RechargeStatus] = fmTask->getPosJsonValue(PosProps.RechargeStatus).toInt();
                onConfirmRechageTimeOut();
                _confirmRechageTimer->stop();   // 停止计时
            }
        }

        rspObj[PosProps.StatusCode] = FM_API_SUCCESS;
        rspObj[PosProps.Prompt] =  0;
        rspObj[PosProps.Fm_id] = fmTask->getPosJsonValue(PosProps.Fm_id).toString();
        QJsonDocument doc(rspObj);
        rspData = doc.toJson(QJsonDocument::Compact);
        backupReqObj = QJsonObject();
    }else {
        rspData = fmTask->doTask();

        if(fmTask->FM_Type() == FM_Manage) {
            QJsonObject posRspJsonObj = *(fmTask->getPosRspJsonObj());
            if(!posRspJsonObj.empty() && posRspJsonObj.contains(PosProps.FundInfo)) {
                backupReqObj = *(fmTask->getPosReqJsonObj());
                backupReqObj[PosProps.TransId] = fmTask->session()->data(PosProps.TransId).toString();
                backupReqObj[PosProps.Fm_id] = fmTask->session()->data(PosProps.Fm_id).toString();
                backupReqObj[PosProps.Print] = fmTask->session()->data(PosProps.Print).toString();
                // 启动定时器,30秒后对该充值订单进行冲正
                if(_confirmRechageTimer->isActive()) {
                    onConfirmRechageTimeOut();
                    _confirmRechageTimer->stop();
                }
                _confirmRechageTimer->start(30000);
            }
        }
    }
    qDebug() << "Send to pos: " << rspData.data();
    qDebug() << "===============";

    condition.wakeAll();

    del_p(fmTask);
}

void FMVipDispatcher::onCheckSocket()
{
    if(_socket <=0 ) {
        emit stopSocketTimer();
        return;
    }

    WORD wVersionRequested;
    WSADATA wsaData;

    wVersionRequested = MAKEWORD(2, 2);
    WSAStartup(wVersionRequested, &wsaData);

    HANDLE closeEvent = WSACreateEvent();
    WSAEventSelect(_socket, closeEvent, FD_CLOSE);

    DWORD dwRet = WaitForSingleObject(closeEvent, 0);

    if(dwRet == WSA_WAIT_EVENT_0) {
        onDisconnected();
    }

    WSACloseEvent(closeEvent);

    WSACleanup();
}

void FMVipDispatcher::onDisconnected()
{
    fmTask->stopTask();
    FMMsgWnd::FailureWnd("操作时间过长，POS已断开Socket连接，请重新操作！");
    condition.wakeAll();
}

void FMVipDispatcher::createSysIcon()
{
    QIcon icon = QIcon(":/FreemudPOS.ico");
    _sysIcon.setIcon(icon);
    _sysIcon.setToolTip("FMVIP");
    auto menu = new QMenu();
    auto settingAction = new QAction("设置", this);
    connect(settingAction, SIGNAL(triggered(bool)), this, SLOT(onSetting()));
    menu->addAction(settingAction);
    auto aboutAction = new QAction("关于", this);
    connect(aboutAction, SIGNAL(triggered(bool)), this, SLOT(onAbout()));
    menu->addAction(aboutAction);
    auto quitAction = new QAction("退出", this);
    connect(quitAction, SIGNAL(triggered(bool)), this, SLOT(onQuit()));
    menu->addAction(quitAction);
    _sysIcon.setContextMenu(menu);

    _sysIcon.show();

    _sysIcon.showMessage("非码VIP提醒", "非码VIP程序已启动");
}

void FMVipDispatcher::onQuit()
{
    if(fmTask != nullptr) {
        fmTask->stopTask();
    }
    qDebug() << __FUNCTION__;
    LPSTR data = QuitCMD;
    unsigned int a;
    Transfer(data, data, a);
    qApp->quit();
}

void FMVipDispatcher::onSetting()
{
    FMVipSetting settingWnd;
    settingWnd.exec();
}

void FMVipDispatcher::onAbout()
{
    FMAboutWnd aboutWnd;
    aboutWnd.About();
}

void FMVipDispatcher::onConfirmRechageTimeOut()
{
    if(!backupReqObj.empty()) {
        qDebug() << "冲正备份的请求：" << backupReqObj;
        FMTask *reversalTask = new TaskRefundOrder(backupReqObj, this);
        reversalTask->doTask();
        reversalTask->deleteLater();
        backupReqObj = QJsonObject();
    } else {
        qDebug() << "备份的请求为空，未执行冲正.";
    }
}
