﻿#include "taskpay.h"
#include "tasklogin.h"
#include "fmviporder.h"
#include "fmp_logger_i.h"
#include "fmp_vip_settings.h"
#include "fmnetwork.h"
#include "taskcoupon.h"
#include <QJsonDocument>
#include <QJsonObject>
#include <QCryptographicHash>

#define RunFunction(function)   do {    \
        function##();                   \
        if(error() != FM_API_SUCCESS) { \
            FMP_WARN() << QString("Task error %1: %2").arg(error()).arg(errorString());   \
            return QString(ErrorMsgJson).arg(error()).arg(errorString()).toLatin1();    \
        }                               \
} while(0);

TaskPay::TaskPay(QJsonObject &jsonObj, Session *session, QObject *parent)
    :QObject(parent)
{
    posReqJsonObj = jsonObj;
    _window = nullptr;
    _session = session;
    _error = FM_API_SUCCESS;
}

TaskPay::~TaskPay()
{
    del_p(_window);
}

QByteArray TaskPay::doTask()
{    
    FMP_DEBUG() << __FUNCTION__;
    //获取可用的代金券
    TaskCoupon taskCoupon(posReqJsonObj, _session);
    QByteArray ret = taskCoupon.doTask();
    if(taskCoupon.error() != FM_API_SUCCESS)
    {
        return ret;
    }
    
    //查询余额和积分
    ret = queryMemberInfo();
    if(error() != FM_API_SUCCESS)
    {
        return ret;
    }
    //保存到session，用于界面显示
    session()->addData("amount", getServerJsonValue("amount"));
    session()->addData("score", getServerJsonValue("score"));
    session()->addData("name", getServerJsonValue("name"));
    
    //清空相应json对象
    QStringList keys = serverReqJsonObj.keys();
    foreach (QString key, keys) 
    {
        serverReqJsonObj.remove(key);
    }
    keys = serverRspJsonObj.keys();
    foreach(QString key, keys)
    {
        serverRspJsonObj.remove(key);
    }
    
    RunFunction(setWindow);                   
    RunFunction(showWindow);
    RunFunction(packagePOSRsp);
    
    QJsonDocument json(posRspJsonObj);
    return json.toJson(QJsonDocument::Compact);
}

QByteArray TaskPay::queryMemberInfo()
{
    serverReqJsonObj["appId"] = APP_ID;
    serverReqJsonObj["reqType"] = FM_Login;
    serverReqJsonObj["stationId"] = getPosJsonValue("pos_id");
    serverReqJsonObj["storeId"] = getPosJsonValue("store_id");
    serverReqJsonObj["operatorId"] = getPosJsonValue("operator_id");
    serverReqJsonObj["partnerId"] = PARTNER_ID;
    serverReqJsonObj["code"] = getPosJsonValue("fm_open_id");
    serverReqJsonObj["t"] = QString::number(QDateTime::currentMSecsSinceEpoch());
    
    QJsonDocument json(serverReqJsonObj);
    QByteArray data = json.toJson(QJsonDocument::Compact);

    url = QString("%1?sign=%2").arg(FMPVipSettings::instance()->getServerUrl()).arg(sign());
    QByteArray rspData;

    FMNetwork net;
    net.send(url, data, rspData);

    // 网络错误
    if(net.error != FM_API_SUCCESS) {
        setError(net.error, net.errorMsg);
    } else {
        QJsonParseError jsonErr;
        QJsonDocument rspJson = QJsonDocument::fromJson(rspData, &jsonErr);

        // Json错误
        if(jsonErr.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON);
        } else {
            serverRspJsonObj = rspJson.object();

            // 服务器返回的错误
            if(serverRspJsonObj.contains("errcode")) {
                setError(FM_API_SERVERERROR, serverRspJsonObj["errcode"].toInt(), serverRspJsonObj["errmsg"].toString());
            }
        }
    }

    FMP_INFO() << "Server rsponse: " << serverRspJsonObj;

    return rspData;
}

void TaskPay::setWindow()
{
    _window = new FMVipOrder;
    session()->addData("operator_id", getPosJsonValue("operator_id").toString());
    session()->addData("order_amount", getPosJsonValue("order_amount").toInt());
    session()->addData("store_id", getPosJsonValue("store_id").toString());
    session()->addData("pos_id", getPosJsonValue("pos_id").toString());
    session()->addData("pay_amount", getPosJsonValue("order_amount").toInt() - getPosJsonValue("paid_amount").toInt());
    session()->addData("business_date", getPosJsonValue("business_date").toString());
    session()->addData("fm_open_id", getPosJsonValue("fm_open_id").toString());
    session()->addData("products", getPosJsonValue("products").toArray());

    connect(qobject_cast<FMVipOrder*>(_window), SIGNAL(pay()), this, SLOT(onPay()));
}

void TaskPay::showWindow()
{
    FMP_DEBUG() << __FUNCTION__;

    if(_window != nullptr) {
        _window->initWnd(this->session());
        int ret = _window->exec();
        if(ret == -1) {
            setError(FM_API_WINDOWCLOSE);
        }
    }
}

void TaskPay::packageServerReq()
{
    serverReqJsonObj["appId"] = APP_ID;
    serverReqJsonObj["reqType"] = FM_Pay;
    serverReqJsonObj["stationId"] = getPosJsonValue("pos_id");
    serverReqJsonObj["storeId"] = getPosJsonValue("store_id");
    serverReqJsonObj["operatorId"] = getPosJsonValue("operator_id");
    serverReqJsonObj["transId"] = getPosJsonValue("trans_id");
    serverReqJsonObj["businessDate"] = getPosJsonValue("business_date");
    serverReqJsonObj["partnerId"] = PARTNER_ID;

    QJsonObject transaction;
    transaction["memberTransId"] = "";
    transaction["account"] = getPosJsonValue("fm_open_id");
    transaction["payAmount"] = getPosJsonValue("order_amount").toInt() - getPosJsonValue("paid_amount").toInt();
    transaction["isUseScore"] = session()->data("isUseScore").toInt();

    QJsonArray products;
    QJsonArray clientArray = getPosJsonValue("products").toArray();
    for(int i = 0; i < clientArray.size(); i++)
    {
        QJsonObject obj = clientArray[i].toObject();
        QJsonObject serverObj;
        serverObj["consumeNum"] = obj["consume_num"];
        serverObj["price"] = obj["price"].toInt();
        serverObj["productId"] = obj["pid"];
        products.push_back(serverObj);
    }

    QJsonArray payList;
    //余额支付详情
    QJsonObject balance;
    balance["amount"] = session()->data("codeAmount").toDouble();
    balance["transId"] = getPosJsonValue("trans_id");
    balance["typeModeFlag"]  = 20001;
    balance["code"] = "";
    payList.push_back(balance);

    //积分支付详情
    if(transaction["isUseScore"].toInt())
    {
        QJsonObject score;
        score["amount"] = session()->data("score").toInt();
        score["transId"] = getPosJsonValue("trans_id");
        score["typeModeFlag"] = 20002;
        score["code"] = "";
        payList.push_back(score);
    }

    //代金券/商品券支付详情
    QMap<QString, QVariant> coupons = session()->data("payCouponMap").toMap();
    for(QMap<QString, QVariant>::const_iterator it = coupons.constBegin(); it != coupons.constEnd(); it++)
    {
        QJsonObject coupon;
        coupon["amount"] = it.value().value<Coupon>().disAmount;
        coupon["transId"] = getPosJsonValue(PosProps.TransId);
        coupon["typeModeFlag"] = it.value().value<Coupon>().typeModeFlag;
        coupon["code"] = it.value().value<Coupon>().code;
        payList.push_back(coupon);
    }

    transaction["products"] = products;
    transaction["payList"] = payList;

    serverReqJsonObj[PosProps.Transaction] = transaction;
}

bool TaskPay::sendToServer(bool isShowMsg)
{
    FMP_DEBUG() << __FUNCTION__;
    packageServerReq();

    QJsonDocument json(serverReqJsonObj);
    QByteArray data = json.toJson(QJsonDocument::Compact);

    url = QString("%1?sign=%2").arg(FMPVipSettings::instance()->getServerUrl()).arg(sign());
    QByteArray rspData;

    FMNetwork net;
    net.send(url, data, rspData);

    // 网络错误
    if(net.error != FM_API_SUCCESS) {
        setError(net.error, net.errorMsg);
    } else {
        QJsonParseError jsonErr;
        QJsonDocument rspJson = QJsonDocument::fromJson(rspData, &jsonErr);

        // Json错误
        if(jsonErr.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON);
        } else {
            serverRspJsonObj = rspJson.object();
        }
    }

    FMP_INFO() << "Server rsponse: " << serverRspJsonObj;


    if(_window != nullptr) {
        _window->setIsBusy(false);
    }

    bool isOk = (error() == FM_API_SUCCESS);
    if(!isOk && isShowMsg)
    {
        FMMsgWnd::FailureWnd(errorString(), _window);
    }

    return isOk;
}

void TaskPay::packagePOSRsp()
{
    posRspJsonObj[PosProps.StatusCode] = getServerJsonValue("statusCode");
    posRspJsonObj[PosProps.Msg] = getServerJsonValue("msg");
    posRspJsonObj[PosProps.Prompt] = 0;
    posRspJsonObj[PosProps.Settlement] = 1;
    posRspJsonObj[PosProps.Fm_open_id] = getServerJsonValue("account").toString();
    posRspJsonObj[PosProps.Fm_transId] = getServerJsonValue("memberTransId").toString();

    QJsonArray servPayArray = getServerJsonValue("payList").toArray();
    //支付方式描述
    QMap<QString, QString> description;
    description["20001"] = "会员储值金支付";
    description["20002"] = "会员积分支付";
    description["20003"] = "代金券支付";
    description["20004"] = "商品券支付";
    description["20005"] = "现金支付";

    //计算总支付额
    QJsonArray pay_ids;
    int totalPaid = 0;
    for(int i = 0; i < servPayArray.size(); i++)
    {
        QJsonObject pay_id;
        pay_id["pay_id"] = QString::number(servPayArray[i].toObject()["typeModeFlag"].toInt());
        pay_id["pay_str"] = description[QString::number(servPayArray[i].toObject()["typeModeFlag"].toInt())];
        pay_id["pay_amount"] = servPayArray[i].toObject()["amount"].toInt();
        pay_id["code"] = servPayArray[i].toObject()["code"].toString();
        pay_ids.push_back(pay_id);

        totalPaid += pay_id["pay_amount"].toInt();
    }
    posRspJsonObj["paid_total_amount"] = totalPaid;
    posRspJsonObj["pay_ids"] = pay_ids;
}

void TaskPay::onPay()
{
    sendToServer();
    _window->close();
}

QJsonValue TaskPay::searchJsonValue(QJsonObject &searchJson, QString searchKey)
{
    if(searchJson.contains(searchKey)) {
        return searchJson[searchKey];
    } else {
        foreach(QString key , searchJson.keys()) {
            if(searchJson[key].isObject()) {
                QJsonObject ob = searchJson[key].toObject();
                QJsonValue value = searchJsonValue(ob, searchKey);
                if(!value.isNull()){
                    return value;
                }
            }
        }
    }
    return QJsonValue();
}

QJsonValue TaskPay::getServerJsonValue(const QString prop)
{
    return searchJsonValue(serverRspJsonObj, prop);
}

QJsonValue TaskPay::getPosJsonValue(const QString prop)
{
    return searchJsonValue(posReqJsonObj, prop);
}

QString TaskPay::sign() const
{
    FMP_DEBUG() << __FUNCTION__;
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;

    for(int i=0; i<sizeof(SignProps)/sizeof(SignProps[0]); ++i) {
        QString word = SignProps[i];
        if(serverReqJsonObj[word].isDouble()) {
            mapData[word] = QString::number(serverReqJsonObj[word].toDouble());
        } else {
            mapData[word] = serverReqJsonObj[word].toString();
        }
    }

    if (serverRspJsonObj[ServerProps(PosProps.Fm_cmd)].toInt() == FM_Fund) {
        mapData[ServerProps(PosProps.TransId)] = serverReqJsonObj[ServerProps(PosProps.TransId)].toString();
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KEY_CODE);

    QByteArray bt;
    bt.append(sb);

    FMP_INFO() << "Sign String: " << bt;

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}
