﻿#include "tasklogin.h"
#include "fmviplogin.h"
#include "fmp_vip_settings.h"
#include "fmnetwork.h"
#include <QJsonDocument>
#include <QCryptographicHash>

#define RunFunction(function)   do {    \
        function##();                   \
        if(error() != FM_API_SUCCESS) { \
            FMP_WARN() << QString("Task error %1: %2").arg(error()).arg(errorString());   \
            return QString(ErrorMsgJson).arg(error()).arg(errorString()).toLatin1();    \
        }                               \
} while(0);

TaskLogin::TaskLogin(QJsonObject &jsonObj, Session *session, QObject *parent)
    :QObject(parent)
{
    posReqJsonObj = jsonObj;
    _window = NULL;
    _session = session;
    _error = FM_API_SUCCESS;
}

TaskLogin::~TaskLogin()
{
    del_p(_window);
}

QByteArray TaskLogin::doTask()
{
    FMP_DEBUG() << __FUNCTION__;
    RunFunction(setWindow);
    RunFunction(showWindow);
    RunFunction(packagePOSRsp);

    QJsonDocument json(posRspJsonObj);
    return json.toJson(QJsonDocument::Compact);
}

void TaskLogin::setWindow()
{
    _window = new FMVipLogin;

    connect(qobject_cast<FMVipLogin*>(_window), &FMVipLogin::login, this, &TaskLogin::onLogin);
    session()->addData("store_id", getPosJsonValue("store_id"));
    session()->addData("pos_id", getPosJsonValue("pos_id"));
    session()->addData("operator_id", getPosJsonValue("operator_id"));
    session()->addData("business_date", getPosJsonValue("business_date"));
}

void TaskLogin::showWindow()
{
    FMP_DEBUG() << __FUNCTION__;

    if(_window != nullptr) {
        _window->initWnd(session());
        int ret = _window->exec();
        if(ret == -1) {
            setError(FM_API_WINDOWCLOSE);
        }
    }
}

QJsonValue TaskLogin::searchJsonValue(QJsonObject &searchJson, QString searchKey)
{
    if(searchJson.contains(searchKey)) {
        return searchJson[searchKey];
    } else {
        qDebug() << searchJson.keys();
        foreach(QString key , searchJson.keys()) {
            if(searchJson[key].isObject()) {
                QJsonObject ob = searchJson[key].toObject();
                QJsonValue value = searchJsonValue(ob, searchKey);
                if(!value.isNull()){
                    return value;
                }
            }
        }
    }
    return QJsonValue();
}

QJsonValue TaskLogin::getServerJsonValue(const QString prop)
{
    return searchJsonValue(serverRspJsonObj, prop);
}

QJsonValue TaskLogin::getPosJsonValue(const QString prop)
{
    return searchJsonValue(posReqJsonObj, prop);
}

void TaskLogin::packageServerReq()
{
    serverReqJsonObj["appId"] = APP_ID;
    serverReqJsonObj["reqType"] = FM_Login;
    serverReqJsonObj["stationId"] = getPosJsonValue("pos_id");
    serverReqJsonObj["storeId"] = getPosJsonValue("store_id");
    serverReqJsonObj["operatorId"] = getPosJsonValue("operator_id");
    serverReqJsonObj["partnerId"] = PARTNER_ID;
    serverReqJsonObj["code"] = session()->data("code").toString();
    serverReqJsonObj["t"] = QString::number(QDateTime::currentMSecsSinceEpoch());
}

bool TaskLogin::sendToServer(bool isShowMsg)
{
    FMP_DEBUG() << __FUNCTION__;
    packageServerReq();

    QJsonDocument json(serverReqJsonObj);
    QByteArray data = json.toJson(QJsonDocument::Compact);

    url = QString("%1?sign=%2").arg(FMPVipSettings::instance()->getServerUrl()).arg(sign());
    QByteArray rspData;

    FMNetwork net;
    net.send(url, data, rspData);

    // 网络错误
    if(net.error != FM_API_SUCCESS) {
        setError(net.error, net.errorMsg);
    } else {
        QJsonParseError jsonErr;
        QJsonDocument rspJson = QJsonDocument::fromJson(rspData, &jsonErr);

        // Json错误
        if(jsonErr.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON);
        } else {
            serverRspJsonObj = rspJson.object();
        }
    }

    FMP_INFO() << "Server rsponse: " << serverRspJsonObj;

    if(_window != nullptr) {
        _window->setIsBusy(false);
    }

    bool isOk = (error() == FM_API_SUCCESS);
    if(!isOk && isShowMsg)
    {
        FMMsgWnd::FailureWnd(errorString(), _window);
    }

    return isOk;
}

void TaskLogin::packagePOSRsp()
{
    posRspJsonObj["statusCode"] = getServerJsonValue("statusCode").toInt();
    posRspJsonObj["msg"] = getServerJsonValue("msg").toString();
    posRspJsonObj["prompt"] = getServerJsonValue("prompt").toInt();
    posRspJsonObj["fm_open_id"] = getServerJsonValue("account").toString();
    posRspJsonObj["account"] = getServerJsonValue("memberNo").toString();
    posRspJsonObj["type_code"] = getServerJsonValue("typecode").toString();
    posRspJsonObj["type_name"] = getServerJsonValue("typename").toString();
    posRspJsonObj["amount"] = getServerJsonValue("amount").toInt();
    posRspJsonObj["score"] = getServerJsonValue("score").toInt();
    posRspJsonObj["name"] = getServerJsonValue("name").toString();
    posRspJsonObj["sex"] = getServerJsonValue("sex").toString();
    posRspJsonObj["birthday"] = getServerJsonValue("birthday").toString();
    posRspJsonObj["phone"] = getServerJsonValue("mobile").toString();
    posRspJsonObj["address"] = getServerJsonValue("address").toString();
    posRspJsonObj["email"] = getServerJsonValue("email").toString();
}

void TaskLogin::onLogin()
{
    bool isOk = sendToServer();
    if(!isOk) {
        if(_window != nullptr) {
            _window->close();
        }
        return;
    }

    // 认证失败
    if(getServerJsonValue("statusCode").toInt() != FM_API_SUCCESS) {
        QString info = serverRspJsonObj["msg"].toString();
        FMP_WARN() << "Login failed: " << info;
        FMMsgWnd::FailureWnd(info, _window);
    }
    // 认证成功
    else {
        QString account = getServerJsonValue("account").toString();
        QString birthday = getServerJsonValue("birthday").toString();
        QString name = getServerJsonValue("name").toString();
        FMMsgWnd::LoginSuccess(account, name, birthday, _window);
        _window->close();
    }
    if(_window!=nullptr) {
        _window->resetWnd();
    }
}

QString TaskLogin::sign() const
{
    FMP_DEBUG() << __FUNCTION__;
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;

    for(int i=0; i<sizeof(SignProps)/sizeof(SignProps[0]); ++i) {
        QString word = SignProps[i];
        if(serverReqJsonObj[word].isDouble()) {
            mapData[word] = QString::number(serverReqJsonObj[word].toDouble());
        } else {
            mapData[word] = serverReqJsonObj[word].toString();
        }
    }

    if (serverRspJsonObj[ServerProps(PosProps.Fm_cmd)].toInt() == FM_Fund) {
        mapData[ServerProps(PosProps.TransId)] = serverReqJsonObj[ServerProps(PosProps.TransId)].toString();
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KEY_CODE);

    QByteArray bt;
    bt.append(sb);

    FMP_INFO() << "Sign String: " << bt;

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}