#include "taskcoupon.h"
#include "fmnetwork.h"
#include "fmp_logger_i.h"
#include "fmmsgwnd.h"
#include "fmp_vip_settings.h"
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QCryptographicHash>

TaskCoupon::TaskCoupon(QJsonObject &jsonObj, Session *session, QObject *parent) : QObject(parent)
{
    posReqJsonObj = jsonObj;
    _session = session;
    _error = FM_API_SUCCESS;
}

QByteArray TaskCoupon::doTask()
{
    bool isOk = sendToServer();
    if(!isOk) {
        FMP_WARN() << QString("Task error %1: %2").arg(error()).arg(errorString());
        return QString(ErrorMsgJson).arg(error()).arg(errorString()).toLatin1();
    }
    session()->addData("fm_open_id", getServerJsonValue("account"));
    
    QMap<QString, QVariant> couponMap;
    foreach (auto value , getServerJsonValue("coupon_list").toArray())
    {
        Coupon c;
        QJsonObject couponOb = value.toObject();
        QJsonArray applyProducts = couponOb["apply_products"].toArray();
        for(int i = 0; i < applyProducts.size(); i++)
        {
            ApplyProduct p;
            p.price = applyProducts[i].toObject()["price"].toInt();
            p.coupon_discount_amount = applyProducts[i].toObject()["coupon_discount_amount"].toInt();
            p.pid = applyProducts[i].toObject()["pid"].toString();
            p.consume_num = applyProducts[i].toObject()["consume_num"].toInt();
            c.applyProducts.push_back(p);
        }
        c.code = couponOb["code"].toString();
        c.limit_amount = couponOb["limit_amount"].toInt();
        c.dis_price = couponOb["dis_price"].toInt();
        c.desc = couponOb["name"].toString();
        c.compatible_flag = couponOb["compatible_flag"].toInt();
        c.disAmount = couponOb["dis_amount"].toInt();
        c.typeModeFlag = couponOb["typeModeFlag"].toInt();
        couponMap[c.code] = QVariant::fromValue(c);
    }
    session()->addData("payCouponMap", couponMap);
    packagePOSRsp();
    return QJsonDocument(posRspJsonObj).toJson(QJsonDocument::Compact);
}

void TaskCoupon::packageServerReq()
{
    serverReqJsonObj["appId"] = APP_ID;
    serverReqJsonObj["reqType"] = FM_Coupon;
    serverReqJsonObj["stationId"] = getPosJsonValue("pos_id");
    serverReqJsonObj["storeId"] = getPosJsonValue("store_id");
    serverReqJsonObj["operatorId"] = getPosJsonValue("operator_id");
    serverReqJsonObj["transId"] = getPosJsonValue("trans_id");
    serverReqJsonObj["businessDate"] = getPosJsonValue("business_date");
    serverReqJsonObj["partnerId"] = PARTNER_ID;
    serverReqJsonObj["t"] = QString::number(QDateTime::currentMSecsSinceEpoch());
    QJsonObject transaction;
    transaction["memberTransId"] = "";
    transaction["account"] = getPosJsonValue("fm_open_id");
    transaction["isUseScore"] = 1;
    transaction["payAmount"] = getPosJsonValue("order_amount").toInt() - getPosJsonValue("paid_amount").toInt();
    QJsonArray products;
    QJsonArray clientArr = getPosJsonValue("products").toArray();
    for(int i = 0; i < clientArr.size(); i++)
    {
        QJsonObject product;
        product["consumeNum"] = clientArr[i].toObject()["consume_num"];
        product["price"] = clientArr[i].toObject()["price"];
        product["productId"] = clientArr[i].toObject()["pid"];
        products.append(product);
    }
    transaction["products"] = products;
    serverReqJsonObj["transaction"] = transaction;
}

bool TaskCoupon::sendToServer(bool isShowMsg)
{
    FMP_DEBUG() << __FUNCTION__;
    packageServerReq();

    QJsonDocument json(serverReqJsonObj);
    QByteArray data = json.toJson(QJsonDocument::Compact);

    url = QString("%1?sign=%2").arg(FMPVipSettings::instance()->getServerUrl()).arg(sign());
    QByteArray rspData;

    FMNetwork net;
    net.send(url, data, rspData);

    // 网络错误
    if(net.error != FM_API_SUCCESS) {
        setError(net.error, net.errorMsg);
    } else {
        QJsonParseError jsonErr;
        QJsonDocument rspJson = QJsonDocument::fromJson(rspData, &jsonErr);

        // Json错误
        if(jsonErr.error != QJsonParseError::NoError) {
            setError(FM_API_BADJSON);
        } else {
            serverRspJsonObj = rspJson.object();

            // 服务器返回的错误
            if(serverRspJsonObj.contains("errcode")) {
                setError(FM_API_SERVERERROR, serverRspJsonObj["errcode"].toInt(), serverRspJsonObj["errmsg"].toString());
            }
        }
    }

    FMP_INFO() << "Server rsponse: " << serverRspJsonObj;

    bool isOk = (error() == FM_API_SUCCESS);
    if(!isOk && isShowMsg)
    {
        FMMsgWnd::FailureWnd(errorString(), nullptr);
    }

    return isOk;
}

void TaskCoupon::packagePOSRsp()
{
    posRspJsonObj["fm_open_id"] = getServerJsonValue("account");
    posRspJsonObj["statusCode"] = getServerJsonValue("status_code");
    posRspJsonObj["msg"] = getServerJsonValue("msg");
}

QJsonValue TaskCoupon::searchJsonValue(QJsonObject &searchJson, QString searchKey)
{
    if(searchJson.contains(searchKey)) {
        return searchJson[searchKey];
    } else {
        qDebug() << searchJson.keys();
        foreach(QString key , searchJson.keys()) {
            if(searchJson[key].isObject()) {
                QJsonObject ob = searchJson[key].toObject();
                QJsonValue value = searchJsonValue(ob, searchKey);
                if(!value.isNull()){
                    return value;
                }
            }
        }
    }
    return QJsonValue();
}

QJsonValue TaskCoupon::getServerJsonValue(const QString prop)
{
    return searchJsonValue(serverRspJsonObj, prop);
}

QJsonValue TaskCoupon::getPosJsonValue(const QString prop)
{
    return searchJsonValue(posReqJsonObj, prop);
}

QString TaskCoupon::sign() const
{
    FMP_DEBUG() << __FUNCTION__;
    // 解析JSON插入MAP中按字典排序
    QMap<QString, QString> mapData;

    for(int i=0; i<sizeof(SignProps)/sizeof(SignProps[0]); ++i) {
        QString word = SignProps[i];
        if(serverReqJsonObj[word].isDouble()) {
            mapData[word] = QString::number(serverReqJsonObj[word].toDouble());
        } else {
            mapData[word] = serverReqJsonObj[word].toString();
        }
    }

    if (serverRspJsonObj[ServerProps(PosProps.Fm_cmd)].toInt() == FM_Fund) {
        mapData[ServerProps(PosProps.TransId)] = serverReqJsonObj[ServerProps(PosProps.TransId)].toString();
    }

    // 使用URL键值对的格式拼接
    QString sb = "";
    foreach(QString key , mapData.keys())
    {
        sb += (key + "=" + mapData.value(key) + "&");
    }
    sb.remove(sb.length() - 1, 1);      // 去掉最后一个&

    sb.append(KEY_CODE);

    QByteArray bt;
    bt.append(sb);

    FMP_INFO() << "Sign String: " << bt;

    QByteArray md5Bt = QCryptographicHash::hash(bt, QCryptographicHash::Md5);
    return md5Bt.toHex();
}
