﻿#include "fmp_syncer_p.h"
#include "fmp_syncer_def.h"

#include <fmp_network_i.h>
#include <fmp_settings_i.h>
#include <fmp_logger_i.h>
#include <fmp_home_i.h>

#include <QJsonObject>
#include <QJsonDocument>
#include <QJsonArray>
#include <QApplication>
#include <QFile>
#include <QDir>
#include <QTimer>
#include <QEventLoop>
#include <QCryptographicHash>
#include <QProcess>
#include "JlCompress.h"

#include "../fmp_manager/version.h"

FMPSyncerPrivate::FMPSyncerPrivate(FMPSyncer *q)
    : _inited(false),
      sets(0),
      nw(0),
      q_ptr(q)
{
}

int FMPSyncerPrivate::Init()
{
    if (_inited) return FMP_SUCCESS;

    Q_Q(FMPSyncer);
    sets = FMP::GetService<FMPSettingsInterface>();
    nw = FMP::GetService<FMPNetworkInterface>();

//    if (_sets && _logger && _nw) {
        _inited = true;
        //! 启动线程
        start();

        return FMP_SUCCESS;
//    } else {
//        return FMP_FAILURE;
//    }
}

int FMPSyncerPrivate::Uninit()
{
    if (!_inited) return FMP_SUCCESS;

    quit();

    _inited = false;

    return FMP_SUCCESS;
}

QString FMPSyncerPrivate::GetPartnerId()
{
    return _GetValue(FMP_INIKEY_LOGINPARTNERID, FMPCOMMON_DEFAULT_PARTNERID).toString();
}

QString FMPSyncerPrivate::GetStoreId()
{
    return _GetValue(FMP_INIKEY_LOGINSTOREID, FMPCOMMON_DEFAULT_STOREID).toString();
}

QString FMPSyncerPrivate::GetPosId()
{
    return _GetValue(FMP_INIKEY_LOGINPOSID, FMPCOMMON_DEFAULT_POSID).toString();
}

int FMPSyncerPrivate::GetRequestTimeout()
{
    return _GetValue(FMP_INIKEY_NW_REQTIMEOUT, FMPNW_DEFAULT_TIMEOUT).toInt() * 1000;
}

int FMPSyncerPrivate::GetSyncInterval()
{
    return _GetValue(FMP_INIKEY_SYNCINTERVAL, FMPSYNC_DEFAULT_INTERVAL).toInt() * 1000;
}

QString FMPSyncerPrivate::GetTaskUrl()
{
    return _GetValue(FMP_INIKEY_SYNCTASKURL, FMPSYNC_DEFAULT_TASKURL).toString();
}

QString FMPSyncerPrivate::GetTaskReportUrl()
{
    return _GetValue(FMP_INIKEY_SYNCTASKRPTURL, FMPSYNC_DEFAULT_TASKRPTURL).toString();
}

QString FMPSyncerPrivate::GetDownloadPath()
{
    return _GetValue(FMP_INIKEY_DOWNLOADPATH, FMPSYNC_DEFAULT_DOWNLOADPATH).toString();
}

QString FMPSyncerPrivate::GetDeployPath()
{
    return _GetValue(FMP_INIKEY_DEPLOYPATH, FMPSYNC_DEFAULT_DEPLOYPATH).toString();
}

void FMPSyncerPrivate::SetStoreId(const QString &id)
{
    _SetValue(FMP_INIKEY_LOGINSTOREID, id);
}

void FMPSyncerPrivate::SetPosId(const QString &id)
{
    _SetValue(FMP_INIKEY_LOGINPOSID, id);
}

void FMPSyncerPrivate::SetRequestTimeout(int sec)
{
    _SetValue(FMP_INIKEY_NW_REQTIMEOUT, sec);
}

void FMPSyncerPrivate::SetSyncInterval(int sec)
{
    _SetValue(FMP_INIKEY_SYNCINTERVAL, sec);
}

void FMPSyncerPrivate::SetTaskUrl(const QString &url)
{
    _SetValue(FMP_INIKEY_SYNCTASKURL, url);
}

void FMPSyncerPrivate::SetTaskReportUrl(const QString &url)
{
    _SetValue(FMP_INIKEY_SYNCTASKRPTURL, url);
}

void FMPSyncerPrivate::SetDownloadPath(const QString &path)
{
    _SetValue(FMP_INIKEY_DOWNLOADPATH, path);
}

void FMPSyncerPrivate::SetDeployPath(const QString &path)
{
    _SetValue(FMP_INIKEY_DEPLOYPATH, path);
}

QVariant FMPSyncerPrivate::_GetValue(const QString &k, QVariant default_val)
{
    if (!sets) {
        Q_Q(FMPSyncer);
        sets = FMP::GetService<FMPSettingsInterface>();
    }

    if (sets) {
        return sets->GetValue(k, default_val);
    }
    else {
        FMP_WARN() << "Settings service not available";
    }

    return default_val;
}

bool FMPSyncerPrivate::_SetValue(const QString &k, QVariant val)
{
    if (!sets) {
        Q_Q(FMPSyncer);
        sets = FMP::GetService<FMPSettingsInterface>();
    }

    if (sets) {
        sets->SetValue(k, val);
        return true;
    }
    else {
        FMP_WARN() << "Settings service not available";
    }

    return false;
}

void FMPSyncerPrivate::run()
{
    Q_Q(FMPSyncer);
    if (QFileInfo(q->_deploy_path).isAbsolute()) {
        _deploy_dir = q->_deploy_path;
    }
    else {
        _deploy_dir = QApplication::applicationDirPath() + "/" + q->_deploy_path;
    }
    int sync_interval = q->_sync_interval;
    QString task_id;

    while(1) {
        FMP_INFO() << QString("check update after %1 ms").arg(sync_interval);
        QEventLoop eventLoop;
        QTimer::singleShot(sync_interval, &eventLoop, &QEventLoop::quit);
        eventLoop.exec();

        //! 如果有更新结果向服务器汇报结果
        ReportTask();

        //! 向服务器发送检查更新的请求
        QJsonObject task_json;
        if (!GetTask(task_json)) {
            continue;
        }
        sync_interval = task_json[FMP_JSONKEY_NEXTCHECK].toInt() * 1000;
        task_id = QString::number(task_json[FMP_JSONKEY_PLANID].toInt());

#ifdef QT_DEBUG
        //!TEST
//        task_json["NeedUpdate"] = "yes";
//        task_json["UpdatePath"] = "http://192.168.110.150:8080/UpgradeFiles/1101";
//        task_json.insert("NeedUpload", "yes");
//        task_json.insert("UploadFiles", "img/yb.jpg|img/dlb.jpg");
        //!TEST
#endif

        //! 先判断是否需要上传文件(优先处理)
        if(task_json[FMP_JSONKEY_NEEDUPLOAD].toString() == "yes") {
            QStringList filenames = task_json[FMP_JSONKEY_UPLOADFILES].toString().split("|");
            //! 汇报上传结果
            ReportUploadTask(UploadFiles(filenames));
        }

        //! 判断是否有更新
        if(task_json[FMP_JSONKEY_NEEDUPDATE].toString() == "yes") {;}
        else { continue; }

        //! 找出差异文件
        QMap<QString, QString> latest_files;
        QString download_url = task_json[FMP_JSONKEY_UPDATEPATH].toString();
        if (!GetLatestFiles(download_url, latest_files)) {
            ReportTask(task_id, "Client files are up-to-date.", "1");
            continue;
        }

        //! 下载差异文件
        QStringList all_file_names = latest_files.keys();
        QStringList downloaded_files = DownloadFiles(download_url, latest_files);
        if(all_file_names == downloaded_files) {
            FMP_INFO() << QString("Download [%1] files successful").arg(latest_files.count());

            //! 记录升级任务编号
            QFile planIdFile(QString("%1/%2/planid.txt").arg(_deploy_dir).arg(q->_download_path));
            if(planIdFile.open(QIODevice::WriteOnly)) {
                planIdFile.write(QString::number(task_json[FMP_JSONKEY_PLANID].toInt()).toLatin1());
                planIdFile.close();
            }

            //! 将需要替换的文件写入 dfilesInfo.txt
            QFile file(QString("%1/%2/dfilesInfo.txt").arg(_deploy_dir).arg(q->_download_path));
            if(file.open(QIODevice::WriteOnly)) {
                QTextStream stream(&file);
                foreach(QString f, downloaded_files) {
                    stream << f << ",";
                }

                file.close();

                QString sync_helper = _GetValue(FMP_INIKEY_SYNCHELPER, "FreemudSyncer").toString();
                QStringList params;
                params << FMP::GetProperties()[FMP_PROPKEY_PROCID].toString();
                QProcess::startDetached(sync_helper, params);
            }
        }
        else {
            //! 如果失败汇报错误
            QString failed_files;
            foreach(QString f, all_file_names) {
                if (downloaded_files.contains(f)) continue;
                failed_files += f + ",";
            }

            ReportTask(task_id, "Failed downloading files:" + failed_files.section(",", 0, -2), "0");
        }
    }
}

void FMPSyncerPrivate::ReportTask(const QString& tid, const QString& msg, const QString& state)
{
    QString task_id, task_msg, task_state;
    Q_Q(FMPSyncer);
    QFile resultFile(QString("%1/%2/updateresult.txt").arg(_deploy_dir).arg(q->_download_path));
    if(resultFile.exists() && tid.isEmpty()) {
        if(tid.isEmpty()) {
            QFile file(QString("%1/%2/planid.txt").arg(_deploy_dir).arg(q->_download_path));
            if(file.open(QIODevice::ReadOnly)) {
                task_id = QString(file.readAll());
                file.close();
            }
            file.setFileName(QString("%1/%2/updateresult.txt").arg(_deploy_dir).arg(q->_download_path));
            if(file.open(QIODevice::ReadOnly)) {
                QString readStr = file.readAll();
                task_state = readStr.mid(0, 1);
                task_msg = readStr.mid(1);
                file.close();
            }
        }
    }
    else if (!tid.isEmpty()){
        task_id = tid;
        task_msg = msg;
        task_state = state;

        Q_Q(FMPSyncer);
        QJsonObject report_req_jo;
        report_req_jo.insert(FMP_JSONKEY_PARTNERID, q->_partner_id);
        report_req_jo.insert(FMP_JSONKEY_STOREID, q->_store_id);
        report_req_jo.insert(FMP_JSONKEY_POSNO, q->_pos_id);
        report_req_jo.insert(FMP_JSONKEY_PLANID, task_id);
        report_req_jo.insert(FMP_JSONKEY_UPDATESTATE, task_state);
        report_req_jo.insert(FMP_JSONKEY_MSG, task_msg);
        QByteArray report_json = QJsonDocument(report_req_jo).toJson(QJsonDocument::Compact);

        FMP_INFO() << QString("report [%1]").arg(QString(report_json));
        FMPHttpReplyPointer reply = nw->HttpPost(q->_task_report_url, report_json);
        if(reply->WaitResponse() == FMPHttpReply::NO_ERROR) {
            QString report_rsp = reply->Response();
            FMP_INFO() << QString("report successful [%1]").arg(QString(report_rsp));
            resultFile.remove();
        }
        else {
            FMP_ERROR() << QString("report failed [%1]").arg(reply->ErrorString());
        }
    }
}

void FMPSyncerPrivate::ReportUploadTask(bool)
{
    //! TODDO: 上传任务汇报实现

    Q_Q(FMPSyncer);
    QJsonObject rObj;
    rObj.insert(FMP_JSONKEY_PARTNERID, q->_partner_id);
    rObj.insert(FMP_JSONKEY_STOREID, q->_store_id);
    rObj.insert(FMP_JSONKEY_POSNO, "1");
    rObj.insert(FMP_JSONKEY_PLANID, "planId");
    rObj.insert(FMP_JSONKEY_UPLOADSTATE, "state");
    rObj.insert(FMP_JSONKEY_MSG, "msg");
    QJsonDocument(rObj).toJson(QJsonDocument::Compact);
}

bool FMPSyncerPrivate::GetTask(QJsonObject &task_jo)
{
    Q_Q(FMPSyncer);
    bool has_task = false;

    QJsonObject task_req_jo;
    task_req_jo.insert(FMP_JSONKEY_PARTNERID, q->_partner_id);
    task_req_jo.insert(FMP_JSONKEY_STOREID, q->_store_id);
    task_req_jo.insert(FMP_JSONKEY_POSNO, q->_pos_id);
    //! TODO: 版本怎么传
    task_req_jo.insert(FMP_JSONKEY_VERSION, RES_STR_PRODUCT_VER);

    QByteArray task_req = QJsonDocument(task_req_jo).toJson(QJsonDocument::Compact);
    FMP_INFO() << QString("Checking update [%1]").arg(QString(task_req));
    FMPHttpReplyPointer reply = nw->HttpPost(q->_task_url, task_req);
    if( reply->WaitResponse() == FMPHttpReply::NO_ERROR ) {
        QByteArray task_json = reply->Response();
        FMP_DEBUG() << "Task json:" << task_json;
        task_jo = QJsonDocument().fromJson(task_json).object();
        if (task_jo[FMP_JSONKEY_STATUSCODE].toInt() == FMP_SYNCTASK_SUCCESS) {
            FMP_INFO() << QString("Check update successful [%1]").arg(QString(task_json));
            has_task = true;
        }
        else {
            FMP_ERROR() << QString("check update failed [%1 - %2]")
                                  .arg(QString::number(task_jo[FMP_JSONKEY_STATUSCODE].toInt()))
                                  .arg(task_jo[FMP_JSONKEY_MSG].toString());
        }
    }
    else {
        FMP_ERROR() << QString("check update failed [%1]").arg(reply->ErrorString());
    }

    return has_task;
}

bool FMPSyncerPrivate::UploadFiles(const QStringList &files)
{
    bool upload_success = false;
    QStringList uploaded_file;
    foreach(QString f, files) {
        QString file_path = QString("%1/%2").arg(_deploy_dir, f);
        if(!JlCompress::compressFile(file_path + ".zip", file_path)) {
            FMP_ERROR() << QString("Compress file[%1] failed").arg(file_path);
            continue;
        }
        QFile file(file_path + ".zip");
        if(!file.open(QFile::ReadOnly)) {
            FMP_ERROR() << QString("Open file[%1] failed").arg(file_path+".zip");
            continue;
        }

        QByteArray data = file.readAll();
        file.close();

        Q_Q(FMPSyncer);
        QString url = QString("%1/%2_%3_%4").arg(q->_upload_url, q->_store_id, QDate::currentDate().toString("yyyyMMdd"), QFileInfo(file_path + ".zip").fileName());
        FMPHttpReplyPointer reply = nw->HttpPut(url, data);
        if (reply->WaitResponse() == FMPHttpReply::NO_ERROR) {
            //! TODO:
            //! 文件上传判断是否成功
            QByteArray rsp = reply->Response();
            FMP_INFO() << QString("Upload file[%1] successful").arg(file_path + ".zip");
            uploaded_file << f;
        }
        else {
            FMP_ERROR() << QString("Upload file[%1] failed. (%2)").arg(file_path + ".zip", reply->ErrorString());
        }
    }

    upload_success = (uploaded_file == files);

    return upload_success;
}

bool FMPSyncerPrivate::GetLatestFiles(const QString &url, QMap<QString, QString> &latest_files)
{
    bool has_latest = false;
    FMP_INFO() << QString("Get latest files on server [%1]").arg(url);

    FMPHttpReplyPointer reply = nw->HttpGet(url + "/upgradeInfos.txt");
    if (reply->WaitResponse() == FMPHttpReply::NO_ERROR) {
        QByteArray rsp = reply->Response();
        FMP_DEBUG() << "Get server files response:" << rsp;
        QJsonArray file_array = QJsonDocument().fromJson(rsp).array();

        QMap<QString, QString> all_files;
        foreach(QJsonValue jv, file_array) {
            QJsonObject fileObj = jv.toObject();
            all_files[fileObj[FMP_JSONKEY_PATH].toString()] = fileObj[FMP_JSONKEY_MD5].toString();
        }
        if (!all_files.isEmpty()) {
            if(RetainLatestFiles(all_files, latest_files)) {
                has_latest = true;
            }
            else {
                FMP_INFO() << "Local files are up-to-date";
            }
        }
        else {
            FMP_WARN() << "No files on server";
        }
    }
    else {
        FMP_ERROR() << "Get latest files failed." << reply->ErrorString();
    }

    return has_latest;
}

bool FMPSyncerPrivate::RetainLatestFiles(const QMap<QString, QString> &all_files, QMap<QString, QString> &latest_files)
{
    QStringList keys = all_files.keys();
    foreach(QString k, keys) {
        QFile file(QString("%1/%2").arg(_deploy_dir, k));
        if(file.open(QFile::ReadOnly)) {
            QString local_md5 = QCryptographicHash::hash(file.readAll(),QCryptographicHash::Md5).toHex();
            if( local_md5 == all_files[k]) {
                file.close();
                continue;
            }
            file.close();
        }
        latest_files[k] = all_files[k];
    }

    return (latest_files.count() > 0);
}

QString FMPSyncerPrivate::GetFileMd5(const QString &file_path)
{
    QString md5;
    QFile file(file_path);
    if(file.open(QFile::ReadOnly)) {
        md5 = QCryptographicHash::hash(file.readAll(), QCryptographicHash::Md5).toHex();
        file.close();
    }
    else {
        FMP_INFO() << "Failed open file" << file_path << file.errorString();
    }

    return md5;
}

QStringList FMPSyncerPrivate::DownloadFiles(const QString &url, const QMap<QString, QString> files)
{
    QStringList keys = files.keys();
    QStringList downloaded_files;
    Q_Q(FMPSyncer);
    foreach (QString k, keys) {
        QString download_url = QString("%1/%2.zip").arg(url, k);
        QString save_path = QString("%1/%2/%3.zip").arg(_deploy_dir, q->_download_path, k);
        FMP_INFO() << QString("Start downloading file [%1]").arg(download_url);

        //! 下载之前先检测文件是否下载过
        QString local_md5 = GetFileMd5(save_path.left(save_path.lastIndexOf(".")));
        if(local_md5 == files[k]) {
            FMP_INFO() << "file exists";
            downloaded_files << k;
            continue;
        }
        else {
            FMP_WARN() << "Md5 not match, local:" << local_md5 << "remote:" << files[k];
        }

        //! 下载文件
        QDir().mkpath(save_path.left(save_path.lastIndexOf("/")));
        QFile file(save_path);
        if(!file.open(QIODevice::WriteOnly)) {
            FMP_ERROR() << file.errorString();
            continue;
        }
        else {
            FMPHttpReplyPointer reply = nw->HttpGet(download_url);
            if (reply->WaitResponse() == FMPHttpReply::NO_ERROR) {
                file.write(reply->Response());
            }
            file.close();
        }

        //! 解压
        JlCompress::extractDir(save_path, save_path.left(save_path.lastIndexOf("/")));

        //! 检查下载的文件MD5是否正确
        local_md5 = GetFileMd5(save_path.left(save_path.lastIndexOf(".")));
        if(local_md5 != files[k]) {
            FMP_ERROR() << "Md5 not match, local:" << local_md5 << "remote:" << files[k];
        }
        else {
            FMP_INFO() << "download file successful";
            downloaded_files << k;
        }
    }

    return downloaded_files;
}

