﻿#include "fmp_redeem.h"
#include "scanningdialog.h"
#include "consumptiondialog.h"
#include "consumokdialog.h"
#include "errcodedialog.h"
#include "fmp_logger_i.h"
#include "fmp_pe_handlers.h"
#include "fmp_settings_i.h"
#include "fmp_epay_i.h"
#include "fmp_home_i.h"
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonParseError>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QEventLoop>
#include <QTimer>
#include <QJsonArray>
#include <QApplication>
#include <QResource>
#include <QSettings>
#include <QMessageBox>

FMPRedeem::FMPRedeem(const FMPContext context)
    :FMPRedeemInterface(),
      _context(context),
     _inited(false),
     _clientReqCount(0),
     _ste_handler(new FMPStartEventHandler(this))
{
    //注册资源
    QString strpath= QApplication::applicationDirPath();
    QString rccPath = QString("%1/../rcc/%2.rcc").arg(strpath).arg("redeem");
    qDebug()<<"rccPath is"<<rccPath;
    if(!QFile(rccPath).exists())
    {
        FMP_DEBUG() << rccPath << " not exists";
    }
    QResource::registerResource(rccPath);
}

FMPRedeem::~FMPRedeem()
{
    StopService();
}

void FMPRedeem::InitService()
{
    if (_inited) return;
    _inited = true;
}

void FMPRedeem::UninitService()
{
}

QJsonValue FMPRedeem::SearchJsonObject(QJsonObject& searchJson, QString searchKey)
{
    QJsonValue value;
    if(searchJson.contains(searchKey))
    {
        return searchJson[searchKey];
    }
    else
    {
        foreach(QString key, searchJson.keys())
        {
            if(searchJson[key].isObject())
            {
                QJsonObject ob = searchJson[key].toObject();
                value = SearchJsonObject(ob, searchKey);
            }
            else if(searchJson[key].isArray())
            {
                QJsonArray arr = searchJson[key].toArray();
                value = SearchJsonArray(arr, searchKey);
            }
        }
    }
    return value;
}

QJsonValue FMPRedeem::SearchJsonArray(QJsonArray& searchJson, QString searchKey)
{
    QJsonValue value;
    for(int i = 0; i < searchJson.size(); i++)
    {
        if(searchJson[i].isObject())
        {
            QJsonObject ob = searchJson[i].toObject();
            value = SearchJsonObject(ob, searchKey);
        }
        else if(searchJson[i].isArray())
        {
            QJsonArray arr = searchJson[i].toArray();
            value = SearchJsonArray(arr, searchKey);
        }
    }
    return value;
}

QJsonObject FMPRedeem::Redeem(const double needPay, const QJsonArray& productsInfo)
{    
    FMP_DEBUG() << "Recv redeem data: " << QJsonDocument(productsInfo).toJson(QJsonDocument::Compact);
    _products_info = productsInfo;
    _needPay = needPay;
    //读取门店信息
    QString apppath = QCoreApplication::applicationDirPath();
    QSettings settings(QString("%1/FreemudPOS.ini").arg(apppath), QSettings::IniFormat);
    _store_id = settings.value(FMP_INIKEY_LOGINSTOREID).toString();
    _station_id = settings.value(FMP_INIKEY_LOGINPOSID).toString();
    _operator_id = settings.value(FMP_INIKEY_LOGINCASHIER).toString();
    _url = settings.value(FMP_INIKEY_EPAYURL).toString();
    _partner_id = settings.value(FMP_INIKEY_LOGINPARTNERID).toInt();
    
    //显示扫码界面获取券码
    ScanningDialog scanningDialog;
    if(scanningDialog.exec() != QDialog::Accepted)
    {
        QJsonObject ret;
        ret["statusCode"] = FM_API_WINDOWCLOSE;
        ret["msg"] = "窗口关闭";
        FMP_DEBUG() << QJsonDocument(ret).toJson(QJsonDocument::Compact);
        return ret;
    }
    _coupon = scanningDialog.code;
    
    QJsonObject json;
    json.insert("ver", 2);
    json.insert("clientReqCount", ++_clientReqCount%=10000000);
    json.insert("reqtype", 0);
    json.insert("partnerId", _partner_id);
    json.insert("store_id", _store_id);
    json.insert("station_id", _station_id);
    json.insert("operator_id", _operator_id);
    json.insert("coupon", _coupon);
    QByteArray reqData = QJsonDocument(json).toJson(QJsonDocument::Compact);
    reqData = CheckSendArray(reqData);
    
    QJsonObject retJson = SendRequest(reqData);
    int statusCode = retJson["statusCode"].toInt();
    switch (statusCode) 
    {
    case 100: //成功
        return ShowForUnConsum(retJson);
    case 11: //无效码
        return ShowForInvalid(retJson);
    case 12: //过期码
        return ShowForExpird(retJson);
    case 14: //已核销
        return ShowForHasConsum(retJson);
    default: //出错    
        return ShowForErr(retJson);
    }
}

QJsonObject FMPRedeem::SendRequest(const QByteArray &reqData)
{
    FMP_DEBUG() << "Send data: " << reqData;
    
    QJsonObject retJson;
    QNetworkAccessManager nam;
    QNetworkRequest req;
    QNetworkReply* reply = nullptr;
    
    QSslConfiguration config;
    QList<QSslCertificate> certs = QSslCertificate::fromPath(qApp->applicationDirPath() + "/client01.pem");
    config.setPeerVerifyMode(QSslSocket::VerifyNone);
    config.setProtocol(QSsl::SslV3);
    config.setCaCertificates(certs);

    req.setSslConfiguration(config);
    req.setUrl(_url);
    req.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    req.setHeader(QNetworkRequest::ContentLengthHeader, reqData.length());
    
    reply = nam.post(req, reqData);
    reply->deleteLater();
    
    // 使用定时器处理超时
    QEventLoop loop;
    connect(&nam, &QNetworkAccessManager::finished, &loop, &QEventLoop::quit);
    QTimer timer;
    timer.setSingleShot(true);
    connect(&timer, &QTimer::timeout, &loop, &QEventLoop::quit);
    timer.start(1000 * 30);
    loop.exec();
    
    if(timer.isActive())
    {
        timer.stop();
        if(reply->error() == QNetworkReply::NoError)
        {
            QByteArray respons = reply->readAll();
            retJson = QJsonDocument::fromJson(respons).object();
        }
        else
        {
            //网络错误
            retJson["statusCode"] = FM_API_NETWORKERROR;
            retJson["msg"] = reply->errorString();
        }
    }
    else
    {
        //超时
        disconnect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
        reply->abort();
        retJson["statusCode"] = FM_API_TIMEOUT;
        retJson["msg"] = QString::fromLocal8Bit("请求超时");
    }
    
    FMP_DEBUG() << "Server response: " << QJsonDocument(retJson).toJson(QJsonDocument::Compact);
    return retJson;
}

QJsonObject FMPRedeem::ShowForUnConsum(QJsonObject json)
{
    QJsonObject codeInfo = json["codeInfo"].toObject();
    QString act_name = codeInfo["act_name"].toString();
    QString code_name = codeInfo["act_id"].toString();
    QString vdata = codeInfo["vdata"].toString();
    QString coupon = codeInfo["code"].toString();
    QString ebcode = codeInfo["ebcode"].toString();
    QString time_name = QString::fromLocal8Bit("有效期至:    ").append(vdata);
    int couponType = json["couponType"].toInt();

    if( ConsumptionDialog::ShowForUnConsum(act_name, code_name,time_name,coupon))
    {
        QJsonObject redeemJson;
        redeemJson["ver"] = 2;
        redeemJson["clientReqCount"] = (++_clientReqCount%=10000000);
        redeemJson["reqtype"] = 71;
        redeemJson["partnerId"] = _partner_id;
        redeemJson["store_id"] = _store_id;
        redeemJson["station_id"] = _station_id;
        redeemJson["trans_id"] = QString::number(QDateTime::currentDateTimeUtc().toTime_t());
        redeemJson["operator_id"] = _operator_id;
        redeemJson["business_date"] = QDate::currentDate().toString("yyyyMMdd");
    
        if( couponType == 0){ //商品券
            if(!IsContinue(SearchJsonObject(json, "paid").toInt()))
            {
                QJsonObject result;
                result["statusCode"] = FM_API_WINDOWCLOSE;
                result["msg"] = QString::fromLocal8Bit("窗口关闭");
                return result;
            }
            
            QJsonObject transaction;
            QJsonArray transactions;
            transaction["code"] = coupon;
            transaction["ebcode"] = ebcode;
            
            //得到该券所对应商品id，目前一张券只对应一种商品
            QString canUsedProduct = SearchJsonObject(json, "pid").toString();
            
            //遍历传入的商品信息，查找是否有相同的商品id，
            //若有则进行核销，否则，直接返回失败
            int i;
            for(i = 0; i < _products_info.size(); i++){
                QJsonObject obj = _products_info[i].toObject();
                if(obj["pid"].toString() == canUsedProduct)
                    break;
            }
            if(i >= _products_info.size())
            {
                //不存在对应商品
                return ShowForMismatch(json);
            }
            
            QJsonArray products;
            QJsonObject product;
            product.insert("seq", 1);
            product.insert("pid", _products_info[i].toObject()["pid"]);
            product.insert("consume_num", _products_info[i].toObject()["consume_num"]);
            products.append(product);
            
            transaction["products"] = products;
            transactions.append(transaction);
            redeemJson["transactions"] = transactions;
        }
        if( couponType == 1){  //代金券
            if(!IsContinue(SearchJsonObject(json, "amount").toInt()))
            {
                QJsonObject result;
                result["statusCode"] = FM_API_WINDOWCLOSE;
                result["msg"] = QString::fromLocal8Bit("窗口关闭");
                return result;
            }
            
            QJsonObject transaction;
            transaction["code"] = coupon;
            transaction["ebcode"] = ebcode;
    
            QJsonArray transactions;
            transactions.append(transaction);
    
            redeemJson["transactions"] = transactions;
        }
        QByteArray reqData = QJsonDocument(redeemJson).toJson();
        reqData = CheckSendArray(reqData);
        
        QJsonObject redeemResult;
        QJsonObject retJson = SendRequest(reqData);
        int statusCode = retJson["statusCode"].toInt();
        switch(statusCode)
        {
        case 100:
        {
            QString _time = QDateTime::currentDateTime().toString("yyyy-MM-dd");
            QString _time_name;
            _time_name.append(_store_id).append("(").append(_station_id).append( ")").append("     ").append(_time);
            ConsumOkDialog::showConsumOk( act_name, code_name, _time_name, coupon);

            if(couponType == 0) //商品券
            {
                redeemResult["discount"] = SearchJsonObject(json, "paid").toInt()/100;
            }
            else if(couponType == 1) //代金券
            {
                redeemResult["discount"] = SearchJsonObject(json, "amount").toInt()/100;
            }
        }
            break;
        default:
            ErrCodeDialog::showForErr(coupon, QString("error:").append(QString::number(statusCode)));
            break;
        }
        redeemResult["statusCode"] = retJson["statusCode"];
        redeemResult["msg"] = retJson["msg"].toString();
        return redeemResult;
    }
    else
    {
        QJsonObject retJson;
        retJson["statusCode"] = FM_API_WINDOWCLOSE;
        retJson["msg"] = QString::fromLocal8Bit("窗口关闭");
        return retJson;
    }
}

QJsonObject FMPRedeem::ShowForHasConsum(QJsonObject json)
{
    QString storeInfo = QString("%1(%2)").arg(_store_id).arg(_station_id);
    ConsumptionDialog::ShowForHasConsum(json["msg"].toString(), "", storeInfo, _coupon);
    return json;
}

QJsonObject FMPRedeem::ShowForInvalid(QJsonObject json)
{
    ErrCodeDialog::showForInvalid(_coupon);
    return json;
}

QJsonObject FMPRedeem::ShowForExpird(QJsonObject json)
{
    ErrCodeDialog::showForExpird("", "", "", _coupon);
    return json;
}

QJsonObject FMPRedeem::ShowForErr(QJsonObject json)
{
    int statusCode = json["statusCode"].toInt();
    ErrCodeDialog::showForErr(_coupon, QString("error: ").append(QString::number(statusCode)));
    return json;
}

QJsonObject FMPRedeem::ShowForMismatch(QJsonObject json)
{
    ErrCodeDialog::showForMismatch(_coupon, QString::fromLocal8Bit("券和商品不匹配"));
    QJsonObject mismatch;
    mismatch["statusCode"] = 31;
    mismatch["msg"] = QString::fromLocal8Bit("券和商品不匹配");
    return mismatch;
}

bool FMPRedeem::IsContinue(int couponAmount)
{
    if(_needPay < couponAmount/100.0)
    {
        //如果卡券抵扣额已大于待付金额，则给出警告提示
        QMessageBox msg;
        msg.setWindowTitle(QString::fromLocal8Bit("警告"));
        msg.setText(QString::fromLocal8Bit("卡券总额已超出应付金额！是否继续？"));
        msg.setStandardButtons(QMessageBox::Yes|QMessageBox::Cancel);
        msg.setIcon(QMessageBox::Warning);
        //msg.setWindowModality(Qt::ApplicationModal);
        msg.setWindowFlags(msg.windowFlags() | Qt::WindowStaysOnTopHint);
        msg.setAttribute(Qt::WA_QuitOnClose, false);
        int choice = msg.exec();
        if(choice != QMessageBox::Yes)
        {
            return false;
        }
    }
    return true;
}

int FMPRedeem::ZH_caclChkValue(char *pJsonData, int startPos, int endPos)
{
    /* startPos <= && < endPos */
    int value = 0, loop = startPos, count = 0;
    for (; loop < endPos; loop++)
    {
        if( pJsonData[loop] == '"' || pJsonData[loop] == '\'')
            continue;

        if (2 > count){
            value += (int)pJsonData[loop];
        }
        else if  (2 == count){
            value *= (int)pJsonData[loop];
        }else{
            value -= (int)pJsonData[loop];
        }
        count = (count + 1) & 3;
    }
    return value;
}

QByteArray FMPRedeem::CheckSendArray(QByteArray &jsonArray)
{
    char tmpBuf[4096] = {0};

    strcpy(tmpBuf, jsonArray.data());

    int strLength = strlen(tmpBuf);

    while('}' != tmpBuf[strLength - 1])
    {
        strLength--;
    }
    strLength--;

    sprintf(tmpBuf + strLength, ", \"checkValue\":");

    int aLength = strlen(tmpBuf + strLength);

    int nValue = ZH_caclChkValue( tmpBuf, 0, strLength + 2);
    strLength += aLength;

    sprintf(tmpBuf + strLength, " %d", nValue);

    aLength = strlen(tmpBuf + strLength);

    strLength += aLength;

    tmpBuf[strLength] = '}';
    tmpBuf[strLength + 1] = '\0';
    strLength ++;

    QByteArray array(tmpBuf);

    return array;
}