﻿#ifndef FMP_MANAGER_I_H
#define FMP_MANAGER_I_H

#include <QSharedPointer>
#include <QPluginLoader>
#include <QDir>
#include <QFileInfo>
#include <QDirIterator>
#include <QJsonDocument>
#include <QCoreApplication>
#include <fmp_base_i.h>
#include <ctkPluginContext.h>
#include "fmp_manager_d.h"

class FMPManagerInterface;
typedef FMPManagerInterface* FMPManager;
typedef ctkPluginContext*   FMPContext;


/**
 * 非业务插件接口，必须继承自 BaseInterface
 * @brief The FMPManagerInterface class
 */
class FMPManagerInterface : public FMPBaseInterface
{
public:
    virtual long        GetPluginId() const { return GetContext()->getPlugin()->getPluginId(); }
    /**
     * 加载插件目录下的所有新插件
     * 首次运行时，所有插件都视为新插件
     * @brief LoadNewPlugins
     * @return
     */
    virtual void        LoadNewPlugins() = 0;

    /**
     * 加载并启动插件 install, start
     * 插件生命周期开始，使用插件注册的服务前，还需要调用 StartService
     * @brief LoadPlugin
     * @param plugin 插件名称
     * @return
     */
    virtual int         LoadPlugin(const QString &plugin) = 0;

    /**
     * 请求更新旧版本插件
     * @brief UpgradeOldPlugins
     * @param pids
     * @return
     */
    virtual void        UpgradeOldPlugins(const QVariantList &pids) = 0;

    /**
     * 升级指定的插件
     * @brief UpdatePlugin
     * @param pid 插件ID
     * @return
     */
    virtual int         UpgradePlugin(long pid) = 0;

    /**
     * 获取服务
     * @brief GetService
     * @return
     */
    template <class T>
    T*                  GetService()
    {
        T* svc = nullptr;
        FMPContext ctx = GetContext();
        if (ctx) {
            try {
                ctkServiceReference ref = ctx->getServiceReference<T>();
                if (!ref) {
                    if (LoadPlugin(qobject_interface_iid<T*>()) == FMP_SUCCESS) {
                        svc = GetService<T>();
                    }
                }
                else {
                    svc = ctx->getService<T>(ref);
                }
            }
            catch(...) {}

        }
        return svc;
    }
    
    /**
     * 根据服务名称获取服务
     * @brief GetService
     * @param svcName
     * @return
     */
    template <class T>
    T*                  GetService(const QString &svcName)
    {
        T* svc = nullptr;
        FMPContext ctx = GetContext();
        if (ctx) {
            try {
                ctkServiceReference ref = ctx->getServiceReference(svcName.toUtf8().data());
                if (!ref) {
                    ref = ctx->getServiceReference<T>();
                }
                if (!ref) {
                    if (LoadPlugin(svcName) == FMP_SUCCESS) {
                        svc = GetService<T>(svcName);
                    }
                }
                else {
                    svc = ctx->getService<T>(ref);
                }
            }
            catch(...) {}
        }

        return svc;
    }

    /**
     * 注册服务
     * @brief RegisterService
     * @param service
     * @param props
     */
    template <class T>
    void                RegisterService(QObject *service, const FMPProps &props)
    {
        FMPContext ctx = GetContext();
        if (ctx) {
            ctx->registerService<T>(service, props);
        }
    }
};

Q_DECLARE_INTERFACE(FMPManagerInterface, "fmp.manager")



///////////////////////////////////////////////////////////////////////////////
/**
 * 插件管理模块加载类
 * @brief The FMP class
 */
class FMP
{
public:
    static FMP&         Instance()
    {
        static FMP fmp;
        return fmp;
    }

    static FMPManager   GetManager()
    {
        FMP& fmp = FMP::Instance();
        FMPManager svc = fmp.Service();
        return svc;
    }

    template <class T>
    static T*           GetService()
    {
        FMPManager manager = FMP::GetManager();
        return manager->GetService<T>();
    }

    template <class T>
    static T*           GetService(const QString &svcName)
    {
        FMPManager manager = FMP::GetManager();
        return manager->GetService<T>(svcName);
    }

    template <class T>
    static void             RegisterService(QObject *svc, const FMPProps &props)
    {
        FMPManager manager = FMP::GetManager();

        manager->RegisterService<T>(svc, props);
    }

    static void             SetProperties(const QJsonObject &p)
    {
        FMP & fmp = FMP::Instance();
        if (p != fmp.props) {
            fmp.Unload();
            fmp.props = p;
            fmp.Load();
        }
    }

    static QJsonObject&     GetProperties()
    {
        FMP& fmp = FMP::Instance();
        fmp.props.empty();
        return fmp.props;
    }

private:
    explicit    FMP() : svc(nullptr)
    {
        props[FMP_PROPKEY_LOADER] = "fmp.manager";
        props[FMP_PROPKEY_PLUGINPATH] = "../plugins";
        props[FMP_PROPKEY_ENTRY] = "fmp.home";
        props[FMP_PROPKEY_CFG] = "FreemudPOS.ini";

        QFile f(qApp->applicationDirPath() + "/" + props[FMP_PROPKEY_CFG].toString());
        if (f.exists() && f.open(QFile::ReadWrite)) {
            QJsonDocument d = QJsonDocument::fromJson(f.readAll());
            if (!d.isEmpty()) {
                props = d.object();
            }
        }

        //!
        props[FMP_PROPKEY_PROCID] = qApp->applicationPid();

        Load();
    }

    void        Load()
    {
        QString loaderPath = GetPluginPath( props[FMP_PROPKEY_LOADER].toString());
        loader.setFileName(loaderPath);
        bool loaded = loader.load();
        if (!loaded) {
            QString msg = "Failed loading plugin:\n";
            msg += loaderPath + "\n";
            msg += loader.errorString();
            Q_ASSERT_X(loaded, __FILE__, msg.toLocal8Bit().data());
        }

        FMPManagerInterface *fmpManager = qobject_cast<FMPManagerInterface*>(loader.instance());
        Q_ASSERT(fmpManager);

        svc = fmpManager;
    }

    void        Unload() { loader.unload(); }

    FMPManager  Service() const { return svc; }

    QString     GetPluginPath(const QString &pluginName)
    {
        QString pluginFileName(pluginName);
        pluginFileName.replace(".", "_");
        QStringList filters;
#ifdef Q_OS_WIN
        filters << pluginFileName + "*" + ".dll";
#else
#   ifdef Q_OS_LINUX
        filters << pluginFileName + "*" + ".so";
#   else
        filters << pluginFileName + "*" + ".dylib";
#   endif
#endif
        QDirIterator dirIter(qApp->applicationDirPath() + "/" + props[FMP_PROPKEY_PLUGINPATH].toString(), filters, QDir::Files);
        QString pluginVersion;
        QFileInfo pluginFileInfo;
        while(dirIter.hasNext()) {
            dirIter.next();
            QFileInfo fileInfo = dirIter.fileInfo();
            QString fileBasename = fileInfo.completeBaseName();
            if (fileBasename.startsWith("lib")) fileBasename = fileBasename.mid(3);
            QString tmpVer = fileBasename.section(pluginFileName, 1, 1).replace("_", "");
            if (pluginVersion.isEmpty() || pluginVersion < tmpVer) {
                pluginVersion = tmpVer;
                pluginFileInfo = fileInfo;
            }
        }

        return pluginFileInfo.canonicalFilePath();
    }

private:
    FMPManager      svc;
    QJsonObject     props;
    QPluginLoader   loader;
};

#endif // FMP_MANAGER_I_H
