﻿#include "fmp_manager.h"
#include "fmp_me_handlers.h"
#include <ctkPluginFramework.h>
#include <ctkPluginContext.h>
#include <ctkPluginFrameworkLauncher.h>
#include <ctkPluginException.h>
#include <QtPlugin>
#include <QDir>
#include <QDirIterator>
#include <QtConcurrent/QtConcurrent>
#include <QCoreApplication>
#include <fmp_logger_i.h>
#include <fmp_settings_i.h>
#include <fmp_syncer_i.h>

FMPluginManager::FMPluginManager() : fwContext(nullptr)
{
}

FMPluginManager::~FMPluginManager()
{
}

void FMPluginManager::InitService()
{
    if (!fw.isNull()) return;

    //! 每次启动前删除配置文件夹
    QDir(qApp->applicationDirPath() + "/configuration").removeRecursively();

    fwProps = FMP::GetProperties();
    static ctkPluginFrameworkFactory fwFactory(fwProps.toVariantHash());

    fw = fwFactory.getFramework();
    try {
        fw->init();
        fwContext = fw->getPluginContext();
        fwContext->connectFrameworkListener(this, SLOT(OnFrameworkEvent(ctkPluginFrameworkEvent)));
        fwContext->connectPluginListener(this, SLOT(OnPluginEvent(ctkPluginEvent)));
        fwContext->connectServiceListener(this, "OnServiceEvent");

        fw->start();

        //! 设置模块必须第一个加载
        FMP_INFO() << "======================== Started ========================";

        if (GetService<FMPLoggerInterface>()) {
            FMP_INFO() << "Log service is ready.";
        }
        else {
            FMP_WARN() << "Log service is not ready.";
        }

        //!
        if (GetService<ctkEventAdmin>("org.commontk.eventadmin")) {
            FMP_INFO() << "Event service is ready";
        }
        else {
            FMP_ERROR() << "Event service is not ready.";
        }

        //!
        if (!(setting = GetService<FMPSettingsInterface>())) {
            FMP_WARN() << "Setting service is not ready.";
        }
        else {
            FMP_INFO() << "Setting service is ready.";
        }

        //!
        if (!(syncer = GetService<FMPSyncerInterface>())) {
            FMP_WARN() << "Synchronizer service is not ready.";
        }
        else {
            FMP_INFO() << "Synchronizer service is ready.";
        }

        connect(&pluginWatcher, SIGNAL(directoryChanged(QString)), SLOT(OnPluginsChanged(QString)));
        pluginWatcher.addPath(fwProps[FMP_PROPKEY_PLUGINPATH].toString());

        //!
        eventHandlers.append(QSharedPointer<ctkEventHandler>(new FMPUpgradeEventHandler(this)));
        eventHandlers.append(QSharedPointer<ctkEventHandler>(new FMPQuitEventHandler(this)));

        //! 启动配置的入口服务
        QString entry = fwProps[FMP_PROPKEY_ENTRY].toString();

//        fwContext->getPlugin()
        FMPBaseInterface* svc = FMP::GetService<FMPBaseInterface>(entry);
        if (svc) {
            svc->StartService();
        }
    }
    catch(const ctkPluginException& e) {
        FMP_ERROR() << e.cause()->what();
        exit(FMP_FAILURE);
    }
}

void FMPluginManager::UninitService()
{
    if (fwContext) {
        pluginWatcher.removePath(fwProps[FMP_PROPKEY_PLUGINPATH].toString());

        //! 优雅关闭
        while(!plugins.isEmpty()) {
            QSharedPointer<ctkPlugin> p = plugins.pop();
            if (p->getSymbolicName() == qobject_interface_iid<FMPLoggerInterface*>()) continue;

            FMP_INFO() << "Stopping " << p->getSymbolicName();
            p->stop(ctkPlugin::STOP_TRANSIENT);
        }
        FMP_INFO() << "======================== Quit ========================";
        fwContext = nullptr;
        fw->stop();
        fw->waitForStop(30000);
        qApp->quit();
    }
    else {
        FMP_INFO() << "Stopping in progress...";
    }
}


void FMPluginManager::LoadNewPlugins()
{
    QStringList newPlugins = GetNewPlugins();
    foreach(QString plg, newPlugins) {
        LoadPlugin(plg);
    }
}

int FMPluginManager::LoadPlugin(const QString &plugin)
{
    try {
        if(!plugin.isEmpty()) {
          QFileInfo pluginInfo = GetPluginInfo(plugin);
          if (pluginInfo.path().isEmpty()) throw ctkPluginException("Empty path for plugin " + plugin);

          //! 设置程序当前目录为插件目录，防止插件依赖 Dll 搜索不到导致无法加载
          QDir::setCurrent(pluginInfo.path());
          QSharedPointer<ctkPlugin> p = fwContext->installPlugin(QUrl::fromLocalFile(pluginInfo.canonicalFilePath()));
          if (!p.isNull()) {
              p->start(ctkPlugin::START_TRANSIENT);
              plugins << p;
          }
        }
        else {
            FMP_WARN() << "Empty plugin name." << plugin;
            return FMP_FAILURE;
        }
    }
    catch(const ctkPluginException &e) {
        FMP_WARN() << (e.cause() ? e.cause()->what() : e.what());
        return FMP_FAILURE;
    }

    return FMP_SUCCESS;
}

void FMPluginManager::UpgradeOldPlugins(const QVariantList &pids)
{
    foreach (QVariant pid, pids) {
        UpgradePlugin(pid.toInt());
    }
}

int FMPluginManager::UpgradePlugin(long pid)
{
    QSharedPointer<ctkPlugin> p = fwContext->getPlugin(pid);
    if (p) {
        QString pluginName = p->getSymbolicName();
        p->stop();

        //! TODO: 检查停止之前，服务是否处理启动状态，
        //! 如果是，则需要启动服务
        return LoadPlugin(pluginName);
    }

    return FMP_FAILURE;
}

int FMPluginManager::PostEvent(const QString &topic, const FMPProps &pps)
{
    ctkEventAdmin* event_admin = GetService<ctkEventAdmin>();
    if (event_admin) {
        ctkEvent ctk_event(topic, pps);
        event_admin->postEvent(ctk_event);
        FMP_INFO() << "Posted event" << topic << "Properties:" << pps;
        return FMP_SUCCESS;
    }
    else {
        FMP_WARN() << topic << "not send: Event admin not available.";
    }

    return FMP_FAILURE;
}

void FMPluginManager::OnPluginsChanged(const QString &path)
{
    FMP_INFO() << "Directory change detected:" << path;
    LoadNewPlugins();

    QVariantList expiredPlugins = GetExpiredPlugins();
    if (!expiredPlugins.isEmpty()) {
        FMPProps props;
        props[FMP_PROPKEY_PID_LIST] = expiredPlugins;
        //! Request plugin update
        PostEvent(QString(FMP_TOPICS_SERVICES) + FMPE_SERVICE_REQ_UPDATE, props);
    }
}

QFileInfo FMPluginManager::GetPluginInfo(const QString &pluginName, const QString &ver)
{
    QString pluginFileName(pluginName);
    pluginFileName.replace(".", "_");
    QStringList filters;
#ifdef Q_OS_WIN
    filters << "*" + pluginFileName + "*" + ".dll";
#else
#   ifdef Q_OS_LINUX
    filters << "*" + pluginFileName + "*" + ".so";
#   else
    filters << "*" + pluginFileName + "*" + ".dylib";
#   endif
#endif
    QDirIterator dirIter(qApp->applicationDirPath() + "/" + fwProps[FMP_PROPKEY_PLUGINPATH].toString(), filters, QDir::Files);
    QString pluginVersion = ver;
    QFileInfo pluginFileInfo;
    while(dirIter.hasNext()) {
        dirIter.next();
        QFileInfo fileInfo = dirIter.fileInfo();
        QString fileBasename = fileInfo.completeBaseName();
        if (fileBasename.startsWith("lib")) fileBasename = fileBasename.mid(3);
        QString tmpVer = fileBasename.section(pluginFileName, 1, 1).replace("_", "");
        if (pluginVersion.isEmpty()) {
            pluginVersion = tmpVer;
            pluginFileInfo = fileInfo;
        }
        else if (pluginVersion < tmpVer) {
            //! 删除旧版本
            QFile::remove(pluginFileInfo.absoluteFilePath());

            pluginVersion = tmpVer;
            pluginFileInfo = fileInfo;
        }
    }

    return pluginFileInfo;
}

QStringList FMPluginManager::GetNewPlugins()
{
    QStringList newPlugins;

    QStringList filters;
#ifdef Q_OS_WIN
    filters << "*.dll";
#else
#   ifdef Q_OS_LINUX
    filters << "*.so";
#   else
    filters << "*.dylib";
#   endif
#endif
    QMap<QString, QSharedPointer<ctkPlugin> > loadedPlugins;
    //! 管理器自身已经加载过
    loadedPlugins[qobject_interface_iid<FMPManagerInterface*>()] = QSharedPointer<ctkPlugin>();

    QList<QSharedPointer<ctkPlugin> > plugins = fwContext->getPlugins();
    foreach(QSharedPointer<ctkPlugin> p, plugins) {
        if (p->getState() == ctkPlugin::ACTIVE) {
            loadedPlugins[p->getSymbolicName()] = p;
        }
    }

    QDirIterator dirIter(qApp->applicationDirPath() + "/" + fwProps[FMP_PROPKEY_PLUGINPATH].toString(), filters, QDir::Files);
    while(dirIter.hasNext()) {
        dirIter.next();
        QFileInfo fileInfo = dirIter.fileInfo();
        QString fileBasename = fileInfo.completeBaseName();
        if (fileBasename.startsWith("lib")) fileBasename = fileBasename.mid(3);

        QString pluginName;
        pluginName = fileBasename.section("_", 0, -2).replace("_", ".");
        if (!loadedPlugins.contains(pluginName)) {
            newPlugins << pluginName;
        }
    }

    return newPlugins;
}

QVariantList FMPluginManager::GetExpiredPlugins()
{
    QVariantList expiredPlugins;

    QStringList filters;
#ifdef Q_OS_WIN
    filters << "*.dll";
#else
#   ifdef Q_OS_LINUX
    filters << "*.so";
#   else
    filters << "*.dylib";
#   endif
#endif

    QMap<QString, QSharedPointer<ctkPlugin> > loadedPlugins;
    //! 管理器自身已经加载过
    loadedPlugins[qobject_interface_iid<FMPManagerInterface*>()] = QSharedPointer<ctkPlugin>();

    QList<QSharedPointer<ctkPlugin> > plugins = fwContext->getPlugins();
    foreach(QSharedPointer<ctkPlugin> p, plugins) {
        loadedPlugins[p->getSymbolicName()] = p;
    }

    QDirIterator dirIter(qApp->applicationDirPath() + "/" + fwProps[FMP_PROPKEY_PLUGINPATH].toString(), filters, QDir::Files);
    while(dirIter.hasNext()) {
        dirIter.next();
        QFileInfo fileInfo = dirIter.fileInfo();
        QString fileBasename = fileInfo.completeBaseName();
        if (fileBasename.startsWith("lib")) fileBasename = fileBasename.mid(3);

        QString pluginName, plugin_ver;
        pluginName = fileBasename.section("_", 0, -2).replace("_", ".");
        plugin_ver  = fileBasename.section("_", -1);
        QSharedPointer<ctkPlugin> p;
        if (loadedPlugins.contains(pluginName) && (p = loadedPlugins[pluginName])
                && plugin_ver > p->getVersion().toString()) {
            expiredPlugins << p->getPluginId();
        }
    }

    return expiredPlugins;
}

void FMPluginManager::OnFrameworkEvent(const ctkPluginFrameworkEvent &event)
{
    QSharedPointer<ctkPlugin> plugin = event.getPlugin();
    FMP_DEBUG() << plugin->getSymbolicName() << plugin->getVersion() << event.getType();
}

void FMPluginManager::OnPluginEvent(const ctkPluginEvent &event)
{
    QSharedPointer<ctkPlugin> plugin = event.getPlugin();
    FMP_DEBUG() << plugin->getSymbolicName() << plugin->getVersion() << event.getType();
}

void FMPluginManager::OnServiceEvent(const ctkServiceEvent &event)
{
    QSharedPointer<ctkPlugin> plugin = event.getServiceReference().getPlugin();
    FMP_DEBUG() << plugin->getSymbolicName() << plugin->getVersion() << event.getType();
}


#if(QT_VERSION < QT_VERSION_CHECK(5,0,0))
    Q_EXPORT_PLUGIN2(com_fmp_manager, FMPluginManager)
#endif
