#include "fmp_logger_p.h"
#include <fmp_settings_i.h>
#include <QDateTime>
#include <QDir>
#include <QFileInfo>
#include <QCoreApplication>
#include <log4cxx/log4cxx.h>
#include <log4cxx/logger.h>
#include <log4cxx/basicconfigurator.h>
#include <log4cxx/propertyconfigurator.h>

using namespace log4cxx;
using namespace log4cxx::spi;

FMPLoggerPrivate::FMPLoggerPrivate(FMPLogger *q)
    : q_ptr(q),
      _inited(false)
{
}

int FMPLoggerPrivate::Init()
{
    if (_inited) return FMP_SUCCESS;

    _inited = true;

    QString path = qApp->applicationDirPath();
    QString pid = QString::number(qApp->applicationPid());

#ifdef Q_OS_WIN
    QLibrary lib("kernel32.dll");
    QFunctionPointer pfn = lib.resolve("SetEnvironmentVariableW");
    typedef int (__stdcall *FN_SETENV)(const wchar_t*k, const wchar_t *v);
    FN_SETENV fn_setenv = (FN_SETENV)pfn;
    fn_setenv(L"FMP_APP", path.toStdWString().c_str());
    fn_setenv(L"FMP_PID", pid.toStdWString().c_str());
    lib.unload();
#else
    setenv("FMP_APP", path.toStdString().c_str());
    setenv("FMP_PID", pid.toStdString().c_str());
#endif


    BasicConfigurator::configure();
    QByteArray filepath = (qApp->applicationDirPath() + "/log.props").toUtf8();
    PropertyConfigurator::configure(filepath.data());

    return FMP_SUCCESS;
}

int FMPLoggerPrivate::Uninit()
{
    if (!_inited) return FMP_SUCCESS;

    _inited = false;

    return FMP_SUCCESS;
}

void FMPLoggerPrivate::Log(short level, const QString &msg, const char* file, const char* func, int line)
{
    Q_Q(FMPLogger);

    LoggerPtr logger = Logger::getLogger(q->currentModule.toUtf8().data());
    switch(level)
    {
    case FMPLogStream::ERROR:
        if (logger->isErrorEnabled()) {
            ::log4cxx::helpers::MessageBuffer oss_;
            logger->forcedLog(::log4cxx::Level::getError(), oss_.str(oss_ << msg.toUtf8().data()), LocationInfo(file, func, line));
        }
        break;
    case FMPLogStream::WARNING:
        if (logger->isWarnEnabled()) {
            ::log4cxx::helpers::MessageBuffer oss_;
            logger->forcedLog(::log4cxx::Level::getWarn(), oss_.str(oss_ << msg.toUtf8().data()), LocationInfo(file, func, line));
        }
        break;
    case FMPLogStream::DEBUG:
        if (LOG4CXX_UNLIKELY(logger->isDebugEnabled())) {
            ::log4cxx::helpers::MessageBuffer oss_;
            logger->forcedLog(::log4cxx::Level::getDebug(), oss_.str(oss_ << msg.toUtf8().data()), LocationInfo(file, func, line));
        }
        break;
    default:
        if (logger->isInfoEnabled()) {
            ::log4cxx::helpers::MessageBuffer oss_;
            logger->forcedLog(::log4cxx::Level::getInfo(), oss_.str(oss_ << msg.toUtf8().data()), LocationInfo(file, func, line));
        }
        break;
    }
}

short FMPLoggerPrivate::GetLogLevel()
{
    return _GetValue(FMP_INIKEY_LOGLEVEL, FMPLogger::LOG_INFO).toInt();
}

void FMPLoggerPrivate::SetLogLevel(short level)
{
    if (_SetValue(FMP_INIKEY_LOGLEVEL, level)) {
        Q_Q(FMPLogger);
        q->_level = level;
    }
}

uint FMPLoggerPrivate::GetMaxLogSize()
{
    return _GetValue(FMP_INIKEY_LOGSIZE, 2).toUInt();
}

void FMPLoggerPrivate::SetMaxLogSize(uint sz)
{
    if (_SetValue(FMP_INIKEY_LOGSIZE, sz)) {
        Q_Q(FMPLogger);
        q->_max_size = sz;
    }
}

QString FMPLoggerPrivate::GetLogPath()
{
    QString path = _GetValue(FMP_INIKEY_LOGPATH).toString();
    if (path.isEmpty()) path = qApp->applicationDirPath() + "/log";
    path.replace("\\", "/");
    QDir log_dir;
    if (!log_dir.isAbsolutePath(path)) {
        path = qApp->applicationDirPath() + "/" + path;
    }
    return path;
}

void FMPLoggerPrivate::SetLogPath(const QString &path)
{
    if (_SetValue(FMP_INIKEY_LOGPATH, path)) {
        q_func()->_path = path;
    }
}

bool FMPLoggerPrivate::_SetValue(const QString &key, QVariant v)
{
    Q_Q(FMPLogger);
//    FMPSettingsInterface *sets = FMPManagerPlugin::GetManager()->GetService<FMPSettingsInterface>();
//    if (sets) {
//        sets->SetValue(key, v);
//        return true;
//    }
//    else {
//        FMP_WARN() << "Settings service not available";
//    }

    return false;
}

QVariant FMPLoggerPrivate::_GetValue(const QString &key, QVariant default_val)
{
    Q_Q(FMPLogger);
//    FMPSettingsInterface *sets = q->GetService<FMPSettingsInterface>(_ctx);
//    if (sets) {
//        return sets->GetValue(key);
//    }
//    else {
//        FMP_WARN() << "Settings service not available";
//    }

    return default_val;
}
