﻿#include <QCoreApplication>
#include <QProcess>
#include <QFile>
#include <QDir>
#include <QDebug>
#include <QDateTime>
#include <QSettings>

QString download_dir, deploy_dir;

void MessageOutput(QtMsgType type, const QMessageLogContext &context, const QString &msg)
{
    Q_UNUSED(type); Q_UNUSED(context);
    // 设置输出信息格式
    QString strDateTime = QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss.zzz");
    QString strMessage = QString("%1: %2").arg(strDateTime).arg(msg);
    // 输出信息至文件中（读写、追加形式）
    QFile file(QString("%1/porter.log").arg(download_dir));
    file.open(QIODevice::ReadWrite | QIODevice::Append);
    QTextStream stream(&file);
    stream << strMessage << "\r\n";
    file.flush();
    file.close();
}

QString GetProcessNameByPid(int pid);
bool    KillProcessByPid(int pid);

#define FMP_INIKEY_DOWNLOADPATH     "Syncer/DownloadPath"
#define FMP_INIKEY_DEPLOYPATH       "Syncer/DeployPath"

int main(int argc, char *argv[])
{
    if (argc < 2) {
        qDebug() << "Usage: FreemudSyncer [Process ID]";
        return 0;
    }

    QCoreApplication a(argc, argv);
    int instance_pid = QString(argv[1]).toInt();

    QString fmscupdPath = GetProcessNameByPid(instance_pid);
    QString app = fmscupdPath.section("/", -1);

#ifdef Q_OS_WIN
    QSettings sets(QCoreApplication::applicationDirPath() + "/" + QString(app).replace(".exe", ".ini"), QSettings::IniFormat);
#endif
    download_dir = sets.value(FMP_INIKEY_DOWNLOADPATH, "download").toString();
    deploy_dir = sets.value(FMP_INIKEY_DEPLOYPATH, "..").toString();

    if (QFileInfo(deploy_dir).isAbsolute()) {
        download_dir = deploy_dir + "/" + download_dir;
    }
    else {
        deploy_dir = qApp->applicationDirPath() + "/" + deploy_dir;
        download_dir = deploy_dir + "/" + download_dir;
    }

    // 日志输出
    QDir().mkdir(download_dir);
    qInstallMessageHandler(MessageOutput);

    qDebug() << "\r\n\r\n\r\n";
    qDebug() << QString("Process %1[%2] asks for sync.").arg(app).arg(instance_pid);

    QFile dInfoFile(QString("%1/dfilesInfo.txt").arg(download_dir));
    QFile resultFile(QString("%1/updateresult.txt").arg(download_dir));
    QStringList dFilesInfo, bUpdFiles;
    QString error;
    bool updateReault = true;

    // 开始替换文件
    if(dInfoFile.open(QIODevice::ReadOnly))
    {
        dFilesInfo = QString(dInfoFile.readAll()).split(",");
        dInfoFile.close();
    }else
    {
        error  = QString("open dfilesInfo.txt failed [%1]").arg(dInfoFile.errorString());
        qDebug() << error;
        updateReault = false;
        goto end;
    }

    //! 如果更新包含主程序，则关闭主程序
    bool process_closed = false;
    if (dFilesInfo.contains(app)) {
        KillProcessByPid(instance_pid);

        process_closed = true;
    }
    foreach(QString dfile, dFilesInfo) {
        if(!dfile.isEmpty()) {
            QString downloadFile = QString("%1/%2").arg(download_dir, dfile);
            QString oldFile = QString("%1/%2").arg(deploy_dir, dfile);
            QString newName = QString("%1.bak").arg(oldFile);
            qDebug() << QString("checking file [%1]").arg(oldFile);
            if(!QFile(oldFile).exists()) {
                qDebug() << "file not exists";
                QDir parent_dir(oldFile.section("/", 0, -2));
                if(!parent_dir.exists()) {
                    if (parent_dir.mkpath(parent_dir.path())) {
                        qDebug() << "mkdir successful";
                    }
                    else {
                        qDebug() << "mkdir failed";
                    }
                }
            }
            else {
                qDebug() << "file exists";
                QFile(newName).remove();
                if(QFile().rename(oldFile, newName)) {
                    qDebug() << "rename successful";
                }
                else {
                    qDebug() << "rename failed";
                }
            }

            if(QFile().copy(downloadFile, oldFile)) {
                qDebug() << "copy file successful";
            }
            else {
                error = "copy file failed";
                qDebug() << error;
                updateReault = false;
                // 回滚文件
                foreach(QString file, bUpdFiles) {
                  qDebug() << QString("rollback file [%1]").arg(file);
                  QFile(file).remove();
                  QFile().rename(QString("%1.bak").arg(file), file);
                }
                break;
            }
            bUpdFiles.append(oldFile);
        }
    }

end:
    if(resultFile.open(QIODevice::WriteOnly))
    {
        QByteArray buf;
        if(updateReault)
        {
            buf = "1";
        }else
        {
            buf = QString("0%1").arg(error).toLatin1();
        }
        resultFile.write(buf);
        resultFile.close();
    }

    if (process_closed) {
        // 启动服务
        QProcess::startDetached(fmscupdPath, QStringList());
    }

    return 0;
}

#ifdef Q_OS_WIN
#include <Windows.h>
#include <Psapi.h>
#endif

QString GetProcessNameByPid(int pid)
{
    QString proc_name;
#ifdef Q_OS_WIN
    HANDLE hProcess = OpenProcess(PROCESS_ALL_ACCESS, false, pid);
    WCHAR module[MAX_PATH] = { 0 };
    GetModuleFileNameEx(hProcess, NULL, module, MAX_PATH);

    proc_name = QString::fromWCharArray(module);
    proc_name = proc_name.replace("\\", "/");
#endif

    return proc_name;
}

bool    KillProcessByPid(int pid)
{
    bool killed = false;

#ifdef Q_OS_WIN
    HANDLE hProcess = OpenProcess(PROCESS_ALL_ACCESS, false, pid);
    killed = (TerminateProcess(hProcess, 0) == TRUE);
#endif

    return killed;
}
