#include "flowControl.h"
#include "fmPlugin.h"
#include "fmPrinter.h"
#include "QsLog.h"
#include "DTools/dataManger.h"
#include "preDefine.h"
#include "DTools/configManger.h"
#include "Model/cashierObject.h"
#include <QJsonDocument>
#include <QJsonArray>
#include <QTimer>
#include <QDir>
#include <QApplication>

FlowControl &FlowControl::GetInstance()
{
   static FlowControl fc;
   return fc;
}

FlowControl::FlowControl()
{
    m_loginSocket = NULL;
    m_pullOrderSocket = NULL;
    m_procOrderSocket = NULL;
}

bool FlowControl::_GetStoreInfo()
{
    QString error;
    bool result;

    emit showAlert(AlertForm::LOADING, "正在获取门店信息......");
    QLOG_INFO() << QString("[---get storeInfo---]");
    result = FmPlugin::GetInstance().GetStoreInfo(m_storeId, m_posId, m_cashierId, m_bDate, error);
    QLOG_INFO() << QString("get storeInfo finsh. [result:%1][msg:%2][storeId:%3,posId:%4,cashierId:%5,bdate:%6]")
                   .arg(result).arg(error, m_storeId, m_posId, m_cashierId, m_bDate.toString("yyyy-MM-dd"));

    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "获取门店信息失败!");
    }

    return result;
}

bool FlowControl::_Login()
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;

    if(!_GetStoreInfo())
    {
        QLOG_INFO() << QString("%1 msec after login...").arg(VALUE_RELOGINTIME);
        QTimer::singleShot(VALUE_RELOGINTIME, this, &FlowControl::_Login);
        return false;
    }
    sendJson = DataManger::GetInstance().GetLoginData(m_storeId, SERVER_PASSWORD, m_posId, m_cashierId);

    emit showAlert(AlertForm::LOADING, "正在登录......");
    QLOG_INFO() << QString("[---login---][requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_loginSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("login finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "登录失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("登录失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            m_timestamp = recvJson[JSON_TIMESTAMPS].toString();
            DataManger::GetInstance().SetToken(recvJson[JSON_TOKEN].toString());
            emit hideAlert();
            emit setStoreInfo(m_storeId);

            _PullOrder();
            // TODO(用于鲜丰演示)
            //_SynStock();
        }
    }

    if(!result)
    {
        QLOG_INFO() << QString("%1 msec after login...").arg(VALUE_RELOGINTIME);
        QTimer::singleShot(VALUE_RELOGINTIME, this, &FlowControl::_Login);
    }

    return result;
}

bool FlowControl::_PullOrder()
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;
    int syncTime = VALUE_PULLSYNCTIME;

    sendJson = DataManger::GetInstance().GetPullOrderData(m_timestamp);

    QLOG_INFO() << QString("[---pull order---][requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_pullOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("pull order finsh. [result:%1][msg:%2]")
                   .arg(result).arg(error);
    if(!result)
    {
        emit setNetStatus("<font color='#ff0000'>异常</font>");
    }else
    {
        emit setNetStatus("正常");
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            QString error = recvJson[JSON_MESSAGE].toString();
            QLOG_ERROR() << QString("pull orders error.[msg->%1]").arg(error);
            emit showAlert(AlertForm::ERROR, QString("获取订单失败![%1]").arg(error));
        }else
        {
            // 获取门店营业状态
            QString strOpeStatus("正常");
            QJsonArray opeStatus = recvJson[JSON_SHOPSTATUS].toArray();
            foreach(QJsonValue status, opeStatus)
            {
                QJsonObject jsonObject =  status.toObject();
                if(jsonObject[JSON_SHOPSTATUSDESC].toString().compare("营业中"))
                {
                    strOpeStatus = QString("<font color='#ff0000'>异常</font>");
                    break;
                }
            }
            emit setOpeStatus(strOpeStatus);
            // 获取订单信息
            QJsonArray orders = recvJson[JSON_ORDERS].toArray();
            foreach(QJsonValue order, orders)
            {
                QJsonObject jsonObject = order.toObject();
                OrderObject *orderObject = NULL;
                orderObject = m_ordersMap.value(jsonObject[JSON_ORDERID].toString());
                if(orderObject == NULL)
                {
                    //新数据
                    orderObject = new OrderObject(this);
                    orderObject->FromJson(jsonObject);
                    m_ordersMap.insert(orderObject->order_id, orderObject);
                    QLOG_INFO() << QString("new order[%1:%2].[data:%3]")
                                   .arg(orderObject->order_id, orderObject->status_desc)
                                   .arg(_GetJsonStr(jsonObject));
                    emit changeOrderStatus(orderObject);
                }else
                {
                    //旧数据有更新
                    int oldStatus = orderObject->status;
                    orderObject->FromJson(jsonObject);
                    QLOG_INFO() << QString("old order[%1:%2].[data:%3]")
                                   .arg(orderObject->order_id, orderObject->status_desc)
                                   .arg(_GetJsonStr(jsonObject));
                    if(oldStatus != orderObject->status)
                    {
                       emit changeOrderStatus(orderObject, oldStatus);
                    }
                }

                if( orderObject->status != 1 &&
                    orderObject->status != 3 &&
                    orderObject->status != -1 &&
                    orderObject->status != 20 &&
                    orderObject->status != 30 &&
                    orderObject->status != 40 &&
                    orderObject->status != 100)
                {
                    QLOG_INFO() << QString("[---order entry begin---]");
                    // 写入销售单
                    if(!FmPlugin::GetInstance().DoOrderEntry(orderObject, error))
                    {
                        QLOG_ERROR() << QString("order entry failed,[%1]").arg(error);
                    }else
                    {
                        QLOG_INFO() << QString("order entry success.");
                    }
                }

                m_timestamp = orderObject->timestamp;
            }
            syncTime = recvJson[JSON_SYNCTIME].toInt()*1000;
        }
    }

    QLOG_INFO() << QString("%1 msec after pull order...").arg(syncTime);
    QTimer::singleShot(syncTime, this, &FlowControl::_PullOrder);

    return result;
}

bool FlowControl::_GetDelivers(const QString &orderId)
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;
    OrderObject *orderObj;

    orderObj = m_ordersMap.value(orderId);
    sendJson = DataManger::GetInstance().GetPullDeliverData(m_storeId, orderObj->channel);

    emit showAlert(AlertForm::LOADING, "正在获取配送员信息......");
    QLOG_INFO() << QString("[---get delivers---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_procOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("get delivers finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "获取配送员失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("获取配送员失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            QList<DeliverObject> deliverList;
            QJsonArray delivers = recvJson[JSON_DELIVERS].toArray();
            foreach(QJsonValue deliver, delivers)
            {
                QJsonObject jsonObject = deliver.toObject();
                DeliverObject deliverObject;
                deliverObject.FetchDataFromJson(jsonObject);
                deliverList.append(deliverObject);
            }
            emit hideAlert();
            emit showDeliverPickForm(orderId, deliverList);
        }
    }

    return result;
}

bool FlowControl::_ConfirmOrder(const QString &orderId, const DeliverObject &deliverObj)
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;

    sendJson = DataManger::GetInstance().GetConfirmOrderData(orderId, deliverObj.id, deliverObj.name, deliverObj.phone);

    emit showAlert(AlertForm::LOADING, "正在通信......");
    QLOG_INFO() << QString("[---confirm order---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_procOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("confirm order finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "接单失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("接单失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            OrderObject *orderObject = m_ordersMap.value(orderId);


            QString remark(orderObject->remark), deliveryTime;
            remark = remark.isEmpty()? "" : QString("[备注: %1]").arg(remark);
            deliveryTime = QString("[期望送达时间: %1]").arg(orderObject->delivery_time==0? "立即送出" : QDateTime::fromTime_t(orderObject->delivery_time).toString("yyyy-MM-dd hh:mm:ss"));
            emit showAlert(AlertForm::SUCCESS, QString("接单成功!%1<br/>%2").arg(remark).arg(deliveryTime));

            // 通知主界面移动订单
            int oldStatus = orderObject->status;
            orderObject->status = recvJson[JSON_STATUS].toInt();
            orderObject->status_desc = recvJson[JSON_STATUSDESC].toString();
            emit changeOrderStatus(orderObject, oldStatus);

            // 打印
            QLOG_INFO() << QString("[---begin print---].");
            orderObject->courier_name = deliverObj.name;
            orderObject->courier_phone = deliverObj.phone;

            if(FmPrinter::GetInstance().DoPrint(ConfigManger::GetInstance().GetPrinterName(), orderObject))
            {
                QLOG_INFO() << QString("print sucessful");
            }else
            {
                QLOG_INFO() << QString("print failed");
            }
        }
    }

    return result;
}

bool FlowControl::_RefuseOrder(const QString &orderId, int refuseCode)
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;

    sendJson = DataManger::GetInstance().GetRefuseOrderData(refuseCode, orderId);

    emit showAlert(AlertForm::LOADING, "正在通信......");
    QLOG_INFO() << QString("[---refuse order---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_procOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("refuse order finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "拒单失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("拒单失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            emit showAlert(AlertForm::SUCCESS, "拒单成功!");

            // 通知主界面移动订单
            OrderObject *orderObject = m_ordersMap.value(orderId);
            int oldStatus = orderObject->status;
            orderObject->status = recvJson[JSON_STATUS].toInt();
            orderObject->status_desc = recvJson[JSON_STATUSDESC].toString();
            emit changeOrderStatus(orderObject, oldStatus);
        }
    }

    return result;
}

bool FlowControl::_SendOrder(const QString& orderId)
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;

    sendJson = DataManger::GetInstance().GetSendOrderData(orderId);

    emit showAlert(AlertForm::LOADING, "正在通信......");
    QLOG_INFO() << QString("[---sendout order---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_procOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("sendout order finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "送出失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("送出失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            emit showAlert(AlertForm::SUCCESS, "送出成功!");

            // 通知主界面移动订单
            OrderObject *orderObject = m_ordersMap.value(orderId);
            int oldStatus = orderObject->status;
            orderObject->status = recvJson[JSON_STATUS].toInt();
            orderObject->status_desc = recvJson[JSON_STATUSDESC].toString();
            emit changeOrderStatus(orderObject, oldStatus);
        }
    }

    return result;
}

bool FlowControl::_CompleteOrder(const QString& orderId)
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;

    sendJson = DataManger::GetInstance().GetCompleteOrderData(orderId);

    emit showAlert(AlertForm::LOADING, "正在通信......");
    QLOG_INFO() << QString("[---complete order---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_procOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("complete order finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "完成失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("完成失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            emit showAlert(AlertForm::SUCCESS, "完成成功!");

            // 通知主界面移动订单
            OrderObject *orderObject = m_ordersMap.value(orderId);
            int oldStatus = orderObject->status;
            orderObject->status = recvJson[JSON_STATUS].toInt();
            orderObject->status_desc = recvJson[JSON_STATUSDESC].toString();
            emit changeOrderStatus(orderObject, oldStatus);
        }
    }

    return result;
}

bool FlowControl::_RefuseRefund(const QString& orderId)
{
    QString error;
    bool result;
    QJsonObject sendJson;
    QJsonObject recvJson;

    sendJson = DataManger::GetInstance().GetRefuseRefundData(VALUE_REFUSEREFUND_REASON,orderId);

    emit showAlert(AlertForm::LOADING, "正在通信......");
    QLOG_INFO() << QString("[---refuseRefund order---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
    result = m_procOrderSocket->Request(sendJson, recvJson, error);
    QLOG_INFO() << QString("refuseRefund order finsh. [result:%1][msg:%2][recvData:%3]")
                   .arg(result).arg(error, _GetJsonStr(recvJson));
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "拒绝退单失败![网络错误]");
    }else
    {
        if(JSON_STATUSCODE_OK != recvJson[JSON_STATUSCODE].toInt())
        {
            result = false;
            emit showAlert(AlertForm::ERROR, QString("拒绝退单失败![%1]").arg(recvJson[JSON_MESSAGE].toString()));
        }else
        {
            emit showAlert(AlertForm::SUCCESS, "拒绝退单成功!");

            // 通知主界面移动订单
            OrderObject *orderObject = m_ordersMap.value(orderId);
            int oldStatus = orderObject->status;
            orderObject->status = recvJson[JSON_STATUS].toInt();
            orderObject->status_desc = recvJson[JSON_STATUSDESC].toString();
            emit changeOrderStatus(orderObject, oldStatus);
        }
    }

    return result;
}

bool FlowControl::_GetCashiers(QList<CashierObject> &cashiersList)
{
    QString strCashiers, error;
    bool result;

    emit showAlert(AlertForm::LOADING, "正在查询收银员信息.....");
    QLOG_INFO() << QString("[---get cashiers---].");
    result = FmPlugin::GetInstance().GetOnDutyCashiers(cashiersList, error);
    foreach (CashierObject cashier, cashiersList)
    {
        strCashiers.append(QString("|%1,%2,%3,%4|").arg(cashier.id, cashier.name, cashier.shiftId, cashier.shiftName));
    }
    QLOG_INFO() << QString("get cashiers finsh. [result:%1][msg:%2][data:%3]").arg(result).arg(error, strCashiers);
    if(!result)
    {
        emit showAlert(AlertForm::ERROR, "查询收银员信息失败!");
    }
    return result;
}

bool FlowControl::_SynStock()
{
    QList<StockObject> stockList;
    QString strStock, error;
    int synInterval = VALUE_SYNSTOCKINTERVAL;
    bool result;

    QLOG_INFO() << QString("[---get stockInfo---].");
    result = FmPlugin::GetInstance().GetStockInfo(stockList, error);
    foreach (StockObject stock, stockList)
    {
        strStock.append(QString("|%1,%2,%3|").arg(stock.id, stock.upc).arg(QString::number(stock.num)));
    }
    QLOG_INFO() << QString("get stockInfo finsh. [result:%1][msg:%2][data:%3]").arg(result).arg(error, strStock);
    if(result)
    {
        // 获取库存信息成功则推送到服务器
        QJsonObject sendJson;
        QJsonObject recvJson;
        sendJson = DataManger::GetInstance().GetSynStockData(stockList);
        QLOG_INFO() << QString("[---push stockInfo---]. [requestData:%1]").arg(_GetJsonStr(sendJson));
        result = m_syncStockSocket->Request(sendJson, recvJson, error);
        QLOG_INFO() << QString("push stockInfo finsh. [result:%1][msg:%2][recvData:%3]")
                       .arg(result).arg(error, _GetJsonStr(recvJson));
        if(result)
        {
            if(JSON_STATUSCODE_OK == recvJson[JSON_STATUSCODE].toInt())
            {
                synInterval = recvJson[JSON_SYNCTIME].toInt()*1000;
            }
        }
    }

    QLOG_INFO() << QString("%1 msec after synStock...").arg(synInterval);
    QTimer::singleShot(synInterval, this, &FlowControl::_SynStock);

    return result;
}

bool FlowControl::_CheckCashiers()
{
    bool result = false;

    QList<CashierObject> cashiersList;
    if(_GetCashiers(cashiersList))
    {
        foreach (CashierObject cashier, cashiersList)
        {
            if(!cashier.id.compare(m_cashierObject.id))
            {
                m_cashierObject = cashier;
                result = true;
                break;
            }
        }
        if(result == false)
        {
             emit showAlert(AlertForm::ERROR, "当前收银员已下班请重新选择!");
        }
    }

    return result;
}

QString FlowControl::_GetJsonStr(const QJsonObject &json)
{
    return QString(QJsonDocument(json).toJson(QJsonDocument::Compact)).replace("\"","");
}

void FlowControl::onFlowStart()
{
    m_timestamp = "0";
    m_loginSocket = new BillSocket(this);
    m_pullOrderSocket = new BillSocket(this);
    m_procOrderSocket = new BillSocket(this);
    m_syncStockSocket = new BillSocket(this);


    _Login();

    return;
}

void FlowControl::onConnectDb(const QString &host, const QString &username, const QString &password, const QString &dbname)
{
    QString error; bool result;
    QLOG_INFO() << QString("[---connect database---]. [host:%1][username:%2][password:%3][dbname:%4]")
                   .arg(host, username, password, dbname);
    result = FmPlugin::GetInstance().ConnectDb(host, username, password, dbname, error);
    QLOG_INFO() << QString("connect finsh. [result:%1][msg:%2]").arg(result).arg(error);
    emit connectDbFinsh(result, error);
    return;
}

void FlowControl::onProcessOrder(const QString &operation, const QString &orderId, const DeliverObject &deliverObj)
{
    // TODO(用于鲜丰演示)
//    if(!operation.compare(OPERATION_GETDELIVERS))
//    {
//        // 放在这检测收银员的合法性因为接单前会获取配送员
//        if(_CheckCashiers())
//        {
//            _GetDelivers(orderId);
//        }
//    }else if(!operation.compare(OPERATION_CONFRIM))
//    {
//        _ConfirmOrder(orderId, deliverObj);
//    }else if(!operation.compare(OPERATION_REFUSE))
//    {
//        _RefuseOrder(orderId, 2);
//    }else if(!operation.compare(OPERATION_SENDOUT))
//    {
//        _SendOrder(orderId);
//    }else if(!operation.compare(OPERATION_COMPLETE))
//    {
//        _CompleteOrder(orderId);
//    }else if(!operation.compare(OPERATION_REFUSEREFUND))
//    {
//        _RefuseRefund(orderId);
//    }

    if(!operation.compare(OPERATION_GETDELIVERS))
    {
        _ConfirmOrder(orderId, deliverObj);
    }else if(!operation.compare(OPERATION_CONFRIM))
    {
        _ConfirmOrder(orderId, deliverObj);
    }else if(!operation.compare(OPERATION_REFUSE))
    {
        _RefuseOrder(orderId, 2);
    }else if(!operation.compare(OPERATION_SENDOUT))
    {
        _SendOrder(orderId);
    }else if(!operation.compare(OPERATION_COMPLETE))
    {
        _CompleteOrder(orderId);
    }else if(!operation.compare(OPERATION_REFUSEREFUND))
    {
        _RefuseRefund(orderId);
    }
}

void FlowControl::onGetOnDutyCashiers()
{
    QList<CashierObject> cashiersList;
    if(_GetCashiers(cashiersList))
    {
        emit hideAlert();
        emit showCashierPickForm(cashiersList);
    }
}

void FlowControl::onUpdateCashier(const CashierObject &cashier)
{
    m_cashierObject = cashier;
}

void FlowControl::onGetOrderDetails(const QString &orderId)
{
//    if(_CheckCashiers())
//    {
        emit hideAlert();
        emit showOrderDetails(m_ordersMap.value(orderId));
 //   }
}

void FlowControl::onReEntryOrder(const QString &orderId)
{
    emit showAlert(AlertForm::LOADING, "正在补录......");

    QLOG_INFO() << QString("[---order reEntry begin---]");
    // 写入销售单
    OrderObject *orderObject = m_ordersMap.value(orderId);
    QString error;
    if(!FmPlugin::GetInstance().DoOrderEntry(orderObject, error))
    {
        QLOG_ERROR() << QString("order reEntry failed,[%1]").arg(error);
        emit showAlert(AlertForm::ERROR, "<font color=\"#FF0000\">补录失败</font>");
    }else
    {
        QFile file(QString("%1/orders/%2").arg(QApplication::applicationDirPath(), orderObject->order_id));
        file.remove();
        QLOG_INFO() << QString("order reEentry success.");
        emit showAlert(AlertForm::SUCCESS, "补录成功");
    }
}

void FlowControl::onSerachOrder(const QString &text)
{
    QStringList orderIdList;
    if(!text.isEmpty())
    {
        QMap<QString, OrderObject*>::iterator order;
        for(order = m_ordersMap.begin(); order!=m_ordersMap.end(); order++)
        {
            QString orderId = order.key();
            while (!orderId.at(0).isDigit())
            {
              orderId = orderId.mid(1);
            }
            if(orderId.startsWith(text))
            {
                orderIdList.append(order.key());
            }
        }
    }
    emit showSearchOrderResult(orderIdList);
}
