package cn.freemud.management.service.impl;

import cn.freemud.base.entity.BaseResponse;
import cn.freemud.management.entities.dto.request.order.OrderManagerRequest;
import cn.freemud.management.entities.dto.response.order.OrderManagerResponse;
import cn.freemud.management.entities.dto.response.pay.PayRefundResponse;
import cn.freemud.management.enums.OperateType;
import cn.freemud.management.enums.RefundStatus;
import cn.freemud.management.enums.ResponseResult;
import cn.freemud.management.enums.SettlementTypeEnum;
import cn.freemud.management.intercept.OrderServiceException;
import cn.freemud.management.service.OrderBaseService;
import cn.freemud.management.service.OrderManagerService;
import cn.freemud.management.service.handle.*;
import cn.freemud.management.util.ResponseUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.freemud.api.assortment.datamanager.entity.db.AssortmentOpenPlatformPartnerWxappConfig;
import com.freemud.application.sdk.api.couponcenter.offline.domain.TransactionVO;
import com.freemud.application.sdk.api.couponcenter.offline.request.CouponRequest;
import com.freemud.application.sdk.api.log.ApiLog;
import com.freemud.application.sdk.api.log.LogThreadLocal;
import com.freemud.application.sdk.api.ordercenter.enums.AbnormalStateEnum;
import com.freemud.application.sdk.api.ordercenter.enums.AfterSalesType;
import com.freemud.application.sdk.api.ordercenter.enums.OrderClientType;
import com.freemud.application.sdk.api.ordercenter.request.OrderCancelReq;
import com.freemud.application.sdk.api.ordercenter.request.OrderExtInfoDto;
import com.freemud.application.sdk.api.ordercenter.request.SdkUpdateAbnormalState;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.OrderSettlementResp;
import com.freemud.application.sdk.api.ordercenter.service.OrderSdkService;
import com.freemud.application.sdk.api.service.EmailAlertService;
import com.freemud.application.sdk.api.storecenter.request.StoreInfoRequest;
import com.freemud.application.sdk.api.storecenter.response.StoreResponse;
import com.freemud.application.sdk.api.storecenter.service.StoreCenterService;
import com.freemud.sdk.api.assortment.order.enums.*;
import com.freemud.sdk.api.assortment.order.request.order.CancelOrderRequest;
import com.freemud.sdk.api.assortment.order.response.order.BaseOrderResponse;
import com.freemud.sdk.api.assortment.order.response.order.QueryOrdersResponse;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static cn.freemud.management.enums.OperateType.ORDER_AGREE_REFUND;
import static cn.freemud.management.enums.OperateType.ORDER_CANCEL;

/**
 * All rights Reserved, Designed By www.freemud.cn
 *
 * @version V1.0
 * @Title: rrr
 * @Package cn.freemud.management.service.impl
 * @Description: saas订单履单操作
 * @author: shuhu.hou
 * @date: 2020/3/26 16:01
 * @Copyright: 2020 www.freemud.cn Inc. All rights reserved.
 * 注意：本内容仅限于上海非码科技内部传阅，禁止外泄以及用于其他的商业目
 */
@Service
public class SaasOrderMangerServiceImpl implements OrderManagerService {

    @Autowired
    private SaasOrderHandle saasOrderHandle;
    @Autowired
    private OrderBaseService orderBaseService;
    @Autowired
    private StoreCenterService storeCenterService;
    @Autowired
    private PaymentHandle paymentHandle;
    @Autowired
    private StockHandle stockHandle;
    @Autowired
    private DeliveryHandle deliveryHandle;
    @Autowired
    private OrderVerifyHandle orderVerifyHandle;
    @Autowired
    private OrderSdkService orderSdkService;
    @Autowired
    private EmailAlertService emailAlertService;
    private static final int SUCCESS = 100;

    /**
     * 接单
     *
     * @param request   请求参数
     * @param orderBean 订单实体
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderConfirm(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单接单
        BaseResponse baseResponse = saasOrderHandle.orderConfirm(request, orderBean);
        //获取门店信息
        StoreResponse storeInfo = storeCenterService.getStoreInfo(new StoreInfoRequest(orderBean.getCompanyId(), orderBean.getShopId()), "");
        // 创建配送单
        deliveryHandle.createDelivery(orderBean, request, storeInfo);
        // 推送pos、微信消息中心 重复接单不重复推消息（针对蜜雪做的修改）
        if(baseResponse != null && baseResponse.getResult() == null)
            orderBaseService.sendMessage(orderBean, OperateType.ORDER_CONFIRM, request.getReason());
        return ResponseUtil.success(new OrderManagerResponse(storeInfo.getBizVO().getOrderPrintConfig(), orderBean.getGmtExpect(),
                CollectionUtils.isEmpty(orderBean.getRefundList()) ? null : RefundStatus.getByCode(orderBean.getRefundList().get(0).getStatus())));
    }

    /**
     * 制作完成
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderDone(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单制作完成
        saasOrderHandle.orderDone(request, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_DONE, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 订单配送
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderDelivery(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单配送
        saasOrderHandle.orderDelivery(request, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_SEND, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 订单完成
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderAffirm(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单完成
        saasOrderHandle.orderAffirm(request, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_AFFIRM, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 拒单
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderReject(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //支付退款
        PayRefundResponse refundResponse = new PayRefundResponse();
        try {
            if(null != orderBean.getOrderPayItem() && orderBean.getOrderPayItem().size()>0){
                refundResponse = paymentHandle.multiRefund(orderBean);
            }else{
                refundResponse = paymentHandle.refund(request, orderBean);
            }
        } catch (OrderServiceException orderEx){ // 爱马哥兼容，不抛错
            return ResponseUtil.error(orderEx.getResult());
        }
        //订单拒单
        saasOrderHandle.orderReject(request, refundResponse, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_REJECT, request.getReason());
        //todo  可以考虑基于订单回调异步实现
        //冲正库存
        stockHandle.revert(request,orderBean);
        return ResponseUtil.success();
    }



    /**
     * 同意退款
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderAgreeRefund(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //取消配送单
        deliveryHandle.cancelDelivery(orderBean,request);
        //支付退款
        PayRefundResponse refundResponse = new PayRefundResponse();
        try {
            if(null != orderBean.getOrderPayItem() && orderBean.getOrderPayItem().size()>0){
                refundResponse = paymentHandle.multiRefund(orderBean);
            }else{
                refundResponse = paymentHandle.refund(request, orderBean);
            }
        } catch (OrderServiceException orderEx){ // 爱马哥兼容，不抛错
            return ResponseUtil.error(orderEx.getResult());
        }
        //订单同意退款
        saasOrderHandle.refundAgree(request, refundResponse, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, ORDER_AGREE_REFUND, request.getReason());
        //todo  可以考虑基于订单回调异步实现
        //冲正库存
        stockHandle.revert(request,orderBean);
        return ResponseUtil.success();
    }

    /**
     * 拒绝退款
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderRejectRefund(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //拒绝退款
        saasOrderHandle.refundReject(request,orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_REJECT_REFUND, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 取消订单
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderCancel(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean){
        List<Integer> oldOrderStatusList = Arrays.asList(OldOrderStatus.RECEIPT.getCode(),OldOrderStatus.COMPLETE.getCode(),OldOrderStatus.COMPLETE_MAKE.getCode());
        List<String> sourceList = Arrays.asList(OrderSourceType.SAAS.getCode(),OrderSourceType.ALIPAY.getCode(),OrderSourceType.APP.getCode());
        if (sourceList.contains(orderBean.getSource()) && oldOrderStatusList.contains(orderBean.getStatus())) {
            executeRefund(request,orderBean, OperateType.ORDER_CANCEL.getOpType());
        } else if (OrderSourceType.MEAL.getCode().equals(orderBean.getSource())) {
            if (OldOrderStatus.WAIT_PAY.getCode().equals(orderBean.getStatus()) || PayType.CASH_ON_DELIVERY.getCode() == orderBean.getPayType()) {
                executeCancel(request,orderBean, RefundStatus.COMPLETE_REFUND.getCode(), AfterSalesType.USER_SALE_RETURN.getIndex(), Integer.valueOf(orderBean.getOrderClient()));
            } else {
                executeRefund(request,orderBean, OperateType.ORDER_CANCEL.getOpType());
            }
        } else if (OrderSourceType.POS.getCode().equals(orderBean.getSource())) {
            return posCancel(orderBean);
        }
        return ResponseUtil.success();
    }

    /**
     * 退款操作
     * @param request
     * @param orderBean
     * @param operationType
     * @return
     */
    private BaseResponse<OrderManagerResponse> executeRefund(OrderManagerRequest request,QueryOrdersResponse.DataBean.OrderBean orderBean, String operationType) {
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.orderCancel(orderBean, request, OperateType.ORDER_AGREE_REFUND.getOpType());
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return ResponseUtil.error(verifyResponse.getCode());
        }
        BaseResponse<OrderManagerResponse> baseResponse = new BaseResponse<OrderManagerResponse>();
        //调用oms拒绝订单
        if (Objects.equals(operationType,OperateType.ORDER_REJECT_REFUND.getOpType())) {
            baseResponse = this.orderReject(request, orderBean);
        }
        //调用oms同意退款
        if (Objects.equals(operationType , OperateType.ORDER_AGREE_REFUND.getOpType())) {
            // 同意退款在支付退款后的退款/退货完成操作
            baseResponse = this.orderAgreeRefund(request, orderBean);
        }
        if (Objects.equals(operationType , OperateType.ORDER_REJECT_REFUND.getOpType())) {
            // 取消订单
            baseResponse = this.orderReject(request, orderBean);
        }
        if(!ObjectUtils.equals(SUCCESS,Integer.valueOf(baseResponse.getCode()))){
            emailAlertService.sendEmailAlert("商家取消订单失败", "请求json:" + JSONObject.toJSONString(orderBean) + "返回msg:" + baseResponse.getMessage());
            return ResponseUtil.error(ResponseResult.PARTNER_ORDER_CANCEL_ERROR);
        }
        return ResponseUtil.success();
    }

    /**
     * 拒单在支付退款后的取消订单操作
     *
     * @param order          订单详情
     * @param refundStatus   支付退款状态
     * @return
     */
    private BaseResponse<OrderManagerResponse> executeCancel(OrderManagerRequest request,QueryOrdersResponse.DataBean.OrderBean order, Integer refundStatus, Integer afterSalesType, Integer orderClient) {
        Integer createEvent;
        if (PayRefundStatus.RUNNING.getCode().equals(refundStatus)) {
            createEvent = 5;
        } else if (PayRefundStatus.SUCCESS.getCode().equals(refundStatus)) {
            createEvent = 4;
        } else {
            createEvent = 5;
        }
        OrderCancelReq orderCancelReq = new OrderCancelReq();
        orderCancelReq.setPartnerId(order.getCompanyId());
        orderCancelReq.setOrderCode(order.getOid());
        orderCancelReq.setCancelReason(request.getReason());
        orderCancelReq.setAfterSalesType(afterSalesType);
        orderCancelReq.setCreateEvent(createEvent);
        orderCancelReq.setOrderClient(orderClient == null ? OrderClientType.SAAS.getIndex() : orderClient);
        orderCancelReq.setOperator(request.getOperator());
        com.freemud.application.sdk.api.ordercenter.response.BaseResponse baseResponse = orderSdkService.cancelOrder(orderCancelReq,UUID.randomUUID().toString());
        if (!ObjectUtils.equals(SUCCESS,baseResponse.getCode())) {
            return ResponseUtil.error(ResponseResult.PARTNER_ORDER_CANCEL_ERROR);
        }
        return ResponseUtil.success();
    }

    public BaseResponse<OrderManagerResponse> posCancel(QueryOrdersResponse.DataBean.OrderBean order) {
        //调用cancel接口，商家退款
        OrderCancelReq orderCancelReq = new OrderCancelReq();
        orderCancelReq.setPartnerId(order.getCompanyId());
        orderCancelReq.setOrderCode(order.getOid());
        orderCancelReq.setCancelReason("商家取消");
        orderCancelReq.setAfterSalesType(AfterSalesType.PARTNER_CANCEL.getIndex());
        orderCancelReq.setCreateEvent(4);
        orderCancelReq.setOrderClient(OrderClientType.POS.getIndex());
        com.freemud.application.sdk.api.ordercenter.response.BaseResponse baseResponse = orderSdkService.cancelOrder(orderCancelReq,UUID.randomUUID().toString());
        if (!Objects.equals(SUCCESS,baseResponse.getCode())){
            return ResponseUtil.error(ResponseResult.PARTNER_ORDER_CANCEL_ERROR);
        }
        // 储值卡退款
        OrderExtInfoDto orderExtInfoDto = JSONObject.parseObject(order.getExtInfo(), OrderExtInfoDto.class) == null ? new OrderExtInfoDto() : JSONObject.parseObject(order.getExtInfo(), OrderExtInfoDto.class);
        //支付退款
        if (orderExtInfoDto != null  && orderExtInfoDto.getSvcAmount()!= null && StringUtils.isNotEmpty(orderExtInfoDto.getSvcCardCode())  && order.getPayStatus() > 1){
            PayRefundResponse refundResponse = paymentHandle.newOrderRefund(order, orderExtInfoDto);
            if(ObjectUtils.equals(PayRefundStatus.SUCCESS,refundResponse.getPayRefundStatus())){
                return ResponseUtil.error(ResponseResult.PARTNER_ORDER_CANCEL_ERROR);
            }
        }
        return ResponseUtil.success();
    }
}
