package cn.freemud.management.service;

import cn.freemud.base.entity.BaseResponse;
import cn.freemud.base.log.LogTreadLocal;
import cn.freemud.management.util.CommonResp;
import com.freemud.application.sdk.api.deliverycenter.dto.CreateDeliveryOrderRequestDto;
import com.freemud.application.sdk.api.deliverycenter.response.CreateDeliveryOrderResponseDto;
import com.freemud.application.sdk.api.deliverycenter.service.DeliveryService;
import com.freemud.application.sdk.api.storecenter.request.StoreInfoRequest;
import com.freemud.application.sdk.api.storecenter.response.StoreResponse;
import com.freemud.application.sdk.api.storecenter.service.StoreCenterService;
import com.freemud.sdk.api.assortment.order.enums.OrderSourceType;
import com.freemud.sdk.api.assortment.order.response.order.QueryOrdersResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Service
public class OrderDeliveryService {

    @Autowired
    private StoreCenterService storeCenterService;
    @Autowired
    private DeliveryService deliveryService;
    @Autowired
    private OrderBaseService orderBaseService;

    @Value("${saas.order.delivery.callBackUrl}")
    private String deliveryCallBackUrl;

    private int SUCCESS = 100;
    private String delivery_type_self = "self";

    public BaseResponse createDelivery(String orderCode) {
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(orderCode);
        return CommonResp.success(this.createDelivery(orderBean));
    }

    public Boolean createDelivery(QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //判断外卖单
        if (1 != orderBean.getType()) {
            return true;
        }
        if (!orderBean.getSource().equalsIgnoreCase(OrderSourceType.SAAS.getCode())) {
            return true;
        }
        //配送信息为空，则不创建配送单
        if (orderBean.getAddInfo() == null || orderBean.getAddInfo().getDeliveryType() == null) {
            return true;
        }
        //自配送不创建配送单
        if (delivery_type_self.equalsIgnoreCase(orderBean.getAddInfo().getDeliveryType())) {
            return true;
        }
        StoreInfoRequest storeInfoRequestDto = new StoreInfoRequest(orderBean.getCompanyId(), orderBean.getShopId());
        StoreResponse storeInfo = storeCenterService.getStoreInfo(storeInfoRequestDto, LogTreadLocal.getTrackingNo());
        if (SUCCESS != storeInfo.getStatusCode() || storeInfo.getBizVO() == null) {
            return false;
        }

        // 调用配送系统创建配送单
        CreateDeliveryOrderRequestDto requestDto = buildDeliveryOrderRequestDto(orderBean, storeInfo, deliveryCallBackUrl);
        CreateDeliveryOrderResponseDto deliveryResponse = deliveryService.deliveryOrderAdd(requestDto, LogTreadLocal.getTrackingNo());

        //创建配送单失败，更新订单为异常单
        if (deliveryResponse == null || SUCCESS != deliveryResponse.getCode() || deliveryResponse.getData() == null) {
            String deliveryId = deliveryResponse != null && deliveryResponse.getData() != null ? deliveryResponse.getData().getDeliveryId() : "";
            orderBaseService.updateDeliveryAbnormal(orderBean.getCompanyId(), orderBean.getOid(), deliveryId, "");
            return false;
        }
        // 创建配送单成功
        orderBaseService.updateDeliverySuccess(orderBean.getOid(), deliveryResponse.getData().getDeliveryId(), "");
        return true;
    }

    /**
     * 创建配送单请求对象
     *
     * @param order
     * @param storeInfo
     * @param deliveryCallBackUrl
     * @return
     */
    private CreateDeliveryOrderRequestDto buildDeliveryOrderRequestDto(QueryOrdersResponse.DataBean.OrderBean order
            , StoreResponse storeInfo, String deliveryCallBackUrl) {

        CreateDeliveryOrderRequestDto deliveryOrderRequestDto = new CreateDeliveryOrderRequestDto();
        deliveryOrderRequestDto.setDepatchWeight(new BigDecimal(1));
        deliveryOrderRequestDto.setSerialNumber(order.getDaySeq().toString());
        deliveryOrderRequestDto.setOrderId(order.getOid());
        deliveryOrderRequestDto.setOrderRemark(order.getRemark());
        deliveryOrderRequestDto.setOrderTotalAmount(order.getAmount().intValue());
        deliveryOrderRequestDto.setOrderActualAmount(order.getAmount().intValue());
        deliveryOrderRequestDto.setPartnerId(order.getCompanyId());
        deliveryOrderRequestDto.setOrderChannel(order.getSource());
        deliveryOrderRequestDto.setStoreId(storeInfo.getBizVO().getStoreId());
        deliveryOrderRequestDto.setStoreCode(storeInfo.getBizVO().getStoreCode());
        StringBuffer address = new StringBuffer(storeInfo.getBizVO().getCity())
                .append(storeInfo.getBizVO().getProvince())
                .append(storeInfo.getBizVO().getRegion())
                .append(storeInfo.getBizVO().getAddress());
        deliveryOrderRequestDto.setStoreAddress(address.toString());
        deliveryOrderRequestDto.setStoreLng(new BigDecimal(storeInfo.getBizVO().getLongitude()));
        deliveryOrderRequestDto.setStoreLat(new BigDecimal(storeInfo.getBizVO().getLatitude()));
        deliveryOrderRequestDto.setStorePhone(storeInfo.getBizVO().getPhone());
        deliveryOrderRequestDto.setStoreName(order.getShopName());
        deliveryOrderRequestDto.setCityName(storeInfo.getBizVO().getCity());
        deliveryOrderRequestDto.setReceiverName(order.getUserName());
        deliveryOrderRequestDto.setReceiverPrimaryPhone(order.getPhone());
        deliveryOrderRequestDto.setReceiverAddress(order.getAddress());
        deliveryOrderRequestDto.setReceiverLng(new BigDecimal(order.getLongitude()));
        deliveryOrderRequestDto.setReceiverLat(new BigDecimal(order.getLatitude()));
        deliveryOrderRequestDto.setOrderTime(new Date(order.getGmtCreate()));
        deliveryOrderRequestDto.setCallbackUrl(deliveryCallBackUrl);
        // 预约单预计送到时间
        deliveryOrderRequestDto.setExpectTime(new Date(order.getGmtCreate()));
        deliveryOrderRequestDto.setIsPre(order.getGmtExpect() != null ? 1 : 0);
        List<CreateDeliveryOrderRequestDto.DeliveryProductInfo> productInfos = new ArrayList<>();
        order.getProductList().forEach(productList -> {
            if (productList.getPrice() > 0) {
                CreateDeliveryOrderRequestDto.DeliveryProductInfo deliveryProductInfo = new CreateDeliveryOrderRequestDto.DeliveryProductInfo();
                deliveryProductInfo.setProductCode(productList.getProductId());
                deliveryProductInfo.setProductName(productList.getProductName());
                deliveryProductInfo.setProductNumber(productList.getNumber());
                deliveryProductInfo.setProductPrice(productList.getPrice().intValue());
                productInfos.add(deliveryProductInfo);
            }
        });
        deliveryOrderRequestDto.setProductInfos(productInfos);
        return deliveryOrderRequestDto;
    }
}
