package cn.freemud.management.service.adapter;

import cn.freemud.base.entity.BaseResponse;
import cn.freemud.management.entities.dto.request.order.OrderManagerRequest;
import cn.freemud.management.entities.dto.response.order.OrderManagerResponse;
import cn.freemud.management.enums.OperateType;
import cn.freemud.management.enums.OrderStatus;
import cn.freemud.management.enums.RefundStatus;
import cn.freemud.management.enums.ResponseResult;
import cn.freemud.management.service.OrderBaseService;
import cn.freemud.management.service.handle.OrderVerifyHandle;
import cn.freemud.management.util.ResponseUtil;
import com.freemud.sdk.api.assortment.order.enums.OldOrderType;
import com.freemud.sdk.api.assortment.order.response.order.QueryOrdersResponse;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class OrderManagerAdapter {

    @Autowired
    private OrderVerifyHandle orderVerifyHandle;
    @Autowired
    private OrderBaseService orderBaseService;
    @Autowired
    private OrderBeanFactory orderBeanFactory;
    @Value("${mcCafe.partnerId}")
    private String macCafePartnerId;


    /**
     * 接单
     *
     * @param request
     * @return
     */
    public BaseResponse<OrderManagerResponse> orderConfirmTake(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        //订单接单参数校验
        String companyId = orderBean.getCompanyId();
        if (!macCafePartnerId.equals(companyId)) {
            BaseResponse verifyResponse = orderVerifyHandle.orderConfirmTakeVerify(orderBean, request);
            if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
                return verifyResponse;
            }
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderConfirm(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     * 制作完成
     * @param request
     * @return
     */
    public BaseResponse<OrderManagerResponse> orderDone(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.orderDoneVerify(orderBean, request);
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderDone(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     *  送出
     * @param request
     * @return
     */
    public BaseResponse<OrderManagerResponse> orderDelivery(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        /*****兼容爱马哥，送出即完成start******/
        if (!OldOrderType.TAKE_OUT.getCode().equals(orderBean.getType())) {
            request.setOperateType(OperateType.ORDER_AFFIRM.getOpType());
            return orderAffirm(request, orderBean);
        }
        /*****容爱马哥，送出即完成end******/
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.orderDeliveryVerify(orderBean, request);
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return StringUtils.equals(verifyResponse.getCode(), ResponseResult.ORDER_IS_COMPLETE_CAN_NOT_DELIVERY.getCode()) ? handleResponse(request,orderBean,ResponseUtil.success()) : handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderDelivery(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     * 完成
     * @param request
     * @return
     */
    public BaseResponse<OrderManagerResponse> orderAffirm(OrderManagerRequest request,QueryOrdersResponse.DataBean.OrderBean orderBean) {
        if (orderBean==null){
            //查询订单
            orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        }
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.orderAffirmVerify(orderBean, request);
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderAffirm(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     * 拒单
     *
     * @param request
     * @return
     */
    public BaseResponse orderReject(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        //订单接单参数校验
        BaseResponse verifyResponse;
        if (!macCafePartnerId.equals(orderBean.getCompanyId())) {
            verifyResponse = orderVerifyHandle.orderRejectVerify(orderBean, request);
        } else {
            verifyResponse = orderVerifyHandle.mcCafeOrderRejectVerify(orderBean, request);
        }
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderReject(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     * 同意退款
     * @param request
     * @return
     */
    public BaseResponse orderAgreeRefund(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.orderAgreeRefund(orderBean, request);
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderAgreeRefund(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     * 同意退款或者取消订单
     */
    public BaseResponse orderPayRefund(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        if(CollectionUtils.isNotEmpty(orderBean.getRefundList())){
            //订单接单参数校验
            BaseResponse verifyResponse = orderVerifyHandle.orderAgreeRefundMCCafe(orderBean, request);
            if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
                return handleResponse(request,orderBean,verifyResponse);
            }
            BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderAgreeRefund(request, orderBean);
            return handleResponse(request,orderBean,orderManagerResponse);
        }else {
            BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderCancel(request, orderBean);
            return handleResponse(request,orderBean,orderManagerResponse);
        }
    }

    /**
     * 拒绝退款
     * @param request
     * @return
     */
    public BaseResponse orderRejectRefund(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.orderRejectRefund(orderBean, request);
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderRejectRefund(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }

    /**
     * 拒绝退款
     * @param request
     * @return
     */
    public BaseResponse mcCafeOrderRejectRefund(OrderManagerRequest request) {
        //查询订单
        QueryOrdersResponse.DataBean.OrderBean orderBean = orderBaseService.getByOrderCode(request.getOrderId());
        //订单接单参数校验
        BaseResponse verifyResponse = orderVerifyHandle.mcCafeOrderRejectRefund(orderBean, request);
        if (ObjectUtils.notEqual(verifyResponse.getCode(), ResponseResult.SUCCESS.getCode())) {
            return handleResponse(request,orderBean,verifyResponse);
        }
        BaseResponse<OrderManagerResponse> orderManagerResponse = orderBeanFactory.getOrderManagerService(orderBean).orderRejectRefund(request, orderBean);
        return handleResponse(request,orderBean,orderManagerResponse);
    }


    private BaseResponse<OrderManagerResponse> handleResponse(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean,BaseResponse result) {
        if (orderBean==null){
            return result;
        }
        RefundStatus refundStatus = CollectionUtils.isNotEmpty(orderBean.getRefundList()) ? RefundStatus.getByCode(orderBean.getRefundList().get(0).getStatus()) : null;
        OrderStatus orderStatus = OrderStatus.getByCode(orderBean.getStatus());
        if (ObjectUtils.equals(ResponseResult.SUCCESS.getCode(),result.getCode())){
            OperateType byOpType = OperateType.getByOpType(request.getOperateType());
            if (byOpType != null) {
                switch (byOpType) {
                    case ORDER_CONFIRM:
                        orderStatus = OrderStatus.RECEIPT;
                        break;
                    case ORDER_DONE:
                        orderStatus = OrderStatus.COMPLETE_MAKE;
                        break;
                    case ORDER_SEND:
                        orderStatus = OrderStatus.DISTRIBUTION;
                        break;
                    case ORDER_AFFIRM:
                        orderStatus = OrderStatus.COMPLETE;
                        break;
                    case ORDER_REJECT:
                    case ORDER_AGREE_REFUND:
                        orderStatus = OrderStatus.CANCEL;
                        refundStatus = RefundStatus.COMPLETE_REFUND;
                        break;
                    case ORDER_REJECT_REFUND:
                        refundStatus = RefundStatus.REFUSE_REFUND;
                        break;
                    default:
                }
            }
            return ResponseUtil.success(new OrderManagerResponse(orderStatus,refundStatus));
        }
        result.setResult(new OrderManagerResponse(orderStatus,refundStatus));
        return result;
    }
}
