/**
 * All rights Reserved, Designed By www.freemud.cn
 *
 * @Title: OrderCallBackMQService
 * @Package cn.freemud.amp.service
 * @Description:
 * @author: ping.wu
 * @date: 2019/4/16 14:29
 * @version V1.0
 * @Copyright: 2019 www.freemud.cn Inc. All rights reserved.
 * 注意：本内容仅限于上海非码科技内部传阅，禁止外泄以及用于其他的商业目
 */
package cn.freemud.amp.service;

import cn.freemud.amp.config.OrderCallBackConfig;
import cn.freemud.amqp.Header;
import cn.freemud.amqp.MQAction;
import cn.freemud.amqp.MQMessage;
import cn.freemud.amqp.MQService;
import cn.freemud.constant.RedisKeyConstant;
import cn.freemud.entities.dto.OrderExtInfoDto;
import cn.freemud.entities.dto.OrderStatusChangeRequestDto;
import cn.freemud.entities.dto.UserLoginInfoDto;
import cn.freemud.entities.dto.WechatReportOrderDto;
import cn.freemud.redis.RedisCache;
import com.alibaba.fastjson.JSONObject;
import com.freemud.api.assortment.datamanager.entity.db.AssortmentOpenPlatformWxapp;
import com.freemud.api.assortment.datamanager.manager.AssortmentOpenPlatformWxappManager;
import com.freemud.application.sdk.api.log.ApiLog;
import com.freemud.application.sdk.api.ordercenter.entities.vo.OrderCallBackRequestVo;
import com.freemud.application.sdk.api.ordercenter.enums.NewOrderStatus;
import com.freemud.application.sdk.api.ordercenter.enums.OrderClientType;
import com.freemud.application.sdk.api.ordercenter.enums.PayChannelType;
import com.freemud.application.sdk.api.ordercenter.enums.PayType;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.OrderInfoReqs;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.OrderItemResp;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.OrderPayItemResp;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.OrderSettlementResp;
import com.freemud.application.sdk.api.ordercenter.util.LogUtil;
import com.freemud.sdk.api.assortment.orderdistributor.request.DisRequest;
import com.freemud.sdk.api.assortment.orderdistributor.service.OrderDistributionService;
import com.google.common.collect.Lists;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;

@Component
public class OrderCallBackMQService {
    @Autowired
    private MQService mqService;

    @Autowired
    private OrderDistributionService orderDistributionService;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private AssortmentOpenPlatformWxappManager openPlatformWxappManager;
    @Value("${saas.order.invoice.report.partnerIds}")
    private String invoicePartnerIds;


    private static final String backOrdersChangeOrderStatusConsumerQueue = "program.backorders_change_order_status_consumer_queue";
    private static final String backOrdersNotifyActivityExchange = "program.backorders_notify_activity_exchange";
    @Autowired
    private LogUtil logUtil;

    public void sendOrderMQ(OrderCallBackRequestVo body) {
        OrderInfoReqs orderInfoReqs = getOrderInfoReqs(body);
        if (orderInfoReqs == null) {
            return;
        }
        // bizType=3 会员业务,bizType=4 积分商城业务,bizType=5 储值账户业务,bizType=9 礼品卡业务 （会员订单合并到支付宝微信渠道，保持之前业务一样，暂时不处理）
        if (Objects.equals(3, orderInfoReqs.getBizType())
                || Objects.equals(4, orderInfoReqs.getBizType())
                || Objects.equals(5, orderInfoReqs.getBizType())
                || Objects.equals(9, orderInfoReqs.getBizType())) {
            return;
        }
        //发送订单回调
        Header header = new Header(MQAction.UPDATE.getAction(), "call-back-service-api",
                OrderCallBackConfig.QUEUE_ROUTING_KEY, OrderCallBackConfig.QUEUE_NAME);
        MQMessage<OrderCallBackRequestVo> message = new MQMessage<>(header, body);
        mqService.convertAndSend(OrderCallBackConfig.EXCHANGE_NAME, OrderCallBackConfig.QUEUE_ROUTING_KEY, message);
        //订单状态变更通知开放平台
        Boolean filterPushOrder = filterPushOrder(body, orderInfoReqs);
        if (filterPushOrder) {
            orderDistributionService.pushOrder(convent2DisRequest(body, orderInfoReqs.getOrderState()));
        }
        //微信数据上传（创建新订单&支付成功时上传）
        sendWechatReportMQ(body);
        //上报发票信息
        sendInvoiceReportMQ(body);
        //未支付超时关单，发送队列冲正积分、券码等,停车场订单走单独逻辑
        if (checkParkingOrders(orderInfoReqs)) {
            revertParkingOrders(body);
        } else {
            revert(body);
        }

        //  task-center-order-consumer服务 订单状态处理中心queue 优化原有相同queue代码
        this.sendOrderStatusProcessCenterMQ(body);

        //支付宝订单状态变更上报
        sendzfbOrderMQ(body);
    }

    /**
     * 冲正库存、券码等
     *
     * @param body
     */
    public void revert(OrderCallBackRequestVo body) {
        if (ObjectUtils.notEqual(body.getDelayMsgEventType(), 2)) {
            return;
        }
        String redisKey = "kgd:cancel_payment_oid:" + body.getOrderCode();
        //取消支付订单缓存,
        Object value = redisCache.getValue(redisKey);
        if (value != null) {
            return;
        }
        ApiLog.infoMessage("activityReverse start orderCode:{}", body.getOrderCode());
        String orderCode = body.getOrderCode();
        OrderStatusChangeRequestDto requestDto = new OrderStatusChangeRequestDto();
        requestDto.setOid(orderCode);
        //2：未支付
        requestDto.setOrderStatus(2);
        requestDto.setPartnerId(body.getPartnerId());
        try {
            Header header = new Header(MQAction.INSERT.getAction(), "backOrdersStatusChange", orderCode, backOrdersChangeOrderStatusConsumerQueue);
            MQMessage<OrderStatusChangeRequestDto> message = new MQMessage<>(header, requestDto);
            mqService.convertAndSend(backOrdersNotifyActivityExchange, backOrdersChangeOrderStatusConsumerQueue, message);
            ApiLog.infoMessage("activityReverse end orderCode:{}", body.getOrderCode());
        } catch (Exception e) {
            ApiLog.infoMessage("activityReverse Error orderCode:{} e:{}", e, body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
        }
    }

    private void sendWechatReportMQ(OrderCallBackRequestVo body) {
        try {
            OrderInfoReqs orderInfoReqs = getWechatReportOrderInfoReqs(body);
            if (orderInfoReqs != null) {
                OrderExtInfoDto orderExtInfoDto = JSONObject.parseObject(orderInfoReqs.getExtInfo(), OrderExtInfoDto.class);
                UserLoginInfoDto userLoginInfoDto = getUserLoginInfoBySessionId(orderExtInfoDto.getSessionId());
                WechatReportOrderDto wechatReportOrderDto = getWechatReportOrderDto(orderInfoReqs, orderExtInfoDto, userLoginInfoDto);
                Header header = new Header(MQAction.UPDATE.getAction(), "call-back-service-api",
                        OrderCallBackConfig.QUEUE_ROUTING_KEY, OrderCallBackConfig.QUEUE_NAME);
                MQMessage<WechatReportOrderDto> message = new MQMessage<>(header, wechatReportOrderDto);
                mqService.convertAndSend(OrderCallBackConfig.TOPIC_EXCHANGE_NAME, OrderCallBackConfig.WECHAT_ORDER_REPORT_QUEUE_ROUTING_KEY, message);
            }
            ApiLog.infoMessage("sendWechatReportMQ end orderCode:{}", body.getOrderCode());
        } catch (Exception e) {
            ApiLog.infoMessage("sendWechatReportMQ Error orderCode:{} e:{}", body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
        }
    }

    private DisRequest convent2DisRequest(OrderCallBackRequestVo orderBean, Integer orderState) {
        DisRequest disRequest = new DisRequest();
        disRequest.setExtJson(orderBean.getContent());
        disRequest.setStatus(orderState.toString());
        disRequest.setPartnerId(orderBean.getPartnerId());
        disRequest.setStoreCode(orderBean.getStoreId());
        disRequest.setOrderCode(orderBean.getOrderCode());
        disRequest.setVer("2");
        disRequest.setOperateType(orderBean.getOperateType().toString());
        return disRequest;
    }

    private OrderInfoReqs getOrderInfoReqs(OrderCallBackRequestVo body) {
        OrderInfoReqs orderInfoReqs = null;
        try {
            orderInfoReqs = JSONObject.parseObject(body.getContent(), OrderInfoReqs.class);
        } catch (Exception e) {
            ApiLog.infoMessage("getOrderInfoReqs parseObjectOrderInfoReqs Error,orderCode:{} e:{}", body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
            return null;
        }
        return orderInfoReqs;
    }

    private Boolean filterPushOrder(OrderCallBackRequestVo body, OrderInfoReqs orderInfoReqs) {
        // 订单所有状态正常变化通知通知开放平台
        if ((!ObjectUtils.equals("1", body.getMsgType()) && !ObjectUtils.equals("3", body.getMsgType())) || body.getOperateType() == null || body.getOperateType() == 0) {
            return false;
        }
        // 未支付订单不通知开放平台
        if (ObjectUtils.equals(1, orderInfoReqs.getPayState()) || ObjectUtils.equals(NewOrderStatus.PENDING_PAYMENT.getIndex(), orderInfoReqs.getOrderState())) {
            return false;
        }
        // 点餐业务和卖券业务推送开放平台(卖券业务后续到虚拟门店感觉可以去除)
        if (ObjectUtils.equals(1, orderInfoReqs.getBizType()) || ObjectUtils.equals(6, orderInfoReqs.getBizType())) {
            return true;
        }
        //停车场订单关闭状态不通知开放平台
        if (checkParkingOrders(orderInfoReqs)) {
            return false;
        }
        return true;
    }

    private OrderInfoReqs getWechatReportOrderInfoReqs(OrderCallBackRequestVo body) {
        // 订单所有状态正常变化通知通知开放平台
        if (!ObjectUtils.equals("1", body.getMsgType()) || body.getOperateType() == null || body.getOperateType() == 2) {
            return null;
        }
        try {
            OrderInfoReqs orderInfoReqs = JSONObject.parseObject(body.getContent(), OrderInfoReqs.class);
            List<Integer> allowOrderClientList = Lists.newArrayList(OrderClientType.SAAS.getIndex(), OrderClientType.ALIPAY.getIndex());
            if (orderInfoReqs != null && allowOrderClientList.contains(orderInfoReqs.getOrderClient()) && Objects.equals(PayType.ONLINE_PAY.getCode(),orderInfoReqs.getPayType())&&
                    ((body.getOperateType() == 0 && NewOrderStatus.PENDING_PAYMENT.getIndex().equals(orderInfoReqs.getOrderState())) ||
                            (body.getOperateType() == 1 && NewOrderStatus.PLACE_AN_ORDER.getIndex().equals(orderInfoReqs.getOrderState())))) {
                return orderInfoReqs;
            }
        } catch (Exception e) {
            ApiLog.infoMessage("getWechatReportOrderInfoReqs parseObjectOrderInfoReqs Error,orderCode:{} e:{}", body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
            return null;
        }
        return null;
    }

    private UserLoginInfoDto getUserLoginInfoBySessionId(String sessionId) {
        String redisKey = RedisKeyConstant.SAAS_USER_INFO_SESSIONID_KEY_PREFIX + sessionId;
        Map<String, Object> map = redisCache.getValue(redisKey);
        UserLoginInfoDto userLoginInfoDto = new UserLoginInfoDto();
        userLoginInfoDto.setOpenId(String.valueOf(map.get("openid")));
        userLoginInfoDto.setWxAppid(String.valueOf(map.get("wxAppid")));
        userLoginInfoDto.setSessionKey(String.valueOf(map.get("session_key")));
        userLoginInfoDto.setMobile(String.valueOf(map.get("mobile")));
        userLoginInfoDto.setAppsecret(String.valueOf(map.get("appsecret")));
        userLoginInfoDto.setMemberId(String.valueOf(map.get("memberId")));
        userLoginInfoDto.setPartnerId(String.valueOf(map.get("partnerId")));
        userLoginInfoDto.setUnionId(String.valueOf(map.get("unionId")));
        userLoginInfoDto.setNewMemeber(Boolean.valueOf(map.get("newMember").toString()));
        userLoginInfoDto.setNickName(String.valueOf(map.get("nickName")));
        return userLoginInfoDto;
    }

    private WechatReportOrderDto getWechatReportOrderDto(OrderInfoReqs orderInfoReqs, OrderExtInfoDto orderExtInfoDto, UserLoginInfoDto userLoginInfoDto) {
        WechatReportOrderDto wechatReportOrderDto = new WechatReportOrderDto();
        wechatReportOrderDto.setAppId(userLoginInfoDto.getWxAppid());
        wechatReportOrderDto.setPartnerId(orderInfoReqs.getPartnerId());
        wechatReportOrderDto.setShopId(orderInfoReqs.getStoreId());
        wechatReportOrderDto.setSubMchId(orderExtInfoDto.getPlatformMchId());
        wechatReportOrderDto.setGmtCreate(Long.parseLong(orderInfoReqs.getCreateTime()));
        if (StringUtils.isBlank(orderExtInfoDto.getPlatformMchId())) {
            // 获取当前商户微信支付mchId
            AssortmentOpenPlatformWxapp wxApp = openPlatformWxappManager.findByPartnerIdAndWxappId(userLoginInfoDto.getPartnerId(), userLoginInfoDto.getWxAppid());
            wechatReportOrderDto.setSubMchId(wxApp == null ? "" : wxApp.getMchId());
        }
        // 混合支付
        Integer userAmount = 0;
        if (orderInfoReqs.getOrderPayItemCreateReqList() != null && orderInfoReqs.getOrderPayItemCreateReqList().size() > 0) {
            OrderPayItemResp orderPayItemReqs = orderInfoReqs.getOrderPayItemCreateReqList()
                    .stream()
                    .filter(s -> s.getPayChannelType().equals(PayChannelType.WECHAT.getIndex().intValue()) || s.getPayChannelType().equals(PayChannelType.ALIPAY.getIndex().intValue()))
                    .findFirst().orElse(null);
            if (orderPayItemReqs != null) {
                userAmount = orderPayItemReqs.getPayAmount().intValue();
            }
        }
        wechatReportOrderDto.setPayVoucher(orderInfoReqs.getPayRequestNo());
        wechatReportOrderDto.setSubAppId(userLoginInfoDto.getWxAppid());
        wechatReportOrderDto.setOutShopNo(orderInfoReqs.getStoreId());
        wechatReportOrderDto.setSubOpenId(userLoginInfoDto.getOpenId());
        wechatReportOrderDto.setLoginToken(userLoginInfoDto.getSessionKey());
        wechatReportOrderDto.setUserAmount(userAmount > 0 ? userAmount : orderInfoReqs.getSettlementAmount().intValue());
        wechatReportOrderDto.setOutOrderNo(orderInfoReqs.getOrderCode());
        wechatReportOrderDto.setTransactionId(orderExtInfoDto.getPaid_no());
        wechatReportOrderDto.setOutTradeNo(orderExtInfoDto.getPayTransId());

        // 计算优惠金额
        int discountAmount = 0;
        List<OrderSettlementResp> accountList = orderInfoReqs.getOrderSettlementDetailList();
        if (accountList != null && !CollectionUtils.isEmpty(accountList)) {
            for (OrderSettlementResp accountBean : accountList) {
                try {
                    Long price = accountBean.getSettlementAmount() == null ? 0 : accountBean.getSettlementAmount().longValue();
                    if (price < 0) {
                        discountAmount = discountAmount - price.intValue();
                    }
                } catch (Exception e) {
                }
            }
        }
        wechatReportOrderDto.setTotalAmount(discountAmount + (userAmount > 0 ? userAmount : orderInfoReqs.getSettlementAmount().intValue()));
        wechatReportOrderDto.setDiscountAmount(discountAmount);
        //订单状态 (2—用户下单；1—支付完成)
        wechatReportOrderDto.setStatus(NewOrderStatus.PENDING_PAYMENT.getIndex().equals(orderInfoReqs.getOrderState()) ? 2 : 1);
        // 菜品列表
        List<WechatReportOrderDto.DishInfo> dishList = new ArrayList<>();
        //【ID1040698】【数据上报】如果是套餐商品，只上报套餐，不上报明细， 解决数据上报商品数量(20)超限制报错
        for (OrderItemResp item : orderInfoReqs.getOrderItemList()) {
            if (StringUtils.isNotEmpty(item.getParentProductId()) && !"0".equals(item.getParentProductId())) {
                continue;
            }
            WechatReportOrderDto.DishInfo dishInfo = new WechatReportOrderDto.DishInfo();
            dishInfo.setOutDishNo(item.getProductId());
            dishInfo.setCount(item.getProductQuantity());
            dishInfo.setName(item.getProductName());
            dishInfo.setPrice(item.getProductPrice().intValue());
            dishInfo.setUrl(item.getProductPicUrl());
            dishList.add(dishInfo);
        }
        if (dishList.size() > 20) {
            dishList.subList(0, 20);
        }
        wechatReportOrderDto.setDishList(dishList);
        return wechatReportOrderDto;
    }

    /**
     * 上报指定商户已支付的订单到MQ
     * 只上报普通订单
     *
     * @param body
     */
    private void sendInvoiceReportMQ(OrderCallBackRequestVo body) {
        ApiLog.infoMessage("callback sendInvoiceReportMQ start orderCode:{}", body.getOrderCode());
        if (!ObjectUtils.equals("1", body.getMsgType()) || body.getOperateType() == null || body.getOperateType() == 2) {
            return;
        }
        OrderInfoReqs orderInfoReqs = JSONObject.parseObject(body.getContent(), OrderInfoReqs.class);
        if (orderInfoReqs == null
                || !Arrays.asList(invoicePartnerIds.split(",")).contains(orderInfoReqs.getPartnerId())
                || !NewOrderStatus.PLACE_AN_ORDER.getIndex().equals(orderInfoReqs.getOrderState())
                || orderInfoReqs.getBizType() != 1) {
            return;
        }
        try {
            Header header = new Header("report", "call-back-service-api",
                    OrderCallBackConfig.INVOICE_ROUTE_KEY, OrderCallBackConfig.INVOICE_QUEUE);

            MQMessage<OrderInfoReqs> message = new MQMessage<>(header, orderInfoReqs);
            mqService.convertAndSend(OrderCallBackConfig.TOPIC_EXCHANGE_NAME, OrderCallBackConfig.INVOICE_ROUTE_KEY, message);
            ApiLog.infoMessage("callback sendInvoiceReportMQ end orderCode={}", body.getOrderCode());
        } catch (Exception e) {
            ApiLog.infoMessage("callback sendInvoiceReportMQ Error orderCode:{} e:{}", body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
        }
    }

    /**
     * 冲正库存、券码等
     *
     * @param body
     */
    public void revertParkingOrders(OrderCallBackRequestVo body) {
        String orderCode = body.getOrderCode();
        OrderStatusChangeRequestDto requestDto = new OrderStatusChangeRequestDto();
        requestDto.setOid(orderCode);
        //2：未支付
        requestDto.setOrderStatus(2);
        requestDto.setPartnerId(body.getPartnerId());
        try {
            Header header = new Header(MQAction.INSERT.getAction(), "parkingOrderBackOrdersStatusChange", orderCode, backOrdersChangeOrderStatusConsumerQueue);
            MQMessage<OrderStatusChangeRequestDto> message = new MQMessage<>(header, requestDto);
            mqService.convertAndSend(backOrdersNotifyActivityExchange, backOrdersChangeOrderStatusConsumerQueue, message);
            ApiLog.infoMessage("revertParkingOrders parkingOrderBackOrdersStatusChange end orderCode:{}", orderCode);
        } catch (Exception e) {
            ApiLog.infoMessage("revertParkingOrders parkingOrderBackOrdersStatusChange Error orderCode:{} e:{}", body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
        }
    }

    /**
     * 检查是否符合停车场订单
     *
     * @param orderInfoReqs
     * @return
     */
    private boolean checkParkingOrders(OrderInfoReqs orderInfoReqs) {
        if (orderInfoReqs.getOrderClient().equals(OrderClientType.PARKING.getIndex()) && orderInfoReqs.getBizType() == 99 && orderInfoReqs.getPayState() == 3 && orderInfoReqs.getOrderState() == 7) {
            return true;
        }
        return false;
    }



    /**
     * 订单状态公共处理中心 queue
     *
     * @param body
     */
    private void sendOrderStatusProcessCenterMQ(OrderCallBackRequestVo body) {
        try {
            if (Objects.isNull(body) || StringUtils.isEmpty(body.getContent())) {
                return;
            }
            // 正常变化  订单正向操作
            if (body.getOperateType() == 1 && "1".equals(body.getMsgType())) {
                Header header = new Header(MQAction.UPDATE.getAction(), "call-back-service-api",
                        OrderCallBackConfig.ORDER_STATUS_PROCESS_CENTER_QUEUE_ROUTING_KEY, OrderCallBackConfig.ORDER_STATUS_PROCESS_CENTER_QUEUE);
                MQMessage<OrderCallBackRequestVo> message = new MQMessage<>(header, body);
                mqService.convertAndSend(OrderCallBackConfig.TOPIC_EXCHANGE_NAME, OrderCallBackConfig.ORDER_STATUS_PROCESS_CENTER_QUEUE_ROUTING_KEY, message);
                ApiLog.infoMessage("send order_status_process_center_queue end orderCode:{}", body.getOrderCode());
            }
        } catch (Exception e) {
            ApiLog.infoMessage("send order_status_process_center_queue Error orderCode:{} e:{}", body.getOrderCode(), ExceptionUtils.getFullStackTrace(e));
        }
    }

    private void sendzfbOrderMQ(OrderCallBackRequestVo body) {
        try {
            if (Objects.isNull(body) || StringUtils.isEmpty(body.getContent())) {
                return;
            }
            if (OrderClientType.ALIPAY.getIndex() != body.getOrderClient()) {
                return;
            }
            logUtil.info("sendzfbOrderMQ orderCode:{} ", body.getOrderCode());

            // 支付宝订单状态推送　tapd需求id 1032865
            if ("1".equals(body.getMsgType())) {
                Header header = new Header(MQAction.UPDATE.getAction(), "call-back-service-api",
                        null, OrderCallBackConfig.ZFB_ORDER_SEND_QUEUE);
                MQMessage<OrderCallBackRequestVo> message = new MQMessage<>(header, body);
                mqService.convertAndSend(OrderCallBackConfig.TOPIC_EXCHANGE_NAME, OrderCallBackConfig.ZFB_ORDER_SEND_QUEUE_ROUTING_KEY, message);
            }
        } catch (Exception e) {
            logUtil.error("", "sendzfbOrderMQ orderCode:{} ", e, body.getOrderCode());
        }
    }
}
