package com.freemud.sdk.api.assortment.shoppingcart.service.impl;

import ch.qos.logback.classic.Level;
import cn.freemud.base.entity.BaseResponse;
import cn.freemud.redis.RedisCache;
import com.freemud.application.sdk.api.base.SDKCommonBaseContextWare;
import com.freemud.application.sdk.api.couponcenter.online.service.FMActiveSdkService;
import com.freemud.application.sdk.api.log.ErrorLog;
import com.freemud.application.sdk.api.log.LogThreadLocal;
import com.freemud.sdk.api.assortment.shoppingcart.constant.CartResponseConstant;
import com.freemud.sdk.api.assortment.shoppingcart.constant.CommonsConstant;
import com.freemud.sdk.api.assortment.shoppingcart.constant.MealClearOperationEnum;
import com.freemud.sdk.api.assortment.shoppingcart.constant.RedisKeyConstant;
import com.freemud.sdk.api.assortment.shoppingcart.domain.CartAddItem;
import com.freemud.sdk.api.assortment.shoppingcart.domain.CartGoods;
import com.freemud.sdk.api.assortment.shoppingcart.domain.CartParamDto;
import com.freemud.sdk.api.assortment.shoppingcart.exception.ServiceException;
import com.freemud.sdk.api.assortment.shoppingcart.request.CheckCartRequest;
import com.freemud.sdk.api.assortment.shoppingcart.service.ShoppingCartBaseService;
import com.freemud.sdk.api.assortment.shoppingcart.util.CartResponseUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * All rights Reserved, Designed By www.freemud.com
 *
 * @version V1.0
 * @Title:
 * @Package: com.freemud.sdk.api.assortment.shoppingcart.service.impl
 * @Descripttion:
 * @author: pengfei.liu
 * @date: 2020/9/24
 * @Copyright: 2020 www.freemud.cn Inc. All rights reserved.
 * 注意：本内容仅限于上海非码科技内部传阅，禁止外泄以及用于其他的商业目.
 */
@Service("collageCartBaseService")
public class CollageCartBaseServiceImpl implements ShoppingCartBaseService {

    @Value("${coupon.app.id}")
    private String appId;
    @Autowired
    private FMActiveSdkService fmActiveSdkService;

    @Autowired
    private RedisCache redisCache;

    @Autowired
    private RedisTemplate redisTemplate;

    @Override
    public BaseResponse<List<CartAddItem>> detailCart(CartParamDto cartParamDto, String trackingNo) {
        return null;
    }

    @Override
    public BaseResponse<List<CartGoods>> getCartGoodsList(CartParamDto cartParamDto, String trackingNo) {

        doCheck(cartParamDto);
        // 构建缓存key
        String collageCartHashKey = genCollageCarHashKey(cartParamDto.getPartnerId(), cartParamDto.getStoreId(), cartParamDto.getUserId());
        BoundHashOperations<String, String, CartGoods> operations = redisTemplate.boundHashOps(collageCartHashKey);
        List<CartGoods> cartGoodsList = operations.values();
        //对创建时间进行排序
        cartGoodsList.sort(Comparator.comparingLong(CartGoods::getCreateTimeMili));
        return CartResponseUtil.success(cartGoodsList);
    }

    private void doCheck(CartParamDto cartParamDto) {
        if (StringUtils.isEmpty(cartParamDto.getStoreId()) || StringUtils.isEmpty(cartParamDto.getUserId())) {
            //门店和发起人不能为空
            throw new ServiceException(CartResponseConstant.FAIL, "门店号和发起人不能为空");
        }
    }

    /**
     * 构建拼单购物车缓存key
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    private String genCollageCarHashKey(String partnerId, String storeId, String userId) {
        return MessageFormat.format(RedisKeyConstant.COLLAGE_CART_HASH_KEY, partnerId, storeId, userId);
    }


    @Override
    public BaseResponse<List<CartGoods>> setCartGoodsList(CartParamDto cartParamDto, String trackingNo) {
        try {
            String redisKey = genCollageCarHashKey(cartParamDto.getPartnerId(), cartParamDto.getStoreId(), cartParamDto.getUserId());
            BoundHashOperations<String, String, CartGoods> operations = redisTemplate.boundHashOps(redisKey);
            this.clear(cartParamDto, trackingNo);
            Map<String, CartGoods> cartGoodsMap = cartParamDto.getCartGoodsList().parallelStream()
                    .filter(k -> org.apache.commons.lang.StringUtils.isNotEmpty(k.getCartGoodsUid()))
                    .collect(Collectors.toMap(CartGoods::getCartGoodsUid, Function.identity(), (k1, k2) -> k1));
            operations.putAll(cartGoodsMap);
            return CartResponseUtil.success();
        } catch (Exception e) {
            ErrorLog.printErrorLog("assortment-shoppingcart-sdk", trackingNo, e.getMessage(), "setCartGoodsList", cartParamDto, e, Level.ERROR);
            return null;
        }
    }

    @Override
    public BaseResponse clear(CartParamDto cartParamDto, String trackingNo) {
        return null;
    }

    @Override
    public String getCouponAppId() {
        return null;
    }

    @Override
    public Map<String, Boolean> getCouponOrderWay(String partnerId, List<String> activityCodes, Integer orderTye, String trackingNo) {
        return null;
    }

    @Override
    public BaseResponse<CartGoods> getCartGoods(CartParamDto cartParamDto, String trackingNo) {
        return null;
    }

    @Override
    public BaseResponse<List<CartGoods>> updateGoodsQty(CartParamDto cartParamDto, String trackingNo) {
        return null;
    }

    /**
     * 获取当前购物车版本
     * 默认1
     * @param key
     * @return
     */
    public Integer getCartVersion(String key) {
        Integer value = redisCache.getValue(key);
        return value == null ? 1 : value;
    }

    /**
     * 构建拼单基础缓存key
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    private String genCollageBaseHashKey(String partnerId, String storeId, String userId) {
        return MessageFormat.format(RedisKeyConstant.COLLAGE_BASE_HASH_KEY, partnerId, storeId, userId);
    }

    /**
     * 获取拼单基础
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    public Map<String, String> getCollageBase(String partnerId, String storeId, String userId) {
        String collageBaseKey = genCollageBaseHashKey(partnerId, storeId, userId);
        return redisCache.hashGetAll(collageBaseKey);
    }

    /**
     * 构建拼单锁定状态缓存key
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    private String genCollageLockKey(String partnerId, String storeId, String userId) {
        return MessageFormat.format(RedisKeyConstant.COLLAGE_LOCK_HASH_KEY, partnerId, storeId, userId);
    }

    /**
     * 获取拼单的锁定状态
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    public Boolean getCollageLock(String partnerId, String storeId, String userId) {
        String collageLockKey = genCollageLockKey(partnerId, storeId, userId);
        return redisCache.getValue(collageLockKey);
    }

    /**
     * 构建拼单点餐位置缓存key
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    private String genCollageMemberHashKey(String partnerId, String storeId, String userId) {
        return MessageFormat.format(RedisKeyConstant.COLLAGE_MEMBER_KEY, partnerId, storeId, userId);
    }

    /**
     * 获取拼单点餐位
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    public List<String> getCollageMemberHashKey(String partnerId, String storeId, String userId,String sessionId) {
        String collageMemberHashKey = genCollageMemberHashKey(partnerId, storeId, userId);
        return redisCache.hashGet(collageMemberHashKey,sessionId);
    }

    /**
     * 构建拼单用户选好的状态缓存key
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    private String genCollageMemberStateHashKey(String partnerId, String storeId, String userId) {
        return MessageFormat.format(RedisKeyConstant.COLLAGE_MEMBER_STATE_KEY, partnerId, storeId, userId);
    }

    /**
     * 获取拼单用户选好的状态
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    public Boolean getCollageMemStateHashKey(String partnerId, String storeId, String userId,String sessionId) {
        String collageMemberHashKey = genCollageMemberStateHashKey(partnerId, storeId, userId);
        return redisCache.hashGet(collageMemberHashKey,sessionId);
    }

    /**
     * 更新拼单人员缓存购物车行
     * @param cartGoodsUids
     * @param partnerId
     * @param storeId
     * @param crateUserId
     * @param sessionId
     */
    public void updateCollageCartGoodsUids(List<String> cartGoodsUids,String partnerId,String storeId,String crateUserId,String sessionId){
        String collageMemberKey = genCollageMemberHashKey(partnerId, storeId, crateUserId);
        redisCache.hashPut(collageMemberKey,sessionId,cartGoodsUids);
    }

    /**
     * 构建拼单购物车版本号缓存key
     * @param partnerId
     * @param storeId
     * @param userId
     * @return
     */
    private String genCollageCarVerKey(String partnerId, String storeId, String userId) {
        return MessageFormat.format(RedisKeyConstant.COLLAGE_CART_VER_KEY, partnerId, storeId, userId);
    }

    /**
     * 增加购物车版本
     *
     * @param partnerId   商户号
     * @param storeId     门店号
     * @param userId 用户id
     * @return
     */
    public Integer incrementCartVersion(String partnerId, String storeId, String userId) {
        return redisCache.opsForValue().increment(genCollageCarVerKey(partnerId, storeId, userId), 1).intValue();
    }

































}
