package cn.freemud.config;

import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import java.util.*;

/**
 * @author Clover.z
 * @version 1.0.0
 * @since 1.0.0
 */
@Data
@Configuration
@ConfigurationProperties(prefix = "push-order")
public class PushOrderConfig {

    /**
     * 不推送开放平台的商户和门店配置
     * 格式： 商户号:门店号,门店号,...;商户号:门店号,门店号,...;...
     * 如果只配置了商户号， 则表示全门店都不推送
     */
    private String config;

    /**
     * 不推送开放平台的商户门店信息
     * key:商户号
     */
    private Map<String, List<String>> notPushMap;


    public synchronized void initMap() {
        if (null == notPushMap) {
            notPushMap = new HashMap<>();
            if (StringUtils.isBlank(config)) return;
            String[] cfgList = config.split(";");
            for (String cfg : cfgList) {
                //格式必须是 商户号:门店号
                String[] s = cfg.split(":");
                if (s.length == 1) {
                    //全商户
                    notPushMap.put(s[0], new ArrayList<>());
                } else {
                    // 不满足格式直接忽略
                    if (s.length != 2) continue;
                    notPushMap.put(s[0], Arrays.asList(s[1].split(",")));
                }
            }
        }
    }


    /**
     * 判断门店的订单 是否需要推送三方
     * @param partnerId 商户号
     * @param storeCode 门店号
     * @return 是否需要推送
     */
    public boolean needPushOrder(String partnerId, String storeCode) {
        // 未配置， 聚合层全部不推送，由基础服务统一推送开放平台
        if (StringUtils.isBlank(config)) return false;

        if (null == notPushMap) initMap();
        // 商户号是否在推送黑名单配置中
        List<String> storeList = notPushMap.get(partnerId);
        // 配置了商户，没有配置门店， 表示所有门店都不推送
        if (CollectionUtils.isEmpty(storeList)) return false;
        return !storeList.contains(storeCode);
    }

}
