package cn.freemud.management.service.impl;

import cn.freemud.base.entity.BaseResponse;
import cn.freemud.management.entities.dto.request.order.OrderManagerRequest;
import cn.freemud.management.entities.dto.response.order.OrderManagerResponse;
import cn.freemud.management.entities.dto.response.pay.PayRefundResponse;
import cn.freemud.management.enums.OperateType;
import cn.freemud.management.enums.RefundStatus;
import cn.freemud.management.intercept.OrderServiceException;
import cn.freemud.management.service.OrderBaseService;
import cn.freemud.management.service.OrderManagerService;
import cn.freemud.management.service.handle.*;
import cn.freemud.management.util.ResponseUtil;
import com.freemud.application.sdk.api.storecenter.request.StoreInfoRequest;
import com.freemud.application.sdk.api.storecenter.response.StoreResponse;
import com.freemud.application.sdk.api.storecenter.service.StoreCenterService;
import com.freemud.sdk.api.assortment.order.response.order.QueryOrdersResponse;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * All rights Reserved, Designed By www.freemud.cn
 *
 * @version V1.0
 * @Title: rrr
 * @Package cn.freemud.management.service.impl
 * @Description: saas订单履单操作
 * @author: shuhu.hou
 * @date: 2020/3/26 16:01
 * @Copyright: 2020 www.freemud.cn Inc. All rights reserved.
 * 注意：本内容仅限于上海非码科技内部传阅，禁止外泄以及用于其他的商业目
 */
@Service
public class SaasOrderMangerServiceImpl implements OrderManagerService {

    @Autowired
    private SaasOrderHandle saasOrderHandle;
    @Autowired
    private OrderBaseService orderBaseService;
    @Autowired
    private StoreCenterService storeCenterService;
    @Autowired
    private PaymentHandle paymentHandle;
    @Autowired
    private StockHandle stockHandle;
    @Autowired
    private DeliveryHandle deliveryHandle;

    /**
     * 接单
     *
     * @param request   请求参数
     * @param orderBean 订单实体
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderConfirm(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单接单
        saasOrderHandle.orderConfirm(request, orderBean);
        //获取门店信息
        StoreResponse storeInfo = storeCenterService.getStoreInfo(new StoreInfoRequest(orderBean.getCompanyId(), orderBean.getShopId()), "");
        // 创建配送单
        deliveryHandle.createDelivery(orderBean, request, storeInfo);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_CONFIRM, request.getReason());
        return ResponseUtil.success(new OrderManagerResponse(storeInfo.getBizVO().getOrderPrintConfig(), orderBean.getGmtExpect(),
                CollectionUtils.isEmpty(orderBean.getRefundList()) ? null : RefundStatus.getByCode(orderBean.getRefundList().get(0).getStatus())));
    }

    /**
     * 制作完成
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderDone(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单制作完成
        saasOrderHandle.orderDone(request, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_DONE, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 订单配送
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderDelivery(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单配送
        saasOrderHandle.orderDelivery(request, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_SEND, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 订单完成
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderAffirm(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        // 订单完成
        saasOrderHandle.orderAffirm(request, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_AFFIRM, request.getReason());
        return ResponseUtil.success();
    }

    /**
     * 拒单
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderReject(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //支付退款
        PayRefundResponse refundResponse = new PayRefundResponse();
        try {
            if(null != orderBean.getOrderPayItem() && orderBean.getOrderPayItem().size()>0){
                refundResponse = paymentHandle.multiRefund(orderBean);
            }else{
                refundResponse = paymentHandle.refund(request, orderBean);
            }
        } catch (OrderServiceException orderEx){ // 爱马哥兼容，不抛错
            return ResponseUtil.error(orderEx.getResult());
        }
        //订单拒单
        saasOrderHandle.orderReject(request, refundResponse, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_REJECT, request.getReason());
        //todo  可以考虑基于订单回调异步实现
        //冲正库存
        stockHandle.revert(request,orderBean);
        return ResponseUtil.success();
    }



    /**
     * 同意退款
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderAgreeRefund(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //取消配送单
        deliveryHandle.cancelDelivery(orderBean,request);
        //支付退款
        PayRefundResponse refundResponse = new PayRefundResponse();
        try {
            if(null != orderBean.getOrderPayItem() && orderBean.getOrderPayItem().size()>0){
                refundResponse = paymentHandle.multiRefund(orderBean);
            }else{
                refundResponse = paymentHandle.refund(request, orderBean);
            }
        } catch (OrderServiceException orderEx){ // 爱马哥兼容，不抛错
            return ResponseUtil.error(orderEx.getResult());
        }
        //订单同意退款
        saasOrderHandle.refundAgree(request, refundResponse, orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_AGREE_REFUND, request.getReason());
        //todo  可以考虑基于订单回调异步实现
        //冲正库存
        stockHandle.revert(request,orderBean);
        return ResponseUtil.success();
    }

    /**
     * 拒绝退款
     *
     * @param request
     * @param orderBean
     * @return
     */
    @Override
    public BaseResponse<OrderManagerResponse> orderRejectRefund(OrderManagerRequest request, QueryOrdersResponse.DataBean.OrderBean orderBean) {
        //拒绝退款
        saasOrderHandle.refundReject(request,orderBean);
        // 推送pos、微信消息中心
        orderBaseService.sendMessage(orderBean, OperateType.ORDER_REJECT_REFUND, request.getReason());
        return ResponseUtil.success();
    }

}
