package cn.freemud.management.service.impl;

import cn.freemud.base.entity.BaseResponse;
import cn.freemud.management.entities.dto.request.order.OrderManagerRequest;
import cn.freemud.management.entities.dto.response.order.OrderManagerResponse;
import cn.freemud.management.enums.OrderSource;
import cn.freemud.management.enums.OrderStatus;
import cn.freemud.management.enums.ResponseResult;
import cn.freemud.management.util.ResponseUtil;
import com.freemud.application.sdk.api.ordercenter.enums.OperateClient;
import com.freemud.sdk.api.assortment.order.response.order.QueryOrdersResponse;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.ObjectUtils;
import org.springframework.stereotype.Service;

/**
 * All rights Reserved, Designed By www.freemud.cn
 *
 * @version V1.0
 * @Title: OrderVerifyService
 * @Package cn.freemud.management.service.impl
 * @Description:
 * @author: shuhu.hou
 * @date: 2020/3/26 16:04
 * @Copyright: 2020 www.freemud.cn Inc. All rights reserved.
 * 注意：本内容仅限于上海非码科技内部传阅，禁止外泄以及用于其他的商业目
 */
@Service
public class OrderVerifyService {

    /**
     * 接单校验
     *
     * @param orderBean
     * @param request
     * @return
     */
    public BaseResponse orderConfirmTakeVerify(QueryOrdersResponse.DataBean.OrderBean orderBean, OrderManagerRequest request) {
        if (orderBean == null) {
            return ResponseUtil.error(ResponseResult.ORDER_NOT_EXIST);
        }
        if (!OrderStatus.TAKE_ORDER.getCode().equals(orderBean.getStatus())) {
            return ResponseUtil.error(ResponseResult.ORDER_CANNOT_CONFIRM,getOrderManagerResponse(orderBean));
        }
        return commonVerify(orderBean, request);
    }

    /**
     * 订单配送校验
     * @param orderBean
     * @param request
     * @return
     */
    public BaseResponse orderDeliveryVerify(QueryOrdersResponse.DataBean.OrderBean orderBean, OrderManagerRequest request) {
        if (orderBean == null) {
            return ResponseUtil.error(ResponseResult.ORDER_NOT_EXIST);
        }
        //已完成的订单不能配送
        if(OrderStatus.COMPLETE.getCode().equals(orderBean.getStatus())){
            return ResponseUtil.error(ResponseResult.ORDER_IS_COMPLETE_CAN_NOT_DELIVERY,getOrderManagerResponse(orderBean));
        }
        //订单已经发送过配送
        if(OrderStatus.DISTRIBUTION.getCode().equals(orderBean.getStatus())){
            return ResponseUtil.error(ResponseResult.DELIVERY_SEND_REPETITION,getOrderManagerResponse(orderBean));
        }
        //判断订单状态是否为已经申请退款订单
        if (orderBean.getHasRefund() != null
                && orderBean.getHasRefund()
                && (orderBean.getRefundList().get(orderBean.getRefundList().size() - 1).getStatus() == 1
                || orderBean.getRefundList().get(orderBean.getRefundList().size() - 1).getStatus() == 4)) {
            return ResponseUtil.error(ResponseResult.DELIVERY_SEND_APPLY_REFUND,getOrderManagerResponse(orderBean));
        }
        //外卖订单不是接单不能送出
        if (!OrderStatus.RECEIPT.getCode().equals(orderBean.getStatus())) {
            return ResponseUtil.error(ResponseResult.DELIVERY_SEND_STATUS_NOTAKE,getOrderManagerResponse(orderBean));
        }
        return commonVerify(orderBean, request);
    }


    /**
     * 订单制作完成校验
     * @param orderBean
     * @param request
     * @return
     */
    public BaseResponse orderDoneVerify(QueryOrdersResponse.DataBean.OrderBean orderBean, OrderManagerRequest request) {
        if (orderBean == null) {
            return ResponseUtil.error(ResponseResult.ORDER_NOT_EXIST);
        }
        //已制作完成的订单不能重复操作
        if (OrderStatus.COMPLETE_MAKE.getCode().equals(orderBean.getStatus())) {
            return ResponseUtil.error(ResponseResult.DONE_FAIL_REPETITION,getOrderManagerResponse(orderBean));
        }
        //订单非接单状态，不能制作完成
        if (!OrderStatus.RECEIPT.getCode().equals(orderBean.getStatus())) {
            return ResponseUtil.error(ResponseResult.DONE_FAIL_NOT_ACCEPT,getOrderManagerResponse(orderBean));
        }
        //判断订单状态是否为已经申请退款订单
        if (orderBean.getHasRefund() != null
                && orderBean.getHasRefund()
                && (orderBean.getRefundList().get(orderBean.getRefundList().size() - 1).getStatus() == 1
                || orderBean.getRefundList().get(orderBean.getRefundList().size() - 1).getStatus() == 4)) {
            return ResponseUtil.error(ResponseResult.DONE_FAIL_APPLY_REFUND,getOrderManagerResponse(orderBean));
        }
        return ResponseUtil.success();
    }

    /**
     * 订单完成校验
     * @param orderBean
     * @param request
     * @return
     */
    public BaseResponse orderAffirmVerify(QueryOrdersResponse.DataBean.OrderBean orderBean, OrderManagerRequest request) {
        if (orderBean == null) {
            return ResponseUtil.error(ResponseResult.ORDER_NOT_EXIST);
        }
        //判断订单是否已送达
        if (OrderStatus.COMPLETE.getCode().equals(orderBean.getStatus())) {
            return ResponseUtil.error(ResponseResult.DELIVERY_SEND_REPETITION,getOrderManagerResponse(orderBean));
        }
        //已取消的订单不能操作
        if(OrderStatus.CANCEL.getCode().equals(orderBean.getStatus())){
            return ResponseUtil.error(ResponseResult.ORDER_HAS_CANCELED,getOrderManagerResponse(orderBean));
        }
        //判断订单状态是否为已经申请退款订单
        if (orderBean.getHasRefund() != null
                && orderBean.getHasRefund()
                && (orderBean.getRefundList().get(orderBean.getRefundList().size() - 1).getStatus() == 1
                || orderBean.getRefundList().get(orderBean.getRefundList().size() - 1).getStatus() == 4)) {
            return ResponseUtil.error(ResponseResult.AFFIRM_FAIL_APPLY_REFUND,getOrderManagerResponse(orderBean));
        }
        //校验订单是否为配送状态
        if(isWaimai(orderBean) && ObjectUtils.notEqual(OrderStatus.DISTRIBUTION.getCode(),orderBean.getStatus())){
            return ResponseUtil.error(ResponseResult.ORDER_DELIVERY_FAILED,getOrderManagerResponse(orderBean));
        }
        return commonVerify(orderBean, request);
    }

    /**
     * 通用校验
     * @param orderBean
     * @param request
     * @return
     */
    private BaseResponse commonVerify(QueryOrdersResponse.DataBean.OrderBean orderBean, OrderManagerRequest request) {
        if (ObjectUtils.notEqual(orderBean.getCompanyId(), request.getPartnerId())) {
            return ResponseUtil.error(ResponseResult.ORDER_PARTNER_ID_NOT_MATCTH,getOrderManagerResponse(orderBean));
        }
        //开放平台不传入门店
        if (ObjectUtils.notEqual(request.getOperator(), OperateClient.OPEN_PLATFORM.getOpName()) && ObjectUtils.notEqual(orderBean.getShopId(), request.getStoreId())) {
            return ResponseUtil.error(ResponseResult.ORDER_STORE_ID_NOT_MATCTH,getOrderManagerResponse(orderBean));
        }
        return ResponseUtil.success();
    }

    /**
     * 是否是外卖
     * @param orderBean
     * @return
     */
    private boolean isWaimai(QueryOrdersResponse.DataBean.OrderBean orderBean){
        String source = orderBean.getSource();
        if (OrderSource.BDWM.getSource().equals(source) || OrderSource.JDWM.getSource().equals(source)
                || OrderSource.MTWM.getSource().equals(source) || OrderSource.ELEME.getSource().equals(source)){
            return true;
        }
        return false;
    }

    private OrderManagerResponse getOrderManagerResponse(QueryOrdersResponse.DataBean.OrderBean orderBean){
        Integer refundStatus= CollectionUtils.isEmpty(orderBean.getRefundList())?null:orderBean.getRefundList().get(0).getStatus();
        return new OrderManagerResponse(orderBean.getGmtExpect(),orderBean.getStatus(),refundStatus);
    }
}
