package cn.freemud.management.service.handle;

import cn.freemud.base.entity.BaseResponse;
import cn.freemud.management.adapter.BaseQueryOrderRequestAdapter;
import cn.freemud.management.entities.dto.request.order.OrderManagerRequest;
import cn.freemud.management.entities.dto.response.pay.PayRefundResponse;
import cn.freemud.management.enums.AfterSalesApplyClientEnum;
import cn.freemud.management.enums.ResponseResult;
import cn.freemud.management.intercept.OrderServiceException;
import cn.freemud.management.util.ResponseUtil;
import com.alibaba.fastjson.JSON;
import com.freemud.application.sdk.api.log.LogThreadLocal;
import com.freemud.application.sdk.api.ordercenter.entities.v1.OrderBeanV1;
import com.freemud.application.sdk.api.ordercenter.entities.vo.OrderCallBackRequestVo;
import com.freemud.application.sdk.api.ordercenter.enums.orderv1.RefundStatusV1;
import com.freemud.application.sdk.api.ordercenter.request.AfterSalesRefundFailReq;
import com.freemud.application.sdk.api.ordercenter.request.OrderModifyRelatingCodeReq;
import com.freemud.application.sdk.api.ordercenter.request.POSOrderOperationBaseReq;
import com.freemud.application.sdk.api.ordercenter.response.OrderBaseResp;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.OrderInfoReqs;
import com.freemud.application.sdk.api.ordercenter.response.orderInfo.QueryByCodeResponse;
import com.freemud.application.sdk.api.ordercenter.service.OrderSdkService;
import com.freemud.sdk.api.assortment.order.adapter.OrderSdkAdapter;
import com.freemud.sdk.api.assortment.order.entities.other.OtherParamDTO;
import com.freemud.sdk.api.assortment.order.enums.AfterSalesRefunStateEnum;
import com.freemud.sdk.api.assortment.order.enums.PayRefundStatus;
import com.freemud.sdk.api.assortment.order.request.order.BaseQueryOrderRequest;
import com.freemud.sdk.api.assortment.order.request.order.CancelOrderRequest;
import com.freemud.sdk.api.assortment.order.response.order.BaseOrderResponse;
import com.freemud.sdk.api.assortment.order.service.OrderCenterSdkService;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * All rights Reserved, Designed By www.freemud.cn
 *
 * @version V1.0
 * @Title: SaasOrderService
 * @Package cn.freemud.management.service.impl
 * @Description:
 * @author: shuhu.hou
 * @date: 2020/4/23 11:30
 * @Copyright: 2020 www.freemud.cn Inc. All rights reserved.
 * 注意：本内容仅限于上海非码科技内部传阅，禁止外泄以及用于其他的商业目
 */
@Component
public class SaasOrderHandle {

    @Autowired
    private OrderSdkAdapter orderSdkAdapter;
    @Autowired
    private OrderCenterSdkService orderCenterSdkService;

    @Value("${saasDeliveryTimeOut}")
    private Integer wmDeliveryTimeOut;
    @Value("#{'${order.not.automatic.twist.partnerId}'.split(',')}")
    private List<String> notAutomaticTwistPartnerId;
    @Autowired
    private OrderSdkService orderSdkService;

    @Autowired
    private StoreHandle storeHandle;

    /**
     * 确认接单
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse orderConfirm(OrderManagerRequest request, OrderBeanV1 orderBean) {
        //订单接单
        OtherParamDTO dto = new OtherParamDTO();
        dto.setDispatchType(request.getDispatchType());
        dto.setDispatchTimeout(request.getDispatchTimeout());
        BaseQueryOrderRequest orderConfirmRequest = BaseQueryOrderRequestAdapter.getOrderConfirmRequest(request, orderBean, dto);

        orderSdkAdapter.convert2wmDeliveryTimeOut(orderConfirmRequest, this.wmDeliveryTimeOut);
        if (Objects.nonNull(request.getTimeout())) {
            orderConfirmRequest.setPreTimeout(request.getTimeout());
        }
        //合阔pos 商户订单状态不自动扭转
        // 2021-07-07 合阔自动
//        if (CollectionUtils.isNotEmpty(notAutomaticTwistPartnerId) && notAutomaticTwistPartnerId.contains(orderBean.getCompanyId())) {
//            orderConfirmRequest.setNotAutomaticTwist(true);
//        }

        POSOrderOperationBaseReq requestDto = new POSOrderOperationBaseReq();
        requestDto.setOrderCode(request.getOrderId());
        requestDto.setOperator(request.getOperator());
        requestDto.setDispatchType(request.getDispatchType());
        requestDto.setDispatchTimeout(request.getDispatchTimeout());
        requestDto.setPartnerId(request.getPartnerId());
        requestDto.setAcceptMode(orderConfirmRequest.getAcceptMode());
        // 引入门店基础服务 查询自动制作完成配置时间
        if (Objects.nonNull(orderConfirmRequest.getPreTimeout())) {
            requestDto.setTimeout(orderConfirmRequest.getPreTimeout());
        } else {
            requestDto = storeHandle.setTimeout(orderConfirmRequest, requestDto);
        }
        OrderBaseResp response = orderSdkService.acceptOrder(requestDto, LogThreadLocal.getTrackingNo());
        BaseOrderResponse confirmOrderResponse = orderSdkAdapter.convent2BaseOrderResponse(response);
        //处理1秒重复接单问题 订单服务接单返回20067重复操作，20018状态机不支持认为接单成功
        if ("20067".equals(confirmOrderResponse.getErrcode().toString()) || "20018".equals(confirmOrderResponse.getErrcode().toString()) || "50002".equals(confirmOrderResponse.getErrcode().toString())) {
            return ResponseUtil.success(confirmOrderResponse.getErrcode());
        }
        if (ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), confirmOrderResponse.getErrcode().toString())) {
            throw new OrderServiceException(ResponseResult.ORDER_ACCEPT_FAILED);
        }
        return ResponseUtil.success();
    }

    /**
     * 更新取餐码
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse updatePickUpGoodNo(OrderManagerRequest request, OrderBeanV1 orderBean) {
        // 更新取餐码
        OrderModifyRelatingCodeReq relatingCodeReq = new OrderModifyRelatingCodeReq();
        relatingCodeReq.setOrderCode(orderBean.getOid());
        relatingCodeReq.setOperator(request.getOperator());
        relatingCodeReq.setPartnerId(request.getPartnerId());
        relatingCodeReq.setPickUpGoodsNo(request.getPickUpGoodsNo());
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.updatePickUpGoodNo(relatingCodeReq, LogThreadLocal.getTrackingNo());
        if (ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.ORDER_UPDATE_PICKUP_FAILED);
        }
        return ResponseUtil.success();
    }

    /**
     * 订单制作完成
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse orderDone(OrderManagerRequest request, OrderBeanV1 orderBean) {
        BaseQueryOrderRequest orderDoneRequest = orderSdkAdapter.getOrderConfirmRequest(request.getOperator(), orderBean);
        //合阔pos 商户订单状态不自动扭转
        // 2021-07-07 合阔自动
//        if (CollectionUtils.isNotEmpty(notAutomaticTwistPartnerId) && notAutomaticTwistPartnerId.contains(orderBean.getCompanyId())) {
//            orderDoneRequest.setNotAutomaticTwist(true);
//        }
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.orderDone(orderDoneRequest);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.DONE_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 订单配送
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse orderDelivery(OrderManagerRequest request, OrderBeanV1 orderBean) {
        BaseQueryOrderRequest orderDeliveryRequest = orderSdkAdapter.getOrderConfirmRequest(request.getOperator(), orderBean);
        orderDeliveryRequest = orderSdkAdapter.convert2wmDeliveryTimeOut(orderDeliveryRequest, this.wmDeliveryTimeOut);
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.orderSend(orderDeliveryRequest);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.DELIVERY_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 订单完成
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse orderAffirm(OrderManagerRequest request, OrderBeanV1 orderBean) {
        BaseQueryOrderRequest orderAffirmRequest = orderSdkAdapter.getOrderConfirmRequest(request.getOperator(), orderBean);
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.orderAffirm(orderAffirmRequest);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.AFFIRM_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 拒单
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse orderReject(OrderManagerRequest request, PayRefundResponse refundResponse, OrderBeanV1 orderBean) {
        //作废订单
        CancelOrderRequest cancelOrderRequest = orderSdkAdapter.getCancelOrderRequest(request.getOperator(), request.getReason(), refundResponse.getPayRefundStatus(), refundResponse.getRefundId(), orderBean);
        cancelOrderRequest.setOperationClient(request.getOperationClient());
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.orderReject(cancelOrderRequest);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.AFFIRM_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 麦咖啡拒单
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse mcCafeOrderReject(OrderManagerRequest request, PayRefundResponse refundResponse, OrderBeanV1 orderBean) {
        //作废订单
        CancelOrderRequest cancelOrderRequest = orderSdkAdapter.getCancelOrderRequest(request.getOperator(), request.getReason(), refundResponse.getPayRefundStatus(), refundResponse.getRefundId(), orderBean);
        cancelOrderRequest.setReqRemark(request.getRemark());
        cancelOrderRequest.setOperationClient(request.getOperationClient());
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.mcCafeOrderReject(cancelOrderRequest, orderBean);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.AFFIRM_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 同意退款
     *
     * @param request
     * @param refundResponse
     * @param orderBean
     * @return
     */
    public BaseResponse refundAgree(OrderManagerRequest request, PayRefundResponse refundResponse, OrderBeanV1 orderBean) {
        BaseQueryOrderRequest refundAgreeRequest = orderSdkAdapter.getRefundAgreeRequest(refundResponse.getRefundId(), request.getReason(), request.getOperator(), orderBean);
        refundAgreeRequest.setRefundState(PayRefundStatus.SUCCESS.getCode().equals(refundResponse.getPayRefundStatus().getCode()) ? 4 : 2);
        refundAgreeRequest.setOperationClient(request.getOperationClient());
        BaseOrderResponse baseOrderResponse = null;
        if (PayRefundStatus.SUCCESS.equals(refundResponse.getPayRefundStatus())) {
            // fisherman 判断是否在退款中, 如果是退款中 调用退货完成
            if (orderBean.getAfterSalesOrderResp().getAfterSalesStatus().compareTo(AfterSalesRefunStateEnum.STATE_1.getIndex()) != 0) {
                baseOrderResponse = orderCenterSdkService.orderRefundComplete(refundAgreeRequest);
            } else {
                baseOrderResponse = orderCenterSdkService.refundComplete(refundAgreeRequest);
            }
        } else if (PayRefundStatus.RUNNING.equals(refundResponse.getPayRefundStatus()) && orderBean.getAfterSalesOrderResp().getAfterSalesStatus() != 4) {
            //退款中, 基础服务售后状态不能为4，等待商家再次发起同意退款
            baseOrderResponse = orderCenterSdkService.refundReturn(refundAgreeRequest);
        } else {
            return ResponseUtil.success();
        }
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.AGREE_REFUND_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 处理支付 退款回调 订单业务处理
     *
     * @param request
     * @param refundResponse
     * @param orderBean
     */
    public void refundAgreeByCallback(OrderManagerRequest request, PayRefundResponse refundResponse, OrderBeanV1 orderBean) {
        BaseQueryOrderRequest refundAgreeRequest = orderSdkAdapter.getRefundAgreeRequest(refundResponse.getRefundId(), request.getReason(), request.getOperator(), orderBean);
        refundAgreeRequest.setRefundState(PayRefundStatus.SUCCESS.getCode().equals(refundResponse.getPayRefundStatus().getCode()) ? 4 : 2);
        refundAgreeRequest.setOperationClient(request.getOperationClient());
        PayRefundStatus payRefundStatus = refundResponse.getPayRefundStatus();
        if (payRefundStatus.getCode().equals(PayRefundStatus.SUCCESS.getCode())) {
            // 退款成功  调用 -> /afterSales/v2/adultRefundComplete
            orderCenterSdkService.orderRefundComplete(refundAgreeRequest);
        } else {
            // 退款失败  只记录失败日志
            AfterSalesRefundFailReq refundFailReq = this.getAfterSalesRefundFailReq(orderBean,
                    "refundSuccessCallback",
                    AfterSalesApplyClientEnum.STATE_6.getIndex(),
                    refundResponse.getMessage());
            orderSdkService.refundFail(refundFailReq, LogThreadLocal.getTrackingNo());
        }

    }

    private AfterSalesRefundFailReq getAfterSalesRefundFailReq(OrderBeanV1 orderBean, String operator, Integer operationClient, String refundFailReason) {
        AfterSalesRefundFailReq refundFailReq = new AfterSalesRefundFailReq();
        refundFailReq.setOrderCode(orderBean.getOid());
        refundFailReq.setPartnerId(orderBean.getCompanyId());
        refundFailReq.setRefundFailReason(refundFailReason);
        refundFailReq.setAfterSalesCode(orderBean.getAfterSalesOrderResp().getAfterSalesCode());
        refundFailReq.setOperationClient(operationClient);
        refundFailReq.setOperator(operator);
        return refundFailReq;
    }
    /**
     * 麦咖啡同意退款
     *
     * @param request
     * @param refundResponse
     * @param orderBean
     * @return
     */
//    public BaseResponse mcCafeRefundAgree(OrderManagerRequest request, PayRefundResponse refundResponse, OrderBeanV1 orderBean) {
//        BaseQueryOrderRequest refundAgreeRequest = orderSdkAdapter.getRefundAgreeRequest(refundResponse.getRefundId(), request.getReason(), request.getOperator(), orderBean);
//        BaseOrderResponse baseOrderResponse = null;
//        for(int i=0;i<orderBean.getRefundList().size();i++) {
//
//            if (RefundStatusV1.APPLY_REFUND.getCodeV1().equals(orderBean.getRefundList().get(i).getStatus())) {
//                refundAgreeRequest.setAfterSalesCode(orderBean.getRefundList().get(i).getRid());
//                if (PayRefundStatus.SUCCESS.equals(refundResponse.getPayRefundStatus())) {
//                    baseOrderResponse = orderCenterSdkService.refundComplete(refundAgreeRequest);
//                } else {
//                    baseOrderResponse = orderCenterSdkService.refundReturn(refundAgreeRequest);
//                }
//                break;
//            } else if (RefundStatusV1.COMPLETE_REFUND.getCodeV1().equals(orderBean.getRefundList().get(i).getStatus())) {
//                baseOrderResponse = new BaseOrderResponse();
//                baseOrderResponse.setErrcode(Integer.valueOf(ResponseResult.AGREE_REFUND_ORDER_FAIL.getCode()));
//                baseOrderResponse.setErrmsg("订单已经退款完成，不能重复申请");
//                break;
//            }
//        }
//        if(null == baseOrderResponse) {
//            OrderRefundRequest orderRefundRequest = new OrderRefundRequest();
//            orderRefundRequest.setPartnerId(refundAgreeRequest.getPartnerId());
//            orderRefundRequest.setStoreId(refundAgreeRequest.getShopId());
//            orderRefundRequest.setOrderId(refundAgreeRequest.getOrderId());
//            orderRefundRequest.setOrderClient(OrderClientType.SAAS.getIndex());
//
//            orderRefundRequest.setAfterSalesType(AfterSalesType.PARTNER_CANCEL);
//            orderRefundRequest.setReason(refundAgreeRequest.getReason());
//            orderRefundRequest.setOperator(refundAgreeRequest.getOperator());
//            orderRefundRequest.setOrderStatusV1(orderBean.getStatus());
//            if (PayRefundStatus.SUCCESS.equals(refundResponse.getPayRefundStatus())) {
//                orderRefundRequest.setCreateEvent(4);
//            } else {
//                orderRefundRequest.setCreateEvent(5);
//            }
//            baseOrderResponse = orderCenterSdkService.orderRefund(orderRefundRequest);
//        }
//        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
//            return ResponseUtil.error(ResponseResult.AGREE_REFUND_ORDER_FAIL);
//        }
//        return ResponseUtil.success();
//    }

    /**
     * 拒绝退款
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse refundReject(OrderManagerRequest request, OrderBeanV1 orderBean) {
        CancelOrderRequest rejectRefundRequest = orderSdkAdapter.getRejectRefundRequest(request.getOperator(), request.getReason(), orderBean);
        rejectRefundRequest.setOperationClient(request.getOperationClient());
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.orderRefundReject(rejectRefundRequest);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.REJECT_REFUND_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 麦咖啡拒绝退款
     *
     * @param request
     * @param orderBean
     * @return
     */
    public BaseResponse mcCafeRefundReject(OrderManagerRequest request, OrderBeanV1 orderBean) {
        CancelOrderRequest rejectRefundRequest = orderSdkAdapter.getRejectRefundRequest(request.getOperator(), request.getReason(), orderBean);
        BaseOrderResponse baseOrderResponse = null;
        for (int i = 0; i < orderBean.getRefundList().size(); i++) {

            if (RefundStatusV1.APPLY_REFUND.getCodeV1().equals(orderBean.getRefundList().get(i).getStatus())) {
                rejectRefundRequest.setAfterSalesCode(orderBean.getRefundList().get(i).getRid());
                rejectRefundRequest.setOperationClient(request.getOperationClient());
                baseOrderResponse = orderCenterSdkService.orderRefundReject(rejectRefundRequest);
            }
            if (RefundStatusV1.COMPLETE_REFUND.getCodeV1().equals(orderBean.getRefundList().get(i).getStatus())) {
                baseOrderResponse = new BaseOrderResponse();
                baseOrderResponse.setErrcode(100);
                baseOrderResponse.setErrmsg("success");
            }
        }
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.REJECT_REFUND_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 取消订单
     */
    @Deprecated
    public BaseResponse orderCancel(OrderManagerRequest request, PayRefundResponse refundResponse, OrderBeanV1 orderBean) {
        //作废订单
        CancelOrderRequest cancelOrderRequest = orderSdkAdapter.getCancelOrderRequest(request.getOperator(), request.getReason(), refundResponse.getPayRefundStatus(), refundResponse.getRefundId(), orderBean);
        BaseOrderResponse baseOrderResponse = orderCenterSdkService.orderCancel(cancelOrderRequest);
        if (baseOrderResponse == null || ObjectUtils.notEqual(ResponseResult.SUCCESS.getCode(), baseOrderResponse.getErrcode().toString())) {
            return ResponseUtil.error(ResponseResult.AFFIRM_ORDER_FAIL);
        }
        return ResponseUtil.success();
    }

    /**
     * 转换成callback消息体
     *
     * @param orderCode   订单编号
     * @param orderState  cn.freemud.shared.order.enums.order.OrderStatus.API_DESC="订单状态 1:待支付 2:下单 3:已接单 4:制作完成(待发货) 5:配送中 6:已完 7:已关闭";
     * @param operateType 操作类型 0= 其他信息修改，1=订单正向操作，2=订单逆向操作,3=通知消息
     * @param msgType     消息类型 MsgType() 1:订单信息变化通知通知
     * @return
     */
    public OrderCallBackRequestVo convert2CallbackReq(String orderCode, Integer orderState, Integer operateType, String msgType, String partnerId) {
        OrderCallBackRequestVo notifyDto = new OrderCallBackRequestVo();
        //直接查询基础服务获取订单信息
        QueryByCodeResponse orderInfo = orderSdkService.getOrderInfo(null, orderCode, null,
                null, LogThreadLocal.getTrackingNo(),partnerId);
        if (orderInfo == null) {
            return null;
        }
        OrderInfoReqs orderResp = orderInfo.getResult();
        // @see cn.freemud.shared.order.enums.order.OrderStatus.API_DESC = "订单状态 1:待支付 2:下单 3:已接单 4:制作完成(待发货) 5:配送中 6:已完 7:已关闭";
        if (orderState != null && orderState > 0) {
            orderResp.setOrderState(orderState);
        }
        notifyDto.setContent(JSON.toJSONString(orderResp));
        notifyDto.setOrderState(orderState);
        notifyDto.setPartnerId(orderResp.getPartnerId());
        notifyDto.setRequestId(UUID.randomUUID().toString().replaceAll("-", ""));
        notifyDto.setOperateType(operateType != null ? operateType : 1);
        // @see cn.freemud.shared.order.enums.MsgType() 1:订单信息变化通知通知
        notifyDto.setMsgType(StringUtils.isNotBlank(msgType) ? msgType : "1");
        notifyDto.setStoreId(orderResp.getStoreId());
        notifyDto.setOrderCode(orderResp.getOrderCode());
        notifyDto.setOrderClient(orderResp.getOrderClient() != null ? orderResp.getOrderClient().intValue() : null);
        return notifyDto;
    }


}
