#include <QNetworkAccessManager>
#include <QSslConfiguration>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QSslSocket>
#include <QEventLoop>
#include <QTimer>
#include "global.h"
#include "network.h"
#include "QsLog.h"


NetWork::NetWork(QObject *parent) : QObject(parent), _tcpserver(new QTcpServer()), _active_skt(NULL)
{
    connect(_tcpserver, SIGNAL(newConnection()), this, SLOT(on_new_connection_come()));
}

NetWork::~NetWork()
{
    if(_tcpserver != NULL)
    {
        delete _tcpserver;
        _tcpserver = NULL;
    }

}

bool NetWork::listen(unsigned int port, QString &error)
{
     QLOG_INFO() << "start listen 127.0.0.1:" << port;

     if(_tcpserver == NULL)
     {
         QLOG_ERROR() << "new QTcpServer failed";
         error = QString("new QTcpServer failed");
         return false;
     }

     if(_tcpserver->isListening())
         _tcpserver->close();

     if(!_tcpserver->listen(QHostAddress::Any, port))
     {
         QLOG_ERROR() << "listen 127.0.0.1:" << port << " failed";
         error = QString("listen 127.0.0.1:") + QString::number(port) + QString(" failed");
         return false;
     }
     return true;
}

void NetWork::closeconnection()
{
    if(_active_skt->isOpen())
        _active_skt->close();
    _active_skt = NULL;
}

bool NetWork::senddata(QByteArray array)
{
    if(_active_skt == NULL)
    {
        QLOG_ERROR() << "Active sockt is NULL";
        return false;
    }

    if(!_active_skt->isOpen())
    {
        QLOG_ERROR() << "Active sockt is not open";
        closeconnection();
        return false;
    }

    _active_skt->write(array);

    if(!_active_skt->waitForBytesWritten())
    {
        QLOG_ERROR() << "Write data failed";
        closeconnection();
        return false;
    }

     QLOG_WARN() << "close socket connection";

    closeconnection();

    return true;
}

bool NetWork::HttpPost(QString url, QByteArray indata, QByteArray &outdata, QString &error, unsigned int timeout)
{
    QLOG_INFO() << "Request URL : " << url;
    QLOG_INFO() << "Request data : " << QString::fromUtf8(indata);
    QLOG_INFO() << "Request timeout : " << timeout;

    if(url.isEmpty())
    {
        QLOG_ERROR() << ERROR_MSG_1101;
        error = QString(ERROR_MSG_1101);
        return false;
    }

    if(timeout == 0)
        timeout = 60;

    QEventLoop loop;
    QTimer timer;

//    QSslConfiguration config = QSslConfiguration::defaultConfiguration();
//    config.setPeerVerifyMode(QSslSocket::VerifyNone);

    QNetworkAccessManager manager;
    QNetworkRequest request(url);

    request.setRawHeader(CONNECT_TYPE, CONNECT_TYPE_JSON);

    if(manager.networkAccessible() != QNetworkAccessManager::Accessible)
        manager.setNetworkAccessible(QNetworkAccessManager::Accessible);

    QNetworkReply *reply = manager.post(request, indata);

    connect(&timer, SIGNAL(timeout()),  &loop, SLOT(quit()));
    connect(&manager, SIGNAL(finished(QNetworkReply*)), &loop, SLOT(quit()));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError)), &loop, SLOT(quit()));

    timer.start(timeout * 1000);

    loop.exec();

    if(reply->error() != QNetworkReply::NoError)
    {
        outdata = QByteArray();
        error = QString::fromLocal8Bit(ERROR_MSG_1099);
        QLOG_ERROR() << "NetWork error code : " << reply->error() << "; error message : " << reply->errorString();
        return false;
    }

    outdata = reply->readAll();

    if(outdata.isEmpty())
    {
        outdata = QByteArray();
        error = QString::fromLocal8Bit(ERROR_MSG_1099);
        QLOG_ERROR() << "Recv data is null";
        return false;
    }

    reply->deleteLater();

    return true;
}

void NetWork::on_new_connection_come()
{
    QLOG_INFO() << "New connection";

    if(_active_skt != NULL &&  _active_skt->isOpen())
    {
        QLOG_WARN() << "Invalid TCP Connection";
        _tcpserver->nextPendingConnection()->close();
        return ;
    }

    _active_skt = _tcpserver->nextPendingConnection();

    QByteArray data;

    if(_active_skt->waitForReadyRead())
    {
        QByteArray tmp;

        do
        {
            tmp = _active_skt->readAll();
            data += tmp;
            _active_skt->waitForReadyRead(500);
        }while(tmp.isEmpty());

    }
    else
    {
        QLOG_ERROR() << ERROR_MSG_1100;

        emit error_throw(QString::fromLocal8Bit(ERROR_MSG_1100));
        return ;
    }

    QLOG_INFO() << "recv data : " << data.toHex();

    emit recv_data_ready(data);
}
