#ifndef FM_XMLTOJSON
#define FM_XMLTOJSON

#include <vector>
#include <string>
#include <sstream>
#include <iostream>

#include <xmlrapid/rapidxml.hpp>
#include <xmlrapid/rapidxml_utils.hpp>
#include <xmlrapid/rapidxml_print.hpp>

#include <rapidjson/document.h>
#include <rapidjson/prettywriter.h>  
#include <rapidjson/filereadstream.h>  
#include <rapidjson/filewritestream.h> 
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>

#include "Calc.hpp"

#include "tool.h"
#include "global.h"
#include "QsLog.h"

#include <QFile>

#define XML_FREEMUD_CLIENT			"Client.xml"
//extern char XML_FREEMUD_CLIENT[256];

#define XML_NODE_ROOT				"Freemud"

#define XML_NODE_CLIENT				"FreemudFront"

#define XML_NODE_SERVER				"FreemudPost"

#define XML_NODE_REQTYPE			"ReqType"

#define XML_NODE_OBJECT             "object"

#define XML_ATT_TYPE				"type"

#define XML_ATT_KEY				    "key"

#define XML_ATT_MKEY				 "mkey"

#define XML_ATT_OKEY				"okey"

#define XML_ATT_VALUE				"value"

#define XML_ATT_MVALUE				"mvalue"

#define XML_ATT_IS_ZERO				"iszero"

#define XML_ATT_NOT_ZERO			"notzero"

#define XML_ATT_IS_HAVEKEY			"havekey"

#define JSON_TYPE_INT				"int"
#define JSON_TYPE_STRING			"string"
#define JSON_TYPE_DOUBLE			"double"
#define JSON_TYPE_ARRAY				"array"
#define JSON_TYPE_OBJ				"object"


#define RANGE_OF_ERROR				0.0000001

#ifdef WIN32
#include <windows.h>
#include <io.h>
EXTERN_C IMAGE_DOS_HEADER __ImageBase;
#undef GetObject
#else
#include <unistd.h>
#define MAX_PATH 1024
#endif

using std::vector;
using std::string;
using std::stringstream;

using rapidxml::file;
using rapidxml::xml_node;
using rapidxml::xml_document;
using rapidxml::xml_attribute;

using rapidjson::Document;
using rapidjson::Value;
using rapidjson::StringBuffer;
using rapidjson::Writer;
using rapidjson::Type;


typedef vector<string> stringlist;

class fm_xmltojson
{

public:
	static bool GetXMLDocumentHandle(file<> &fdoc, xml_document<> &doc, stringstream &lasterror)
	{
		try
		{	
			if(fdoc.data() == NULL)
			{
				lasterror << "read XML file failed";
				return false;
			}
			doc.parse<0>(fdoc.data());

		}catch (rapidxml::parse_error& ex)
		{
			lasterror << "parse_error : " << ex.what();
			return false;
		}
		catch (...)
		{
			lasterror << "XML file error;";
			return false;
		}
		return true;
	}

	static void GetKeyList(string key, stringlist &keylist)
	{
		if(key.empty())
			return ;

		keylist.clear();

		if(key.find("/") == string::npos)
		{
			keylist.push_back(key);
			return;
		}

		unsigned int i = 0, j = 0, k = 0;

		while(i < key.size())
		{
			if(i == 0 && key.at(i) == '/')
			{
				k = 1;
				j = 0;
				++ i;
				continue;
			}

			if(key.at(i) == '/')
			{
				if(j != 0)
				{
					string tmp = key.substr(k, j);
					keylist.push_back(tmp);
				}
				k = i + 1;
				j = 0;
				++ i;
				continue;
			}
			++ i;
			++ j;

		}

		if(k < key.size())
		{
			string tmp = key.substr(k, key.size() - k);
			keylist.push_back(tmp);
		}
	}

	static void GetKeyList(char splitkey, string key, stringlist &keylist)
	{
		if(key.empty())
			return ;

		keylist.clear();

		unsigned int i = 0;

		while (i < key.size())
		{
			stringstream tmpstr;

			while(i < key.size())
			{
				if(key[i] == '\\' && i < key.size() - 1 && key[i+1] == splitkey)
				{
					tmpstr << key[i+1];
					i += 2;
					continue;
				}

				if(key[i] == splitkey)
				{
					++ i;
					break;
				}
				tmpstr << key[i];
				++i;
			}

			if(!tmpstr.str().empty())
				keylist.push_back(tmpstr.str());
		}
	}


	static void GetProcPath(std::string &path)
	{
#ifdef WIN32 
		int curPos;
		char pathBuf[MAX_PATH] = { 0 };

		GetModuleFileName((HMODULE)&__ImageBase, pathBuf, MAX_PATH); 


		curPos = strlen(pathBuf) - 1;

		while('\\' != pathBuf[curPos])
		{
			curPos--;
		}

		curPos++;

		pathBuf[curPos] = '\0';

		path = std::string(pathBuf);
#else
		int rlt = 0;
		char pathBuf[MAX_PATH] = { 0 };
		int curPos = 0;

		rlt = readlink("/proc/self/exe", pathBuf, MAX_PATH);
		if(rlt < 0 || (rlt >= MAX_PATH - 1))
		{
			//perror("GetCfgFilePath error: ");
			return;
		}
		pathBuf[rlt] = '\0';
		curPos = rlt - 1;
		while(pathBuf[curPos] != '/')
		{
			pathBuf[curPos] = '\0';
			curPos--;
		} 
		path = std::string(pathBuf);
#endif  
	};

	static bool GetJsonNodeWithKeyList(Document &docs, Value &values, const Value &sparentvalue, stringlist nodelist)
	{
		if(!sparentvalue.IsObject())
			return false;

		Document doc;
		Value tmp;

		tmp.CopyFrom(sparentvalue, doc.GetAllocator());

		for(unsigned int i = 0; i < nodelist.size(); ++i)
		{
			if(!tmp.HasMember(nodelist[i].c_str()))
				return false;

			if(tmp[nodelist[i].c_str()].IsInt())
				return false;
			if(tmp[nodelist[i].c_str()].IsDouble())
				return false;
			if(tmp[nodelist[i].c_str()].IsString())
				return false;

			if((tmp[nodelist[i].c_str()].IsArray() || tmp[nodelist[i].c_str()].IsObject()) &&  i == nodelist.size() - 1)
			{
				const Value &arrays = tmp[nodelist[i].c_str()];
				values.CopyFrom(arrays, docs.GetAllocator());
				return true;
			}

			if(tmp[nodelist[i].c_str()].IsArray())
			{
				const Value &arrays = tmp[nodelist[i].c_str()].GetArray();
				tmp.CopyFrom(arrays[0], doc.GetAllocator());
				continue;
			}

			if(tmp[nodelist[i].c_str()].IsObject())
			{
				const Value &obj = tmp[nodelist[i].c_str()].GetObject();
				tmp.CopyFrom(obj, doc.GetAllocator());
				continue;
			}
		}

		return false;
	}

    static bool GetJsonKeyWithKeyList(const Value &sparentvalue, stringlist nodelist)
    {
        if(!sparentvalue.IsObject())
            return false;

        Document doc;
        Value tmp;

        tmp.CopyFrom(sparentvalue, doc.GetAllocator());

        for(unsigned int i = 0; i < nodelist.size(); ++i)
        {
            if(!tmp.HasMember(nodelist[i].c_str()))
                return false;
            if(tmp[nodelist[i].c_str()].IsArray())
            {
                const Value &arrays = tmp[nodelist[i].c_str()].GetArray();
                tmp.CopyFrom(arrays[0], doc.GetAllocator());
                continue;
            }

            if(tmp[nodelist[i].c_str()].IsObject())
            {
                const Value &obj = tmp[nodelist[i].c_str()].GetObject();
                tmp.CopyFrom(obj, doc.GetAllocator());
                continue;
            }

            if(i != nodelist.size() - 1)
                return false;

            if(tmp[nodelist[i].c_str()].IsInt() && i != nodelist.size() - 1)
                return false;
            if(tmp[nodelist[i].c_str()].IsDouble() && i != nodelist.size() - 1)
                return false;
            if(tmp[nodelist[i].c_str()].IsString() && i != nodelist.size() - 1)
                return false;
        }
        return true;
    }

	static bool GetJsonValueWithKeyList(const Value &sparentvalue, stringlist nodelist, string &values)
	{	
		if(!sparentvalue.IsObject())
			return false;

        rapidjson::StringBuffer buffers;
        rapidjson::Writer<rapidjson::StringBuffer> writers(buffers);
        sparentvalue.Accept(writers);

		Document doc;
		Value tmp;

		tmp.CopyFrom(sparentvalue, doc.GetAllocator());

		for(unsigned int i = 0; i < nodelist.size(); ++i)
		{
			if(!tmp.HasMember(nodelist[i].c_str()))
				return false;
			if(tmp[nodelist[i].c_str()].IsArray())
			{
				const Value &arrays = tmp[nodelist[i].c_str()].GetArray();
				tmp.CopyFrom(arrays[0], doc.GetAllocator());
				continue;
			}

			if(tmp[nodelist[i].c_str()].IsObject())
			{
				const Value &obj = tmp[nodelist[i].c_str()].GetObject();
				tmp.CopyFrom(obj, doc.GetAllocator());
				continue;
			}

			if(tmp[nodelist[i].c_str()].IsInt() && i != nodelist.size() - 1)
				return false;
			if(tmp[nodelist[i].c_str()].IsDouble() && i != nodelist.size() - 1)
				return false;
			if(tmp[nodelist[i].c_str()].IsString() && i != nodelist.size() - 1)
				return false;

			stringstream stram;

			//std::cout << nodelist.size();
			//std::cout << tmp[nodelist[i].c_str()].GetType();

			if(tmp[nodelist[i].c_str()].IsInt() && i == nodelist.size() - 1)
			{
				stram << tmp[nodelist[i].c_str()].GetInt();
				values = stram.str();
				return true;
			}

			if(tmp[nodelist[i].c_str()].IsDouble() && i == nodelist.size() - 1)
			{
				stram << tmp[nodelist[i].c_str()].GetDouble();
				values = stram.str();
				return true;
			}

			if(tmp[nodelist[i].c_str()].IsString() && i == nodelist.size() - 1)
			{
				stram << tmp[nodelist[i].c_str()].GetString();
				values = stram.str();
				return true;
			}
		}

		return false;
	}

	static bool GetNodeName(const string &nodename, const string &sparentname, stringlist &realnodelist, bool &usingparent)
	{
		usingparent = false;

		if(nodename.empty())
			return false;

		realnodelist.clear();

		stringlist nodelist, parentlist;

		GetKeyList(nodename, nodelist);
		GetKeyList(sparentname, parentlist);

		if(sparentname.empty())
		{
			realnodelist = nodelist;
			return true;
		}

		if(nodelist.size() == 0)
			return false;

		if(sparentname.size() == 0)
		{
			realnodelist = nodelist;
			return true;
		}

		if(nodelist.size() <= parentlist.size())
		{
			realnodelist = nodelist;
			return true;
		}

		unsigned int i = 0;

		for(i = 0; i < parentlist.size(); ++i)
		{
			if(nodelist[i].empty() || parentlist[i].empty() || nodelist[i].compare(parentlist[i].c_str()) != 0)
				break;
		}

		if(i != parentlist.size())
		{
			realnodelist = nodelist;
			return true;
		}

		for (; i < nodelist.size(); ++i)
		{
			realnodelist.push_back(nodelist[i]);
		}
		usingparent = true;
		return true;
	}


	static bool Check_Is_Oper(string nodename)
	{
		int i = 0, j = nodename.size() - 1;
		bool flag = false;

		while(i < nodename.size())
		{
			if(nodename.at(i) != ' ' && nodename.at(i) != '(')
				break;

			if(nodename.at(i) == '(')
			{
				flag = true;
				break;
			}

			++i;
		}

		if(!flag)
			return false;

		while(j >= 0)
		{
			if(nodename.at(j) != ' ' && nodename.at(j) != ')')
				break;

			if(nodename.at(i) == ')')
			{
				flag = true;
				break;
			}

			--j;
		}

		return flag;
	}

	static bool GetValuePoint(string nodename, int &begin, int &end, int i)
	{
		int j = 0, k = 0;

		begin = 0, end = 0;

		while(j < nodename.size())
		{
			if(nodename.at(j) == '%')
				++ k;
			if(begin == 0 && k == i * 2 + 1)
				begin = j;
			if(end == 0 && k == i * 2 + 2)
				end = j;
			++j;
		}

		return ((begin != 0 && end != 0) ? true : false);
	}

    static bool GetJsonKey(string nodename, const Value &sval, const Value &sparentvalue, string sparentname = string(""))
    {
        stringlist realnodelist;
        bool usingparent = false;

        if(GetNodeName(nodename, sparentname, realnodelist, usingparent) == false)
            return false;

        if(usingparent)
        {
            if(GetJsonKeyWithKeyList(sparentvalue, realnodelist) == false)
                return false;
        }
        else
        {
            if(GetJsonKeyWithKeyList(sval, realnodelist) == false)
                return false;
        }
        return true;
    }

	static bool GetJsonValue(string nodename, string &values, const Value &sval, const Value &sparentvalue, string sparentname = string(""))
	{
		if(!Check_Is_Oper(nodename))
			return Do_GetJsonValue(nodename, values, sval, sparentvalue, sparentname);

		std::string tmp = nodename;
		int begin = 0, end = 0;
		int oldend = 0;
		int i = 0;

		stringstream strstm;
		stringstream rlt;

		while(GetValuePoint(nodename, begin, end, i))
		{
			if(oldend < begin)
				strstm << nodename.substr(oldend, begin - oldend);
			string tmpvalue = nodename.substr(begin, end - begin + 1);

			if(tmpvalue.size() - 1 <= 1)
				return false;

			string tmpstr = tmpvalue.substr(1, tmpvalue.size() - 2);
			string str;

			if(!Do_GetJsonValue(tmpstr, str, sval, sparentvalue, sparentname))
				return false;

			strstm << str;

			oldend = end + 1;
			++ i;
		}

		if(oldend <= nodename.size() - 1)
			strstm << nodename.substr(oldend, nodename.size() - oldend);

		try
		{
			double dl = Calc::parenth_final(strstm.str());
			rlt << dl;
			values = rlt.str();
		}
		catch(...)
		{
			return false;
		}

		return true;
	}


	static bool Do_GetJsonValue(string nodename, string &values, const Value &sval, const Value &sparentvalue, string sparentname = string(""))
	{
		stringlist realnodelist;
		bool usingparent = false;

		if(GetNodeName(nodename, sparentname, realnodelist, usingparent) == false)
			return false;


		if(usingparent)
		{
			if(GetJsonValueWithKeyList(sparentvalue, realnodelist, values) == false)
				return false;
		}
		else
		{
			if(GetJsonValueWithKeyList(sval, realnodelist, values) == false)
				return false;
		}
		return true;

	}

	static bool GetParentNodeVale(string nodename, Document &docs, Value &values, const Value &sval, const Value &sparentvalue, string sparentname = string(""))
	{
		stringlist realnodelist;
		bool usingparent = false;

		if(GetNodeName(nodename, sparentname, realnodelist, usingparent) == false)
			return false;


		if(usingparent)
		{
			if(GetJsonNodeWithKeyList(docs, values, sparentvalue, realnodelist) == false)
				return false;
		}
		else
		{
			if(GetJsonNodeWithKeyList(docs, values, sval, realnodelist) == false)
				return false;
		}
		return true;
	}


	static void GetMaping(xml_node<> *node, string &invalue)
	{
		if(node == NULL)
			return ;

		string tmpvalue = invalue;

		for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
		{
			if(attr->name() == NULL || string("maping").compare(attr->name()) != 0)
				continue;

			if(attr->first_attribute(XML_ATT_MKEY) == NULL || attr->first_attribute(XML_ATT_MVALUE) == NULL)
				continue;

			char *key = attr->first_attribute(XML_ATT_MKEY)->value();
			char *value =  attr->first_attribute(XML_ATT_MVALUE)->value();

			if(tmpvalue.compare(key) == 0)
			{
				invalue = string(value);
				return ;
			}

			if(string(key).empty() && tmpvalue.empty())
			{
				invalue = string(value);
				return ;
			}

		}
	}

    static bool CheckNodeNeed(xml_node<> *node, const Value &sval, const Value &sparentvalue, string sparentname = string(""))
    {
        if(node == NULL)
            return false;

        if( (node->first_attribute(XML_ATT_IS_ZERO) == NULL || string(node->first_attribute(XML_ATT_IS_ZERO)->value()).empty()) &&
             (node->first_attribute(XML_ATT_NOT_ZERO) == NULL || string(node->first_attribute(XML_ATT_NOT_ZERO)->value()).empty()) &&
              (node->first_attribute(XML_ATT_IS_HAVEKEY) == NULL || string(node->first_attribute(XML_ATT_IS_HAVEKEY)->value()).empty()))
              return true;


        if(node->first_attribute(XML_ATT_IS_ZERO) != NULL && !string(node->first_attribute(XML_ATT_IS_ZERO)->value()).empty())
        {
            string tmpvalue;
            if(!GetJsonValue(string(node->first_attribute(XML_ATT_IS_ZERO)->value()), tmpvalue, sval, sparentvalue, sparentname))
                return false;

            if(tmpvalue.compare("0") != 0)
                return false;
        }

        if(node->first_attribute(XML_ATT_NOT_ZERO) != NULL && !string(node->first_attribute(XML_ATT_NOT_ZERO)->value()).empty())
        {
            string tmpvalue;
            if(!GetJsonValue(string(node->first_attribute(XML_ATT_NOT_ZERO)->value()), tmpvalue, sval, sparentvalue, sparentname))
                return false;

            if(tmpvalue.compare("0") == 0)
                return false;
        }

        if(node->first_attribute(XML_ATT_IS_HAVEKEY) != NULL && !string(node->first_attribute(XML_ATT_IS_HAVEKEY)->value()).empty())
        {
            if(!GetJsonKey(string(node->first_attribute(XML_ATT_IS_HAVEKEY)->value()), sval, sparentvalue, sparentname))
                return false;
        }

        return true;

    }

    static void InsertValueToJson(xml_node<> *node, Document &desdoc, Value &dval, const Value &sval, const Value &sparentvalue, string sparentname = string(""))
    {
        if(node == NULL)
            return ;

        if(node->first_attribute(XML_ATT_TYPE) == NULL)
            return ;

        char *pkey = node->name();

        do
        {
            if(node->first_attribute(XML_ATT_OKEY) == NULL && string(JSON_TYPE_INT).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->value() == NULL || string(node->value()).empty())
                    return ;

                if(dval.HasMember(pkey))
                    dval.RemoveMember(pkey);

                dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), std::atoi(node->value()), desdoc.GetAllocator());
                break ;
            }

            if(node->first_attribute(XML_ATT_OKEY) == NULL && string(JSON_TYPE_DOUBLE).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->value() == NULL || string(node->value()).empty())
                    return ;

                if(dval.HasMember(pkey))
                    dval.RemoveMember(pkey);

                dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), std::atof(node->value()), desdoc.GetAllocator());
                break ;
            }

            if(node->first_attribute(XML_ATT_OKEY) == NULL && string(JSON_TYPE_STRING).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->value() == NULL || string(node->value()).empty())
                    return ;

                if(dval.HasMember(pkey))
                    dval.RemoveMember(pkey);

                dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), Value().SetString(node->value(), strlen(node->value()), desdoc.GetAllocator()), desdoc.GetAllocator());
                break ;
            }
            if(node->first_attribute(XML_ATT_OKEY) == NULL && string(JSON_TYPE_ARRAY).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->first_node()->name() != NULL && string(XML_NODE_OBJECT).compare(node->first_node()->name()) ==0 )
                {
                    Value arrayvalue;

                    if(dval.HasMember(pkey) && dval[pkey].IsArray())
                    {
                        arrayvalue.CopyFrom(dval[pkey], desdoc.GetAllocator());
                    }
                    else
                    {
                        arrayvalue.SetArray();
                    }

                    for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                    {
                        if(!CheckNodeNeed(attr, sval, sparentvalue, sparentname))
                            continue;

                        Value objectvalue(rapidjson::kObjectType);

                        for(xml_node<> * attrs = attr->first_node(); attrs != NULL; attrs = attrs->next_sibling())
                        {
                            InsertValueToJson(attrs, desdoc, objectvalue, sval, sparentvalue);
                        }
                        if(objectvalue.GetObject().MemberCount() > 0)
                            arrayvalue.PushBack(objectvalue, desdoc.GetAllocator());
                    }

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), arrayvalue, desdoc.GetAllocator());
                    break ;
                }
                else
                {
                    Value arrayvalue;

                    if(dval.HasMember(pkey) && dval[pkey].IsArray())
                    {
                        arrayvalue.CopyFrom(dval[pkey], desdoc.GetAllocator());
                    }
                    else
                    {
                        arrayvalue.SetArray();
                    }
                    Value objectvalue(rapidjson::kObjectType);

                    for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                    {
                        InsertValueToJson(attr, desdoc, objectvalue, sval, sparentvalue);
                    }
                    if(objectvalue.GetObject().MemberCount() > 0)
                        arrayvalue.PushBack(objectvalue, desdoc.GetAllocator());

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), arrayvalue, desdoc.GetAllocator());
                    break ;
                }
            }

            if(node->first_attribute(XML_ATT_OKEY) == NULL && string(JSON_TYPE_OBJ).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {

                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                Value objectvalue;

                if(dval.HasMember(pkey) && ((const Value &)dval[pkey]).IsObject())
                {
                    objectvalue.CopyFrom(dval[pkey], desdoc.GetAllocator());
                }
                else
                {
                    objectvalue.SetObject();
                }

                for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                {
                    InsertValueToJson(attr, desdoc, objectvalue, sval, sparentvalue);
                }

                if(dval.HasMember(pkey))
                    dval.RemoveMember(pkey);

                if(objectvalue.GetObject().MemberCount() > 0)
                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), objectvalue, desdoc.GetAllocator());

                break ;
            }

            //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
            if(node->first_attribute(XML_ATT_OKEY) != NULL && string(JSON_TYPE_INT).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {

                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->value() != NULL && !string(node->value()).empty())
                {
                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), std::atoi(node->value()), desdoc.GetAllocator());
                    break;
                }
                else
                {
                    string values;
                    if(!GetJsonValue(string(node->first_attribute(XML_ATT_OKEY)->value()), values, sval, sparentvalue, sparentname))
                        return ;

                    GetMaping(node, values);

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), std::atoi(values.c_str()), desdoc.GetAllocator());
                    break ;
                }

            }

            if(node->first_attribute(XML_ATT_OKEY) != NULL && string(JSON_TYPE_DOUBLE).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->value() != NULL && !string(node->value()).empty())
                {
                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), std::atof(node->value()), desdoc.GetAllocator());
                    break;
                }
                else
                {
                    string values;
                    if(!GetJsonValue(string(node->first_attribute(XML_ATT_OKEY)->value()), values, sval, sparentvalue, sparentname))
                        return ;

                    GetMaping(node, values);

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), std::atof(values.c_str()), desdoc.GetAllocator());
                    break ;
                }

            }

            if(node->first_attribute(XML_ATT_OKEY) != NULL && string(JSON_TYPE_STRING).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                if(node->value() != NULL && !string(node->value()).empty())
                {
                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);
                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()),  Value().SetString(node->value(), strlen(node->value()), desdoc.GetAllocator()), desdoc.GetAllocator());
                    break;
                }
                else
                {
                    string values;
                    if(!GetJsonValue(string(node->first_attribute(XML_ATT_OKEY)->value()), values, sval, sparentvalue, sparentname))
                        return ;

                    GetMaping(node, values);

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), Value().SetString(values.c_str(), strlen(values.c_str()), desdoc.GetAllocator()), desdoc.GetAllocator());
                    break ;
                }

            }

            if(node->first_attribute(XML_ATT_OKEY) != NULL && string(JSON_TYPE_ARRAY).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                Document docs;
                Value valus;

                if(GetParentNodeVale(string(node->first_attribute(XML_ATT_OKEY)->value()), docs, valus, sval, sparentvalue, sparentname) == false)
                    break;

                if(valus.IsArray())
                {
                    Value arrayvalue;

                    if(dval.HasMember(pkey) && dval[pkey].IsArray())
                    {
                        arrayvalue.CopyFrom(dval[pkey], desdoc.GetAllocator());
                    }
                    else
                    {
                        arrayvalue.SetArray();
                    }

                    for (unsigned int i = 0; i < valus.Size(); ++i)
                    {
                        Value objectvalue(rapidjson::kObjectType);
                        for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                        {
                            InsertValueToJson(attr, desdoc, objectvalue, sval, valus.GetArray()[i], string(node->first_attribute(XML_ATT_OKEY)->value()));
                        }
                        if(objectvalue.GetObject().MemberCount() > 0)
                            arrayvalue.PushBack(objectvalue, desdoc.GetAllocator());
                    }

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), arrayvalue, desdoc.GetAllocator());
                    break;
                }

                if(valus.IsObject())
                {
                    Value arrayvalue;

                    if(dval.HasMember(pkey) && ((const Value &)dval[pkey]).IsArray())
                    {
                        arrayvalue.CopyFrom(dval[pkey], desdoc.GetAllocator());
                    }
                    else
                    {
                        arrayvalue.SetArray();
                    }

                    Value objectvalue(rapidjson::kObjectType);
                    for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                    {
                        InsertValueToJson(attr, desdoc, objectvalue, sval, valus, string(node->first_attribute(XML_ATT_OKEY)->value()));
                    }
                    if(objectvalue.GetObject().MemberCount() > 0)
                        arrayvalue.PushBack(objectvalue, desdoc.GetAllocator());

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), arrayvalue, desdoc.GetAllocator());
                    break;
                }
                break;
            }

            if(node->first_attribute(XML_ATT_OKEY) != NULL && string(JSON_TYPE_OBJ).compare(node->first_attribute(XML_ATT_TYPE)->value()) == 0)
            {
                if(!CheckNodeNeed(node, sval, sparentvalue, sparentname))
                    break;

                Document docs;
                Value valus;

                if(GetParentNodeVale(string(node->first_attribute(XML_ATT_OKEY)->value()), docs, valus, sval, sparentvalue, sparentname) == false)
                    break;

                if(valus.IsArray())
                {

                    Value objectvalue;

                    if(dval.HasMember(pkey) && ((const Value &)dval[pkey]).IsArray() && ((const Value &)dval[pkey]).GetArray().Size() >= 1)
                    {
                        objectvalue.CopyFrom(dval[pkey][0], desdoc.GetAllocator());
                    }
                    else
                    {
                        objectvalue.SetObject();
                    }

                    for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                    {
                        InsertValueToJson(attr, desdoc, objectvalue, sval, valus.GetArray()[0], string(node->first_attribute(XML_ATT_OKEY)->value()));
                    }

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);
                    if(objectvalue.GetObject().MemberCount() > 0)
                        dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), objectvalue, desdoc.GetAllocator());
                    break;
                }

                if(valus.IsObject())
                {
                    Value objectvalue;

                    if(dval.HasMember(pkey) && ((const Value &)dval[pkey]).IsObject())
                    {
                        objectvalue.CopyFrom(dval[pkey], desdoc.GetAllocator());
                    }
                    else
                    {
                        objectvalue.SetObject();
                    }

                    for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
                    {
                        InsertValueToJson(attr, desdoc, objectvalue, sval, valus, string(node->first_attribute(XML_ATT_OKEY)->value()));
                    }

                    if(dval.HasMember(pkey))
                        dval.RemoveMember(pkey);

                    dval.AddMember(Value().SetString(pkey, strlen(pkey), desdoc.GetAllocator()), objectvalue, desdoc.GetAllocator());
                    break;
                }
                break;
            }

        } while (0);
    }

	static void Do_GetJsonWithXML(xml_node<> *node, const Document &srcdoc, Document &desdoc)
	{
		if(node == NULL)
			return ;

		const Value &sval = srcdoc;
		Value &dval = desdoc;

		for(xml_node<> * attr = node->first_node(); attr != NULL; attr = attr->next_sibling())
		{
			InsertValueToJson(attr, desdoc, dval, sval, sval);
		}
	}

	static bool GetMajorKey(xml_node<> *node, xml_node<> * &reqtype, const Document &srcdoc, string &key, string &values)
	{
		for(reqtype = node->first_node(XML_NODE_REQTYPE); reqtype != NULL; reqtype = reqtype->next_sibling())
		{
			if(reqtype->first_attribute(XML_ATT_KEY) == NULL || string(reqtype->first_attribute(XML_ATT_KEY)->value()).empty())
				continue;

			if(reqtype->first_attribute(XML_ATT_VALUE) == NULL || string(reqtype->first_attribute(XML_ATT_VALUE)->value()).empty())
				continue;

			char *att_key = reqtype->first_attribute(XML_ATT_KEY)->value();
			char *att_val = reqtype->first_attribute(XML_ATT_VALUE)->value();

			if(att_key != NULL && att_val != NULL && srcdoc.HasMember(att_key))
			{
				switch (srcdoc[att_key].GetType())
				{
				case rapidjson::kStringType:
					if(string(att_val).compare(srcdoc[att_key].GetString()) == 0)
					{
						key = string(att_key);
						values = string(att_val);
						return true;
					}
					break;
				case rapidjson::kNumberType:
					if(srcdoc[att_key].IsInt() && srcdoc[att_key].GetInt() == atoi(att_val))
					{
						key = string(att_key);
						values = string(att_val);
						return true;
					}
#ifdef WIN32
					if(srcdoc[att_key].IsInt64() && srcdoc[att_key].GetInt64() == _atoi64(att_val))
#else
					if(srcdoc[att_key].IsInt64() && srcdoc[att_key].GetInt64() == atoll(att_val))
#endif
					{
						key = string(att_key);
						values = string(att_val);
						return true;
					}

					if(srcdoc[att_key].IsDouble() && srcdoc[att_key].GetDouble() - atof(att_val) <= RANGE_OF_ERROR)
					{
						key = string(att_key);
						values = string(att_val);
						return true;
					}

					break;
				default :
					break;
				}
			}
		}
		return false;
	}

	static bool GetMajorKey(xml_node<> *node, xml_node<> * &reqtype, string key, string values)
	{
		for(reqtype = node->first_node(XML_NODE_REQTYPE); reqtype != NULL; reqtype = reqtype->next_sibling())
		{
			if(reqtype->first_attribute(XML_ATT_KEY) == NULL || string(reqtype->first_attribute(XML_ATT_KEY)->value()).empty())
				continue;

			if(reqtype->first_attribute(XML_ATT_VALUE) == NULL || string(reqtype->first_attribute(XML_ATT_VALUE)->value()).empty())
				continue;

			char *att_key = reqtype->first_attribute(XML_ATT_KEY)->value();
			char *att_val = reqtype->first_attribute(XML_ATT_VALUE)->value();

			if(att_key != NULL && att_val != NULL && key.compare(att_key) == 0 && values.compare(att_val) == 0)
				return true;
		}
		return false;
	}

	static bool GetFrontJsonWithXML(xml_document<> &doc, const Document &srcdoc, Document &desdoc, string &key, string &values, stringstream &lasterror)
	{
		xml_node<> *root = doc.first_node(XML_NODE_ROOT);

		if(root == NULL)
		{
			lasterror << " root node not " << XML_NODE_ROOT;
			return false;
		}

		xml_node<> *front = root->first_node(XML_NODE_CLIENT);

		if(front == NULL)
		{
			lasterror << " not found FreemudFront node";
			return false;
		}

		xml_node<> *reqtype = NULL;

		if(GetMajorKey(front, reqtype, srcdoc, key, values) == false)
		{
			lasterror << "error : GetMajorKey failed";
			return false;
		}

		Do_GetJsonWithXML(reqtype, srcdoc, desdoc);

		//StringBuffer buffer;
		//Writer<StringBuffer> writer(buffer);
		//desdoc.Accept(writer);
		//
		//std::cout << buffer.GetString();

		return true;
	}


	static bool GetPostJsonWithXML(xml_document<> &doc, const Document &srcdoc, Document &desdoc, string key, string values, stringstream &lasterror)
	{
		xml_node<> *root = doc.first_node(XML_NODE_ROOT);

		if(root == NULL)
		{
			lasterror << " root node not " << XML_NODE_ROOT;
			return false;
		}

		xml_node<> *front = root->first_node(XML_NODE_SERVER);

		if(front == NULL)
		{
			lasterror << " not found FreemudPost node";
			return false;
		}

		xml_node<> *reqtype = NULL;

		if(GetMajorKey(front, reqtype, key, values) == false)
		{
			lasterror << "error : GetMajorKey failed";
			return false;
		}

		Do_GetJsonWithXML(reqtype, srcdoc, desdoc);

		//StringBuffer buffer;
		//Writer<StringBuffer> writer(buffer);
		//desdoc.Accept(writer);

		//std::cout << buffer.GetString();

		return true;
	}

public:

	static bool JsonMerge(string basejson, string requestjson, string &totaljson, string &lasterror)
	{
		Document s;
		Document d;

		s.Parse<0>(basejson.data());
		d.Parse<0>(requestjson.data());

		if (s.HasParseError() || d.HasParseError())
		{
			lasterror = string("error in json");
			return false;
		}

		if(!s.IsObject() || !d.IsObject())
		{
			lasterror = string("root not object");
			return false;
		}

		for(Value::ConstMemberIterator it = s.MemberBegin(); it != s.MemberEnd(); ++it)
		{
			string key = string(it->name.GetString());

			if(d.HasMember(key.data()))
				continue;

			Value &tmp = s[key.data()];

			d.AddMember(Value().SetString(key.data(), strlen(key.data()), d.GetAllocator()), tmp, d.GetAllocator());

		}

		StringBuffer buffers;
		Writer<StringBuffer> writers(buffers);
		d.Accept(writers);
		totaljson = string(buffers.GetString());

		return true;
	}

	static bool JsonConvertFront(const string &src, string &des, string &key, string &values, string &lasterror)
	{
		Document s;
		Document d;

		s.Parse<0>(src.data());
		d.Parse<0>("{}");
		if (s.HasParseError() || d.HasParseError()) 
			return false;

		bool flg = JsonConvertFront(s, d, key, values, lasterror);

		if(flg)
		{
			StringBuffer buffers;
			Writer<StringBuffer> writers(buffers);
			d.Accept(writers);
			des = string(buffers.GetString());
		}

		return flg;
	}

	static bool JsonConvertPost(const string &src, string &des, string key, string values, string &lasterror)
	{
		Document s;
		Document d;

		s.Parse<0>(src.data());
		d.Parse<0>("{}");
		if (s.HasParseError() || d.HasParseError())
			return false;

		bool flg = JsonConvertPost(s, d, key, values, lasterror);

        std::cout << "**********" << flg;

		if(flg)
		{
			StringBuffer buffers;
			Writer<StringBuffer> writers(buffers);
			d.Accept(writers);
			des = buffers.GetString();
		}

		return flg;
	}


	static bool JsonConvertFront(const Document &srcdoc, Document &desdoc, string &key, string &values, string &lasterror)
	{
		xml_document<> doc;
        //string posxml;
        string currPath;
		stringstream error;

		if (srcdoc.HasParseError() || desdoc.HasParseError() || srcdoc.IsNull()) 
		{
			error << "error json document";
			lasterror = error.str();
			return false;
		}

		if(desdoc.IsNull())
			desdoc.Parse<0>("{}");

        GetProcPath(currPath);
        string realPath = currPath;
        QString filename = Tool::ReadCfg(CFG_SECTION_XMLFILENAME, XML_FREEMUD_CLIENT).toString();
        realPath.append(filename.toStdString());

        if(!QFile::exists(QString::fromStdString(realPath))) {
            realPath = currPath + XML_FREEMUD_CLIENT;
        }

        QLOG_DEBUG() << "JsonConvertFront xml name :" << QString::fromStdString(realPath);

#ifdef WIN32
        if(_access(realPath.data(), 0) == -1)
#else
        if(access(realPath.data(), F_OK) == -1)
#endif
		{
            error << realPath << " not found";
			lasterror = error.str();
			return false;
		}

		try
		{

            file<> fdoc(realPath.c_str());


			if(GetXMLDocumentHandle(fdoc, doc, error) == false)
			{
				lasterror = error.str();
				return false;
			}

			if(!GetFrontJsonWithXML(doc, srcdoc, desdoc, key, values, error))
			{
				lasterror = error.str();
				return false;
			}
			return true;

		}
		catch(...)
		{
			lasterror = "An error throw in JsonConvertFront";
			return false;
		}

	}

	static bool JsonConvertPost(const Document &srcdoc, Document &desdoc, string key, string values, string &lasterror)
	{
		xml_document<> doc;
        //string posxml;
        string currPath;
		stringstream error;

		if(key.empty() || values.empty())
		{
			error << "key or value is null";
            lasterror = error.str();
			return false;
		}

		if (srcdoc.HasParseError() || desdoc.HasParseError() || srcdoc.IsNull()) 
		{
			error << "error json document";
            lasterror = error.str();
			return false;
		}

        GetProcPath(currPath);
        string realPath = currPath;

        QString filename = Tool::ReadCfg(CFG_SECTION_XMLFILENAME, XML_FREEMUD_CLIENT).toString();
        realPath.append(filename.toStdString());

        if(!QFile::exists(QString::fromStdString(realPath))) {
            realPath = currPath + XML_FREEMUD_CLIENT;
        }

        QLOG_DEBUG() << "JsonConvertPost xml name :" << QString::fromStdString(realPath);
#ifdef WIN32
        if(_access(realPath.data(), 0) == -1)
#else
        if(access(realPath.data(), F_OK) == -1)
#endif
		{
            error << realPath << " not found";
			lasterror = error.str();
			return false;
		}

		try
		{
            file<> fdoc(realPath.c_str());

			if(GetXMLDocumentHandle(fdoc, doc, error) == false)
			{
				lasterror = error.str();
				return false;
			}

			if(!GetPostJsonWithXML(doc, srcdoc, desdoc, key, values, error))
			{
				lasterror = error.str();
				return false;
			}
			return true;
		}
		catch(...)
		{
			lasterror = "An error throw in JsonConvertPost";
			return false;
		}
	}
};

#endif
