#include "refreshxmlwork.h"
#include "global.h"
#include "tool.h"
#include "QsLog.h"
#include "fm_xmltojson.hpp"

#include <QString>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QEventLoop>
#include <QTimer>
#include <QDir>
#include <QFile>

RefreshXmlWork::RefreshXmlWork(QObject *parent) :
    QObject(parent)
{
    reply = 0;
}



void RefreshXmlWork::start()
{
    QString host = Tool::ReadCfg(QString(CFG_SECTION_XMLHOST), "").toString();

    QString path, md5;

    if(!getrefreshxml(host, path, md5))
    {
        QLOG_WARN() << "getrefreshxml failed";
        return ;
    }

    QString localpath;

    if(!downloadxml(path, md5, localpath))
    {
        QLOG_WARN() << "downloadxml failed";
        return ;
    }

    appliedcxml(localpath);
}

bool RefreshXmlWork::getrefreshxml(QString &host, QString &downloadxmlpath, QString &md5)
{
    QByteArray recv;

    if(!request(QByteArray(DEFAULT_XML_REQUEST_JSON), recv, host))
        return false;

    QLOG_INFO() << "Request data" << recv;

    return getpathandmd5(recv, downloadxmlpath, md5);
}

bool RefreshXmlWork::downloadxml(const QString &path, const QString &md5, QString &localfilename)
{
    localfilename = QString(XML_FILENAME) + md5 + XML_END;
    QString downpath = Tool::ReadCfg(QString(CFG_SECTION_XMLDOWNLOADHOST), "").toString();

    if(downpath.isEmpty())
    {
        QLOG_WARN() << "download path error : " << downpath;
        return false;
    }

    downpath += localfilename;
    if(!downloadfile(downpath, localfilename))
    {
        QLOG_WARN() << "downloadfile error";
        return false;
    }

    QString localmd5 = Tool::FileMd5(localfilename);

    QLOG_DEBUG() << localmd5;
    QLOG_DEBUG() << md5;

    if(localmd5.compare(md5, Qt::CaseInsensitive) != 0)
    {
        QLOG_ERROR() << "md5 check error; local : " << localmd5 << "; md5 : " << md5;
        QFile::remove(localfilename);
        return false;
    }
    return true;
}

void RefreshXmlWork::appliedcxml(QString &localfilename)
{
    Tool::SetCfg(CFG_SECTION_XMLFILENAME, localfilename);
}

bool RefreshXmlWork::request(const QByteArray &data, QByteArray &recv, const QString &url)
{
    QNetworkAccessManager manager;

    if(QNetworkAccessManager::Accessible != manager.networkAccessible())
        manager.setNetworkAccessible(QNetworkAccessManager::Accessible);

    QNetworkRequest request;
    request.setUrl(url);

    QEventLoop loop;
    QNetworkReply *reply = manager.post(request, data);
    connect(&manager, SIGNAL(networkAccessibleChanged(QNetworkAccessManager::NetworkAccessibility)), &loop, SLOT(quit()));
    connect(reply, SIGNAL(finished()), &loop, SLOT(quit()));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError)), &loop, SLOT(quit()));
    QTimer::singleShot(10000, &loop, SLOT(quit()));
    loop.exec();

    if(reply->error() != QNetworkReply::NoError)
    {
        QLOG_ERROR() << "RefreshXmlWork::request "<< url << " error :" << reply->errorString();
        return false;
    }

    recv = reply->readAll();


    if(recv.size() == 0)
    {
        QLOG_ERROR() << "recv size 0";
        return false;
    }

    return true;
}

bool RefreshXmlWork::getpathandmd5(const QByteArray &data, QString &downloadxmlpath, QString &md5)
{
    std::string src(data.data());

    try
    {
        rapidjson::Document s;

        s.Parse<0>(src.data());

        if(!(s.HasMember(JSON_STATUSCODE_XML) && s[JSON_STATUSCODE_XML].GetInt() == 100))
        {
            QLOG_WARN() << "status_code error";
            return false;
        }

        if(s.HasMember(JSON_PATH_XML) && s.HasMember(JSON_MD5_XML))
        {
            std::string tmppath = s[JSON_PATH_XML].GetString();
            std::string tmpmd5 = s[JSON_MD5_XML].GetString();

            downloadxmlpath = QString::fromStdString(tmppath);
            md5 = QString::fromStdString(tmpmd5);

            QLOG_INFO() << "downloadxmlpath : " << downloadxmlpath;
            QLOG_INFO() << "md5 : " << md5;

            return true;
        }



        return false;
    }catch(...)
    {
        return false;
    }
}

bool RefreshXmlWork::downloadfile(const QString &path, QString filename)
{
    std::string dirpath;

    fm_xmltojson::GetProcPath(dirpath);

    QString relpath = QString::fromStdString(dirpath).append(filename);

    QLOG_INFO() << "XML download path : " << relpath;

    file.setFileName(relpath);

    if(!file.open(QIODevice::WriteOnly))
    {
        QLOG_ERROR() << "file open filed : " << filename;
        return false;
    }

    QNetworkAccessManager manager;
    QNetworkRequest request;
    QEventLoop loop;

    request.setUrl(path);

    QLOG_INFO() << "Request URL: " << path;

    reply = manager.get(request);

    connect(&manager, SIGNAL(networkAccessibleChanged(QNetworkAccessManager::NetworkAccessibility)), &loop, SLOT(quit()));
    connect(reply, SIGNAL(finished()), &loop, SLOT(quit()));
    connect(reply, SIGNAL(error(QNetworkReply::NetworkError)), &loop, SLOT(quit()));
    connect(reply, SIGNAL(readyRead()), this, SLOT(onreadyread()));
    QTimer::singleShot(10000, &loop, SLOT(quit()));
    loop.exec();

    file.close();
    reply->deleteLater();

    if(reply->error() != QNetworkReply::NoError)
    {
        QLOG_ERROR() << "RefreshXmlWork::request "<< path << " error :" << reply->errorString();
        return false;
    }

    return true;
}

void RefreshXmlWork::onreadyread()
{
    file.write(reply->readAll());
}
