#include "control.h"
#include "fm_xmltojson.hpp"
#include "network.h"
#include "global.h"
#include "QsLog.h"
#include "tool.h"
#include <QTimer>
#include <QEventLoop>
#include <QFile>
#include <QMap>
#include <QStringList>

control::control(QObject *parent) : QObject(parent)
{
    _isrollbacking = false;
    _baseinfo.clear();
    loadbaseinfo();
    _network = new NetWork();
    connect(_network, SIGNAL(recv_data_ready(QByteArray)), this, SLOT(recv_data_pos(QByteArray)));
    connect(_network, SIGNAL(error_throw(QString)), this, SLOT(recv_error_pos(QString)));
}

void control::start()
{
    unsigned int port = Tool::ReadCfg(QString(CFG_SECTION_LISTENPORT), 38888).toUInt();

    QString error;

    while(1)
    {
        QEventLoop loop;
        QTimer timer;

        connect(&timer, SIGNAL(timeout()),  &loop, SLOT(quit()));

        if(_network->listen(port, error))
            break;

        QLOG_ERROR() << "Start listen port : " << port << "failed; Sleep 60S";

        timer.start(60*1000);

        loop.exec();
    }

    checkneedrollback();

}

bool control::checkbaseinfo(std::string data, std::string &error)
{
    try
    {
        rapidjson::Document s;

        s.Parse<0>(data.data());

        if (s.HasParseError() || !s.IsObject())
        {
            error = std::string(ERROR_MSG_1102);
            return false;
        }

        if(!s.HasMember(JSON_BUSINESSDATE_POS))
        {
            error = std::string(ERROR_MSG_1103);
            return false;
        }

        if(!s.HasMember(JSON_OPERATORID_POS))
        {
            error = std::string(ERROR_MSG_1104);
            return false;
        }

        if(!s.HasMember(JSON_POSID_POS))
        {
            error = std::string(ERROR_MSG_1105);
            return false;
        }

        if(!s.HasMember(JSON_STOREID_POS))
        {
            error = std::string(ERROR_MSG_1106);
            return false;
        }

        if(!s.HasMember(JSON_PARTNERID_POS))
        {
            error = std::string(ERROR_MSG_1107);
            return false;
        }

        return true;
    }
    catch(...)
    {
        error = std::string(ERROR_MSG_1102);
        return false;
    }
}

bool control::AddSigntoJson(int reqtype, std::string &data, std::string &error)
{
    if(Tool::ReadCfg(QString(CFG_SECTION_ADDSIGN), 1).toInt() == 0)
        return true;

    QString signlist = Tool::ReadCfg(QString(CFG_SECTION_REQSIGN)).toString();

    if(signlist.isEmpty())
        return true;

    QStringList tmplist = signlist.split("|");

    QLOG_INFO() << tmplist;

    if(!tmplist.contains(QString::number(reqtype)))
        return true;

    try
    {
        rapidjson::Document s;

        s.Parse<0>(data.data());

        std::string signdata;
        std::stringstream stm;

        if(!s.HasMember(JSON_STOREID_POS))
        {
            error = std::string(ERROR_MSG_1106);
            return false;
        }

        rapidjson::Value &stordid = s[JSON_STOREID_POS];
        //rapidjson::Value &partid = s[JSON_PARTNERID_POS];

        QString parid;

        if(s.HasMember(JSON_PARTNERID_POS))
        {
            rapidjson::Value &tmpparid = s[JSON_PARTNERID_POS];
            parid = QString::number(tmpparid.GetInt());
        }
        else
        {
           parid = Tool::ReadCfg(QString(CFG_SECTION_PARTNERID), "0000").toString();
        }


        if(stordid.IsString())
        {
            QString key = Tool::ReadCfg(QString(CFG_SECTION_KEY), "AABB729B20E347CD92EFA4331D0DAAAA").toString();

            stm << stordid.GetString() << parid.toUtf8().data() << key.toUtf8().data();

            QLOG_INFO() << "sign up data : " << stm.str().c_str();

            Tool::GetMD5(stm.str(), signdata);

            QLOG_INFO() << "sign down data : " << signdata.c_str();

            if(s.HasMember(JSON_SIGNPARAM_XJ))
            {
                //rapidjson::Value &val = s[JSON_SIGNPARAM_XJ];
                //val.SetString(signdata.c_str(), s.GetAllocator());
                return true;
            }
            else
            {
                s.AddMember(JSON_SIGNPARAM_XJ, rapidjson::Value().SetString(signdata.c_str(), strlen(signdata.c_str()), s.GetAllocator()), s.GetAllocator());
            }

            rapidjson::StringBuffer buffers;
            rapidjson::Writer<rapidjson::StringBuffer> writers(buffers);
            s.Accept(writers);
            data = std::string(buffers.GetString());
            return true;
        }

        return false;
    }
    catch(...)
    {
        error = std::string(ERROR_MSG_1102);
        return false;
    }
}

bool control::setbaseinfo(std::string data, int reqtype, std::string &error, bool &isbaseinfo)
{

   QLOG_INFO() << "baseinfo" << Tool::ReadCfg(QString("all/baseinfo"), -1).toInt();
   QLOG_INFO() << "checkbaseinfo" << Tool::ReadCfg(QString("all/checkbaseinfo"), 0).toInt();
   QLOG_INFO() << reqtype;

   isbaseinfo = false;
   if(Tool::ReadCfg(QString(CFG_SECTION_BASEINFO), 1000).toInt() != reqtype)
       return true;
   isbaseinfo = true;
   if(Tool::ReadCfg(QString(CFG_SECTION_CHECKBASEINFO), 0).toInt() == 0)
       return true;

   if(!checkbaseinfo(data, error))
       return false;

//    if(!AddSigntoJson(data, error))
//        return false;

    if(!_baseinfo.empty())
        _baseinfo.clear();

    _baseinfo = data;

    QFile file(qApp->applicationDirPath() + "/" + BASEINFO);

    if(file.open(QFile::WriteOnly))
    {
        file.write(data.data());
        file.close();
    }

    return true;
}

bool control::getreqtype(std::string data, int &reqtype, std::string &error)
{
    try
    {
        rapidjson::Document s;

        s.Parse<0>(data.data());

        if (s.HasParseError() || !s.IsObject())
        {
            error = std::string(ERROR_MSG_1102);
            return false;
        }

        if(s.HasMember(JSON_REQTYPE_POS))
        {
            Value &val = s[JSON_REQTYPE_POS];

            if(val.IsInt())
            {
                reqtype = val.GetInt();
                return true;
            }

            if(val.IsString())
            {
                reqtype = atoi(val.GetString());
                return true;
            }

        }
        error = std::string(ERROR_MSG_1102);
        return false;
    }catch(...)
    {
        error = std::string(ERROR_MSG_1102);
        return false;
    }
}

bool control::getcomplejson(std::string srcdata, std::string &compledata, std::string &error)
{
    if(_baseinfo.empty())
    {
        QLOG_ERROR() << "Base info is empty";
        error = std::string(ERROR_MSG_1200);
        return false;
    }

    if(!fm_xmltojson::JsonMerge(_baseinfo, srcdata, compledata, error))
    {
        QLOG_ERROR() << "JsonMerge failed : " << error.data();
        error = std::string(ERROR_MSG_1102);
        return false;
    }

    return true;
}

bool control::senddatatoserver(std::string data, std::string &outdata, std::string &error, int reqtype)
{
    QByteArray senddata(data.data()), tmpoutdata;
    QString tmperror;

    QString url;

    //QString url = Tool::ReadCfg(QString(CFG_SECTION_SERVERHOST), QString("")).toString();

    //QLOG_INFO() << "Tool::ReadCfg URL : " << url;

    if(!getrealurl(url, reqtype))
    {
        QLOG_ERROR() << "getrealurl failed";
        error = std::string("get server real url failed, not support reqtype");
        return false;
    }

    if(!NetWork::HttpPost(url, senddata, tmpoutdata, tmperror))
    {
        QLOG_ERROR() << "NetWork::HttpPost failed " << tmperror;
        error = tmperror.toStdString();
        return false;
    }

    outdata = std::string(tmpoutdata.data());

    return true;

}

void control::checkneedrollback(std::string data, int reqtype)
{
    QLOG_INFO() << "reqtype : " << reqtype << ";checkneedrollback data : " << data.data();

    QStringList rolllist = Tool::ReadCfg(CFG_SECTION_ROLLBACKLIST, QString("10032|10031")).toString().split(",");
    QMap<int, int> rollmap;

    //QLOG_DEBUG() << rolllist << Tool::ReadCfg(CFG_SECTION_ROLLBACKLIST, QString("10032|10031")).toString();

    for(int i = 0; i < rolllist.size(); ++i)
    {
        if(rolllist[i].isEmpty())
            continue;
        QStringList tmplist = rolllist[i].split("|");

        //QLOG_DEBUG() << tmplist;

        if(tmplist.size() != 2 || tmplist[0].isEmpty() || tmplist[1].isEmpty())
            continue;

        rollmap.insert(tmplist[0].toInt(), tmplist[1].toUInt());
    }

    //QLOG_DEBUG() << "need roolback maping : " << rollmap;

    try
    {
        if(!rollmap.contains(reqtype))
            return ;

        rapidjson::Document s;

        s.Parse<0>(data.data());

        if (s.HasParseError() || !s.IsObject())
        {
           QLOG_ERROR() << "Error data in checkneedrollback";
           return ;
        }

        if(!s.HasMember(JSON_REQTYPE_POS))
            return ;

        rapidjson::Value &val = s[JSON_REQTYPE_POS];

        val.SetInt(rollmap[reqtype]);

        rapidjson::StringBuffer buffers;
        rapidjson::Writer<rapidjson::StringBuffer> writers(buffers);
        s.Accept(writers);
        std::string totaljson = std::string(buffers.GetString());

        QLOG_INFO() << "totaljson rollback json : " << totaljson.data();

        swap_data(totaljson, rollmap[reqtype]);

        checkneedrollback();
    }
    catch(...)
    {
        QLOG_INFO()  << "An error throw in checkneedrollback(std::string, int)";
        return ;
    }
    return ;
}

void control::checkneedrollback()
{
    QFile file(qApp->applicationDirPath() + QString("/") + QString(ROLLBACK_FILL));

    if(!file.exists())
        return ;

   _isrollbacking = true;

   QByteArray data, reqtype;

   if(!file.open(QFile::ReadOnly))
       return ;

   reqtype = file.readLine();
   data = file.readLine();
   file.close();

   QByteArray debasedata = QByteArray::fromBase64(data);

   QLOG_WARN() << "reqtype : " << reqtype << "need rollback : " << data;

   QString url;

   //String url = Tool::ReadCfg(QString(CFG_SECTION_SERVERHOST), QString("")).toString();

   //QLOG_INFO() << "Tool::ReadCfg URL : " << url;

   if(!getrealurl(url, QString::fromUtf8(reqtype).toInt()))
   {
       QLOG_ERROR() << "getrealurl failed on rollback";
       file.remove();
       return ;
   }

   QByteArray outdata;
   QString error;

   while(!NetWork::HttpPost(url, debasedata, outdata, error, 20))
   {
       QEventLoop loop;
       QTimer timer;

       connect(&timer, SIGNAL(timeout()),  &loop, SLOT(quit()));

       timer.start(5*1000);

       loop.exec();
   }

   QLOG_INFO() << "rollback success : " << outdata;

   file.remove();

   _isrollbacking = false;
}

bool control::getrealurl(QString &url, int reqtype)
{

    //QLOG_DEBUG() << "reqtype : " << reqtype;

    QString tmpurl = Tool::ReadCfg(QString(CFG_SECTION_URLPATHLIST), QString("")).toString();

    if(tmpurl.isEmpty())
        return false;

    QStringList list = tmpurl.split("|");

    for(int i = 0; i < list.size(); ++i)
    {
        if(list[i].isEmpty())
            continue;
        QStringList tmplist = list[i].split(",");

        if(tmplist.size() == 2)
        {
            if(!tmplist[0].isEmpty() && tmplist[0].toInt() == reqtype && !tmplist[1].isEmpty())
            {
                  url.append(tmplist[1]);
                  return true;
            }
        }
    }

    return false;

//    switch (reqtype) {
//    case 10032:
//        url.append(SERVER_URL_HX);
//        break;
//    case 10039:
//        url.append(SERVER_URL_CHECK);
//        break;
//    default:
//        return false;
//    }
//    return true;
}

control::~control()
{
    if(_network != NULL)
    {
        delete _network;
        _network = NULL;
    }
}

void control::swap_and_send_data(std::string compledata, int reqtype)
{
    std::string error;
    std::string key;
    std::string value;
    std::string response_sevice;
    std::string request_sevice;
    std::string respons_pos;

    if(!fm_xmltojson::JsonConvertFront(compledata, request_sevice, key, value, error))
    {
        QLOG_ERROR() << "JsonConvertFront failed" << error.data();
        recv_error_pos(QString::fromLocal8Bit(ERROR_MSG_1001));
        return ;
    }

    QLOG_INFO() << "POS json to server json : " << QString::fromUtf8(request_sevice.data());
    QLOG_INFO() << "key : " << key.data();
    QLOG_INFO() << "value : " << value.data();

    if(!senddatatoserver(request_sevice, response_sevice, error, reqtype))
    {
        QLOG_ERROR() << "senddatatoserver failed" << error.data();
        recv_error_pos(QString::fromLocal8Bit(ERROR_MSG_1003));
        checkneedrollback(compledata, reqtype);
        return ;
    }

    QLOG_INFO() << "recv data from server " << QString::fromUtf8(response_sevice.data());

    if(!fm_xmltojson::JsonConvertPost(response_sevice, respons_pos, key, value, error))
    {
        QLOG_ERROR() << "JsonConvertPost failed" << error.data();
        recv_error_pos(QString::fromLocal8Bit(ERROR_MSG_1001));
        return ;
    }

    QLOG_INFO() << "POS json to server json : " << QString::fromUtf8(respons_pos.data());

    if(respons_pos.empty() || respons_pos.compare("{}") == 0)
    {
        respons_pos = std::string(EMPTY_JSON_FMCONPON);
        QLOG_INFO() << "chang error msg to pos : " << QString::fromUtf8(respons_pos.data());
    }

    if(!_network->senddata(QByteArray(respons_pos.data())))
    {
        QLOG_ERROR() << "send data to Pos failed";
        checkneedrollback(compledata, reqtype);
    }
}

void control::swap_data(std::string compledata, int reqtype)
{
    std::string error;
    std::string key;
    std::string value;
    std::string request_sevice;

    if(!fm_xmltojson::JsonConvertFront(compledata, request_sevice, key, value, error))
    {
        QLOG_ERROR() << "JsonConvertFront failed" << error.data();
        //recv_error_pos(QString::fromLocal8Bit(ERROR_MSG_1001));
        return ;
    }

    QLOG_INFO() << "rollback data : " << request_sevice.data();
    QLOG_INFO() << "key : " << key.data();
    QLOG_INFO() << "value : " << value.data();

    QFile file(qApp->applicationDirPath() + QString("/") + QString(ROLLBACK_FILL));

    if(!file.open(QFile::WriteOnly))
    {
        QLOG_ERROR() << qApp->applicationDirPath() + QString("/") + QString(ROLLBACK_FILL) << "open failed";
        return ;
    }

    QString strreqtype = QString::number(reqtype);

#ifdef WIN32
    strreqtype.append("\n");
#else
    strreqtype.append("\r\n");
#endif
    file.write(strreqtype.toUtf8());
    file.write(QByteArray(request_sevice.data()).toBase64());
    file.close();

    QLOG_INFO() << "base64 info : " << QByteArray(request_sevice.data()).toBase64();

    QLOG_INFO() << "save data to rollback file success";

    return ;
}

void control::loadbaseinfo()
{
    QFile file(qApp->applicationDirPath() + "/" + BASEINFO);

    if(file.open(QFile::ReadOnly))
    {
        _baseinfo = std::string(file.readAll().data());
        file.close();
    }
}

bool control::checkneedhead(QByteArray &data)
{
    //QLOG_INFO() << data.toHex();
    QLOG_INFO() << data.size();

    QByteArray tmpdata = data;

    if(Tool::ReadCfg(QString(CFG_SECTION_CHECKHEADER), 0).toInt() == 0)
        return true;

    fm_socket_head head = {0, 0, 0};

    if(sizeof(fm_socket_head) >= data.size())
        return false;

    memcpy((void *)&head, data.data(), sizeof(fm_socket_head));

    QLOG_INFO() << "flag:" << head.flag << ";len:" << head.len << ";ver:" << head.ver;

    if(head.len > (data.size() - sizeof(fm_socket_head)))
        return false;

    data = tmpdata.right(data.size() - sizeof(fm_socket_head));

    return true;
}

void control::recv_data_pos(QByteArray data)
{
    if(!checkneedhead(data))
    {
        QLOG_ERROR() << "checkneedhead failed";
        recv_error_pos(QString::fromLocal8Bit(ERROR_REQ_HEADER));
        return ;
    }

    QLOG_INFO() << "recv date from pos : " << QString::fromUtf8(data);

    std::string stddata = std::string(data.data()), error;
    std::string compledata;

    int reqtype  = 0;

    if(!getreqtype(stddata, reqtype, error))
    {
        QLOG_ERROR() << "getreqtype failed";
        recv_error_pos(QString::fromLocal8Bit(error.data()));
        return ;
    }

    QLOG_INFO() << "reqtype form POS : " << reqtype;

    bool isbaseifo = false;

    if(!setbaseinfo(stddata, reqtype, error, isbaseifo))
    {
        QLOG_ERROR() << "setbaseinfo failed";
        recv_error_pos(QString::fromLocal8Bit(error.data()));
        return ;
    }

    if(isbaseifo)
    {
        QLOG_INFO() << "send data to pos : " << QString::fromLocal8Bit(SUCCESS_JSON_FMCONPON);
        _network->senddata(QString::fromLocal8Bit(SUCCESS_JSON_FMCONPON).toUtf8());
        return ;
    }

    if(_isrollbacking)
    {
        QLOG_ERROR() << "check status failed" << QString::fromLocal8Bit(ERROR_MSG_1002);
        QString str = QString(ERRRO_JSON_FMCONPON).arg(QString::fromLocal8Bit(ERROR_MSG_1002));
        recv_error_pos(str);
        return ;
    }

    if(Tool::ReadCfg(QString(CFG_SECTION_CHECKBASEINFO), 0).toInt() == 1)
    {
        if(!getcomplejson(stddata, compledata, error))
        {
            QLOG_ERROR() << "getcomplejson failed";
            recv_error_pos(QString::fromLocal8Bit(error.data()));
            return ;
        }
    }
    else
    {
        compledata = stddata;
        compressjson(compledata);
    }

    QLOG_INFO() << "POS comple json : " << QString::fromUtf8(compledata.data());

    if(!AddSigntoJson(reqtype, compledata, error))
    {
        QLOG_ERROR() << "AddSigntoJson failed";
        recv_error_pos(QString::fromLocal8Bit(error.data()));
        return ;
    }

    QLOG_INFO() << "AddSigntoJson json : " << QString::fromUtf8(compledata.data());

    swap_and_send_data(compledata, reqtype);
}

bool control::compressjson(std::string &data)
{
    try
    {
        rapidjson::Document s;

        s.Parse<0>(data.data());

        rapidjson::StringBuffer buffers;
        rapidjson::Writer<rapidjson::StringBuffer> writers(buffers);
        s.Accept(writers);
        data = std::string(buffers.GetString());
        return true;
    }
    catch(...)
    {
        return false;
    }
}

void control::recv_error_pos(QString error)
{
    QString str = QString(ERRRO_JSON_FMCONPON).arg(error);

    QLOG_WARN() << "fmcoupon return error message to pos : " << str;

    if(!_network->senddata(str.toUtf8()))
    {
        QLOG_ERROR() << "send data to Pos failed";
    }
}
